{
	"translatorID": "5c6895a1-b6a9-4939-9c65-43f8ae0ef096",
	"label": "METS",
	"creator": "Abe Jellinek",
	"target": "xml",
	"minVersion": "3.0",
	"maxVersion": "",
	"priority": 50,
	"configOptions": {
		"dataMode": "xml/dom"
	},
	"inRepository": true,
	"translatorType": 1,
	"lastUpdated": "2021-07-13 20:44:34"
}

/*
	***** BEGIN LICENSE BLOCK *****

	Copyright © 2021 Abe Jellinek
	
	This file is part of Zotero.

	Zotero is free software: you can redistribute it and/or modify
	it under the terms of the GNU Affero General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Zotero is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU Affero General Public License for more details.

	You should have received a copy of the GNU Affero General Public License
	along with Zotero. If not, see <http://www.gnu.org/licenses/>.

	***** END LICENSE BLOCK *****
*/


function detectImport() {
	var line;
	var i = 0;
	while ((line = Zotero.read()) !== false) {
		if (line !== "") {
			if (line.includes("<mets")) {
				return true;
			}
			else if (i++ > 7) {
				return false;
			}
		}
	}
	return false;
}

function doImport() {
	let xml = Zotero.getXML();
	
	for (let mets of xml.querySelectorAll('mets')) {
		let attachments = createAttachments(mets);
		
		for (let dmd of mets.querySelectorAll('dmdSec')) {
			let mdWrap = dmd.querySelector('mdWrap');
			if (!mdWrap) {
				Z.debug('No metadata found in METS item. External metadata is not supported.');
			}
			
			let mdType = mdWrap.getAttribute('MDTYPE');
			Z.debug(`Found metadata of type '${mdType}'`);
			
			let data = extractData(mdWrap);
			processData(mdType, data, attachments);
		}
	}
}

function createAttachments(mets) {
	let attachments = [];
	
	for (let file of mets.querySelectorAll('fileSec file')) {
		if (attachments.length >= 5) {
			Z.debug('Too many attachments. Something is probably wrong.');
			attachments = [];
			break;
		}
		
		let mimeType = file.getAttribute('MIMETYPE');
		let locator = file.querySelector('FLocat[LOCTYPE="URL"]');
		let url = locator && locator.getAttribute('xlink:href');
		if (mimeType && url) {
			let title = 'Attachment';
			
			if (mimeType == 'application/pdf') {
				title = 'PDF'; // full text? often it isn't text!
			}
			else if (mimeType.startsWith('audio/')) {
				title = 'Audio';
			}
			else if (mimeType.startsWith('video/')) {
				title = 'Video';
			}
			else if (mimeType.startsWith('image/')) {
				title = 'Image';
			}
			
			attachments.push({
				title,
				mimeType,
				url
			});
		}
	}
	
	return attachments;
}

function extractData(mdWrap) {
	for (let child of mdWrap.children) {
		if (child.tagName.endsWith('xmlData')) {
			Z.debug('Metadata is XML');
			return child.innerHTML;
		}
		else if (child.tagName.endsWith('binData')) {
			Z.debug('Metadata is base64 encoded');
			return atob(child.innerHTML);
		}
	}
	
	Z.debug('Metadata is in XML (unwrapped)');
	return mdWrap.innerHTML;
}

function processData(mdType, data, attachments) {
	switch (mdType) {
		case 'MARC':
			// charmingly, we don't get to know what KIND of MARC we're
			// getting. we could go off of whether it's encoded as
			// binary, but that isn't really valid according to the
			// spec. so we'll just try MARCXML, then binary MARC, then
			// MAB2 for good measure (it's MARC-ish!).
			callImport(
				data,
				[
					'edd87d07-9194-42f8-b2ad-997c4c7deefd', // MARCXML
					'a6ee60df-1ddc-4aae-bb25-45e0537be973', // MARC
					'91acf493-0de7-4473-8b62-89fd141e6c74', // MAB2
				],
				attachments
			);
			break;
		case 'MODS':
			callImport(
				data,
				'0e2235e7-babf-413c-9acf-f27cce5f059c', // MODS
				attachments,
				item => item.callNumber = ''
			);
			break;
		case 'OTHER':
			// this usually indicates some kind of internal metadata that isn't
			// essential to parse.
			Z.debug('Skipping \'OTHER\' metadata');
			break;
		case 'EAD':
			// todo: an EAD translator. does anyone use EAD?
			// https://www.loc.gov/ead/
		default:
			throw new Error(`Unsupported metadata type: ${mdType}`);
	}
}

function callImport(data, translator, attachments, callback) {
	let trans = Zotero.loadTranslator('import');
	trans.setTranslator(translator);
	trans.setHandler('itemDone', function (_, item) {
		let numFieldsSet = 0;
		for (let field of Object.keys(item)) {
			if (field == 'itemType') continue;
			
			if (item[field] && typeof item[field] == 'string') {
				numFieldsSet++;
			}
		}
		if (numFieldsSet <= 1) {
			Z.debug(`Skipping item with < 2 fields set: '${item.title}'`);
			return;
		}
		
		if (attachments) {
			item.attachments.push(...attachments);
		}
		
		if (item.language == 'zxx') {
			delete item.language;
		}
		
		if (callback) callback(item);
		item.complete();
	});
	trans.setString(data);
	trans.translate();
}

/** BEGIN TEST CASES **/
var testCases = [
	{
		"type": "import",
		"input": "<?xml version=\"1.0\" encoding=\"UTF-8\"?><mets:mets xmlns:mets=\"http://www.loc.gov/METS/\" xmlns:mods=\"http://www.loc.gov/mods/v3\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" LABEL=\"Antietam, Maryland. Allan Pinkerton, President Lincoln, and Major General John A. McClernand: Another View\" OBJID=\"https://www.wdl.org/1\" xml:lang=\"en\" xml:id=\"1\" xsi:schemaLocation=\"http://www.loc.gov/METS/ http://www.loc.gov/standards/mets/mets.xsd http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-6.xsd\" TYPE=\"Prints, Photographs\">\n  <mets:metsHdr CREATEDATE=\"2010-05-17T13:04:33+00:00\" LASTMODDATE=\"2017-04-04T18:54:12+00:00\">\n    <mets:agent ROLE=\"DISSEMINATOR\">\n      <mets:name>World Digital Library</mets:name>\n    </mets:agent>\n    <mets:agent ROLE=\"CUSTODIAN\">\n      <mets:name>Library of Congress</mets:name>\n    </mets:agent>\n  </mets:metsHdr>\n  <mets:dmdSec ID=\"MODS1\">\n    <mets:mdWrap MDTYPE=\"MODS\">\n      <mets:xmlData>\n        <mods:mods version=\"3.6\">\n          <mods:typeOfResource>text</mods:typeOfResource>\n          <mods:titleInfo>\n            <mods:title>Antietam, Maryland. Allan Pinkerton, President Lincoln, and Major General John A. McClernand: Another View</mods:title>\n          </mods:titleInfo>\n          <mods:abstract>At the outset of the U.S. Civil War, Mathew Brady dispatched a team of photographers to document the conflict. Among them was a Scottish-born immigrant named Alexander Gardner, the photographer who took this photo of Lincoln at Antietam as well as other famous wartime shots. The man to Lincoln's right is Allan Pinkerton, founder of the Pinkerton National Detective Agency, whom Lincoln had as head of a personal security detail during the war. Gardner titled another shot of Pinkerton and his brother William at Antietam &#8220;The Secret Service.&#8221; Gardner photographed Lincoln on seven separate occasions, the last one on February 5, 1865, only a few weeks before Lincoln&#8217;s assassination. In 1866 he published Gardner&#8217;s Sketchbook of the War, combining plates and text, commemorating such battles as Fredericksburg, Gettysburg, and Petersburg, but the book was a commercial failure. Photographic historians also have suggested that Gardner staged many of his photos, moving dead bodies and using a regular prop gun to create romanticized pictorial narratives.</mods:abstract>\n          <mods:physicalDescription>\n            <mods:note>1 negative : glass, wet collodion</mods:note>\n          </mods:physicalDescription>\n          <mods:language>\n            <mods:languageTerm type=\"code\" authority=\"iso639-2b\">zxx</mods:languageTerm>\n          </mods:language>\n          <mods:name type=\"personal\">\n            <mods:namePart>Gardner, Alexander, 1821-1882</mods:namePart>\n            <mods:role>\n              <mods:roleTerm type=\"text\" authority=\"marcrelator\">Photographer</mods:roleTerm>\n            </mods:role>\n          </mods:name>\n          <mods:originInfo>\n            <mods:dateIssued>1862-10-03</mods:dateIssued>\n          </mods:originInfo>\n          <mods:subject>\n            <mods:temporal encoding=\"iso8601\">1862-10-03</mods:temporal>\n          </mods:subject>\n          <mods:subject>\n            <mods:hierarchicalGeographic>\n              <mods:region>North America</mods:region>\n              <mods:country>United States of America</mods:country>\n              <mods:state>Maryland</mods:state>\n              <mods:city>Antietam</mods:city>\n            </mods:hierarchicalGeographic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Antietam, Battle of, Maryland, 1862</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Generals</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Lincoln, Abraham, 1809-1865</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>McClernand, John A. (John Alexander), 1812-1900</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Military camps</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Pinkerton, Allan, 1819-1884</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Presidents</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>Tents</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>United States--History--Civil War, 1861-1865</mods:topic>\n          </mods:subject>\n          <mods:subject authority=\"lcsh\">\n            <mods:topic>United States. Army</mods:topic>\n          </mods:subject>\n          <mods:classification authority=\"ddc\">973</mods:classification>\n          <mods:relatedItem type=\"otherVersion\">\n            <mods:location>\n              <mods:url>http://hdl.loc.gov/loc.pnp/cwpb.04326</mods:url>\n            </mods:location>\n          </mods:relatedItem>\n          <mods:relatedItem type=\"otherVersion\">\n            <mods:location>\n              <mods:url>http://hdl.loc.gov/loc.wdl/wdl.1</mods:url>\n            </mods:location>\n          </mods:relatedItem>\n        </mods:mods>\n      </mets:xmlData>\n    </mets:mdWrap>\n  </mets:dmdSec>\n  <mets:fileSec>\n    <mets:fileGrp USE=\"MASTER\">\n      <mets:file MIMETYPE=\"image/tiff\" GROUPID=\"G1\" ID=\"F-1-1\" SIZE=\"17196009\">\n        <mets:FLocat LOCTYPE=\"URL\" xlink:href=\"master/04326u.tif\"/>\n      </mets:file>\n    </mets:fileGrp>\n    <mets:fileGrp USE=\"SERVICE\">\n      <mets:file MIMETYPE=\"application/pdf\" GROUPID=\"0\" ID=\"2468401\" SIZE=\"247363\">\n        <mets:FLocat LOCTYPE=\"URL\" xlink:href=\"https://dl.wdl.org/1/service/1.pdf\"/>\n      </mets:file>\n    </mets:fileGrp>\n  </mets:fileSec>\n  <mets:structMap TYPE=\"logical\">\n    <mets:div>\n      <mets:fptr FILEID=\"F-1-1\"/>\n    </mets:div>\n  </mets:structMap>\n</mets:mets>\n",
		"items": [
			{
				"itemType": "document",
				"title": "Antietam, Maryland. Allan Pinkerton, President Lincoln, and Major General John A. McClernand: Another View",
				"creators": [
					{
						"firstName": "Alexander",
						"lastName": "Gardner",
						"creatorType": "author"
					}
				],
				"date": "1862-10-03",
				"abstractNote": "At the outset of the U.S. Civil War, Mathew Brady dispatched a team of photographers to document the conflict. Among them was a Scottish-born immigrant named Alexander Gardner, the photographer who took this photo of Lincoln at Antietam as well as other famous wartime shots. The man to Lincoln's right is Allan Pinkerton, founder of the Pinkerton National Detective Agency, whom Lincoln had as head of a personal security detail during the war. Gardner titled another shot of Pinkerton and his brother William at Antietam “The Secret Service.” Gardner photographed Lincoln on seven separate occasions, the last one on February 5, 1865, only a few weeks before Lincoln’s assassination. In 1866 he published Gardner’s Sketchbook of the War, combining plates and text, commemorating such battles as Fredericksburg, Gettysburg, and Petersburg, but the book was a commercial failure. Photographic historians also have suggested that Gardner staged many of his photos, moving dead bodies and using a regular prop gun to create romanticized pictorial narratives.",
				"attachments": [
					{
						"title": "Image",
						"mimeType": "image/tiff"
					},
					{
						"title": "PDF",
						"mimeType": "application/pdf"
					}
				],
				"tags": [
					{
						"tag": "Antietam, Battle of, Maryland, 1862"
					},
					{
						"tag": "Generals"
					},
					{
						"tag": "Lincoln, Abraham, 1809-1865"
					},
					{
						"tag": "McClernand, John A. (John Alexander), 1812-1900"
					},
					{
						"tag": "Military camps"
					},
					{
						"tag": "Pinkerton, Allan, 1819-1884"
					},
					{
						"tag": "Presidents"
					},
					{
						"tag": "Tents"
					},
					{
						"tag": "United States--History--Civil War, 1861-1865"
					},
					{
						"tag": "United States. Army"
					}
				],
				"notes": [],
				"seeAlso": []
			}
		]
	},
	{
		"type": "import",
		"input": "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n<mets ID=\"ID1\" OBJID=\"1262473\" LABEL=\"El cruzado : semanario tradicionalista - Portada\" PROFILE=\"http://www.loc.gov/standards/mets/profiles/00000044.xml\" xmlns=\"http://www.loc.gov/METS/\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:marc=\"http://www.loc.gov/MARC21/slim\" xmlns:mix=\"http://www.loc.gov/mix/v20\" xsi:schemaLocation=\"http://www.loc.gov/METS/ http://www.loc.gov/standards/mets/mets.xsd http://www.loc.gov/MARC21/slim http://www.loc.gov/standards/marcxml/schema/MARC21slim.xsd http://www.loc.gov/mix/v20 http://www.loc.gov/standards/mix/mix20/mix20.xsd\">\n <metsHdr CREATEDATE=\"2021-07-10T02:07:44\"/>\n <dmdSec ID=\"DM1\">\n  <mdWrap MDTYPE=\"MARC\">\n   <xmlData> <collection xmlns=\"http://www.loc.gov/MARC21/slim\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.loc.gov/MARC21/slim http://www.loc.gov/standards/marcxml/schema/MARC21slim.xsd\">\n  <record>\n    <leader>00000cas 82200000   4500</leader>\n    <controlfield tag=\"001\">BDG20080002497</controlfield>\n    <controlfield tag=\"003\">BDG</controlfield>\n    <controlfield tag=\"005\">20210421075144.0</controlfield>\n    <controlfield tag=\"008\">100609d19121917espwr|n      |0|||b|spa d</controlfield>\n    <datafield tag=\"040\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">BDG</subfield>\n      <subfield code=\"b\">glg</subfield>\n      <subfield code=\"c\">BDG</subfield>\n      <subfield code=\"e\">rda</subfield>\n      <subfield code=\"d\">BDG</subfield>\n    </datafield>\n    <datafield tag=\"245\" ind1=\"0\" ind2=\"3\">\n      <subfield code=\"a\">El cruzado</subfield>\n      <subfield code=\"b\">: semanario tradicionalista</subfield>\n    </datafield>\n    <datafield tag=\"264\" ind1=\" \" ind2=\"1\">\n      <subfield code=\"a\">Mondoñedo</subfield>\n      <subfield code=\"b\">[editor non identificado]</subfield>\n      <subfield code=\"c\">1912-1917</subfield>\n    </datafield>\n    <datafield tag=\"264\" ind1=\" \" ind2=\"3\">\n      <subfield code=\"a\">Mondoñedo</subfield>\n      <subfield code=\"b\">Tip. Mancebo</subfield>\n      <subfield code=\"c\">1912-1917</subfield>\n    </datafield>\n    <datafield tag=\"300\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">v.</subfield>\n      <subfield code=\"c\">47 cm</subfield>\n    </datafield>\n    <datafield tag=\"310\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">Semanal </subfield>\n    </datafield>\n    <datafield tag=\"500\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">Director: José Polo Folgueira</subfield>\n    </datafield>\n    <datafield tag=\"500\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">Propietario: Enrique Costas Márquez</subfield>\n    </datafield>\n    <datafield tag=\"500\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">Redactor: Carolino Costas Márquez</subfield>\n    </datafield>\n    <datafield tag=\"500\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">Colaboradores: Antonio Noriega Varela e Manuel Amor Meilán.</subfield>\n    </datafield>\n    <datafield tag=\"590\" ind1=\"0\" ind2=\" \">\n      <subfield code=\"a\">n. 13 (jul. 1912)</subfield>\n    </datafield>\n    <datafield tag=\"650\" ind1=\" \" ind2=\"4\">\n      <subfield code=\"0\">BDGA20090004351</subfield>\n      <subfield code=\"a\">Prensa</subfield>\n      <subfield code=\"z\">Galicia</subfield>\n    </datafield>\n    <datafield tag=\"700\" ind1=\"1\" ind2=\" \">\n      <subfield code=\"0\">BDGA20180473692</subfield>\n      <subfield code=\"a\">Costas Márquez, Enrique</subfield>\n    </datafield>\n    <datafield tag=\"700\" ind1=\"2\" ind2=\"0\">\n      <subfield code=\"0\">BDGA20181256447</subfield>\n      <subfield code=\"a\">Costas Márquez, Carolino</subfield>\n    </datafield>\n    <datafield tag=\"700\" ind1=\"1\" ind2=\" \">\n      <subfield code=\"0\">BDGA20120008724</subfield>\n      <subfield code=\"a\">Noriega Varela, Antonio</subfield>\n      <subfield code=\"d\">1869-1947</subfield>\n    </datafield>\n    <datafield tag=\"700\" ind1=\"1\" ind2=\"0\">\n      <subfield code=\"0\">BDGA20130018034</subfield>\n      <subfield code=\"a\">Amor Meilán, Manuel</subfield>\n      <subfield code=\"d\">1867-1933</subfield>\n    </datafield>\n    <datafield tag=\"710\" ind1=\" \" ind2=\" \">\n      <subfield code=\"0\">BDGA20180698774</subfield>\n      <subfield code=\"a\">Tipografía Mancebo</subfield>\n    </datafield>\n    <datafield tag=\"752\" ind1=\" \" ind2=\" \">\n      <subfield code=\"0\">BDGA20150000309</subfield>\n      <subfield code=\"a\">España</subfield>\n      <subfield code=\"b\">Galicia</subfield>\n      <subfield code=\"c\">Lugo</subfield>\n      <subfield code=\"d\">Mondoñedo</subfield>\n    </datafield>\n    <datafield tag=\"853\" ind1=\"2\" ind2=\"1\">\n      <subfield code=\"i\">(Día)</subfield>\n      <subfield code=\"j\">(Mes)</subfield>\n      <subfield code=\"k\">(Año)</subfield>\n      <subfield code=\"w\">w</subfield>\n    </datafield>\n    <datafield tag=\"856\" ind1=\" \" ind2=\" \">\n      <subfield code=\"q\">image/jpeg</subfield>\n      <subfield code=\"w\">1262473</subfield>\n      <subfield code=\"y\">Portada</subfield>\n    </datafield>\n    <datafield tag=\"866\" ind1=\" \" ind2=\" \">\n      <subfield code=\"a\">13:jul.1912 - 14:sept.1912, 25:oct.1912 - 25:oct.1912, 2:nov.1912 - 7:dic.1912</subfield>\n    </datafield>\n  </record>\n</collection></xmlData>\n  </mdWrap>\n </dmdSec>\n <dmdSec ID=\"DGBGOM\">\n  <mdWrap MDTYPE=\"OTHER\" OTHERMDTYPE=\"DGBGOM\">\n   <xmlData><grupoObjetoMultimedia mimeType=\"image/jpeg\" presentacionDef=\"portada\" tieneOCR=\"false\" nombre=\"El cruzado : semanario tradicionalista\" descripcion=\"Portada\"><imagenFavorita>FID0</imagenFavorita></grupoObjetoMultimedia></xmlData>\n  </mdWrap>\n </dmdSec>\n <fileSec>\n  <fileGrp ID=\"FG1\">\n   <file ID=\"FID0\" MIMETYPE=\"image/jpeg\" SEQ=\"1\" OWNERID=\"12559894\" DMDID=\"DM1\">\n    <FLocat LOCTYPE=\"URL\" xlink:type=\"simple\" xlink:href=\"http://biblioteca.galiciana.gal/es/catalogo_imagenes/imagen_id.do?idImagen=12559894\"/>\n   </file>\n  </fileGrp>\n </fileSec>\n <structMap ID=\"SM1\" TYPE=\"PHYSICAL\">\n  <div ORDER=\"1\" LABEL=\"El cruzado : semanario tradicionalista - Portada\" TYPE=\"book\">\n   <div ORDER=\"1\" LABEL=\"imaxe representativa\" TYPE=\"page\">\n    <fptr FILEID=\"FID0\"/>\n   </div>\n  </div>\n </structMap>\n</mets>\n",
		"items": [
			{
				"itemType": "book",
				"title": "El cruzado: semanario tradicionalista",
				"creators": [
					{
						"firstName": "Enrique",
						"lastName": "Costas Márquez",
						"creatorType": "editor"
					},
					{
						"firstName": "Carolino",
						"lastName": "Costas Márquez",
						"creatorType": "editor"
					},
					{
						"firstName": "Antonio",
						"lastName": "Noriega Varela",
						"creatorType": "editor"
					},
					{
						"firstName": "Manuel",
						"lastName": "Amor Meilán",
						"creatorType": "editor"
					},
					{
						"lastName": "Tipografía Mancebo",
						"creatorType": "editor",
						"fieldMode": true
					}
				],
				"date": "1912",
				"place": "Mondoñedo",
				"publisher": "editor non identificado",
				"attachments": [
					{
						"title": "Image",
						"mimeType": "image/jpeg"
					}
				],
				"tags": [
					{
						"tag": "Galicia"
					},
					{
						"tag": "Prensa"
					}
				],
				"notes": [
					{
						"note": "Director: José Polo Folgueira Propietario: Enrique Costas Márquez Redactor: Carolino Costas Márquez Colaboradores: Antonio Noriega Varela e Manuel Amor Meilán"
					}
				],
				"seeAlso": []
			}
		]
	},
	{
		"type": "import",
		"input": "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<!-- This METS file was created on Wed Oct 12 13:00:22 CEST 2011 using the UGH Metadata Library: ugh.fileformats.mets.MetsModsImportExport (version 1.9-20100505) -->\n<mets:mets xmlns:mets=\"http://www.loc.gov/METS/\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-3.xsd http://www.loc.gov/METS/ http://www.loc.gov/standards/mets/version17/mets.v1-7.xsd\"><mets:dmdSec ID=\"DMDLOG_0000\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:classification authority=\"GDZ\">Antiquitates_und_Archaeologia</mods:classification><mods:classification authority=\"GDZ\">Antiquitates_und_Archaeologia_ARCHAEO18</mods:classification><mods:recordInfo><mods:recordIdentifier source=\"gbv-ppn\">PPN143342827</mods:recordIdentifier></mods:recordInfo><mods:identifier type=\"PPNanalog\">PPN143342819</mods:identifier><mods:titleInfo><mods:title>Vite De' Pittori, Scultori Ed Architetti Napoletani</mods:title><mods:subTitle>Non mai date alla luce da Autore alcuno ...</mods:subTitle></mods:titleInfo><mods:language><mods:languageTerm authority=\"iso639-2b\" type=\"code\">it</mods:languageTerm></mods:language><mods:originInfo><mods:place><mods:placeTerm type=\"text\">Neapel</mods:placeTerm></mods:place><mods:dateIssued encoding=\"w3cdtf\" keyDate=\"yes\">1742</mods:dateIssued><mods:publisher>Ricciardi</mods:publisher></mods:originInfo><mods:subject authority=\"gdz\"><mods:topic>domivite</mods:topic></mods:subject><mods:name type=\"personal\"><mods:role><mods:roleTerm authority=\"marcrelator\" type=\"code\">aut</mods:roleTerm></mods:role><mods:namePart type=\"family\">Dominici</mods:namePart><mods:namePart type=\"given\">Bernardo</mods:namePart><mods:displayForm>Dominici, Bernardo</mods:displayForm></mods:name></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:amdSec ID=\"AMD\"><mets:rightsMD ID=\"RIGHTS\"><mets:mdWrap MDTYPE=\"OTHER\" MIMETYPE=\"text/xml\" OTHERMDTYPE=\"DVRIGHTS\"><mets:xmlData><dv:rights xmlns:dv=\"http://dfg-viewer.de/\"><dv:owner>Digitalisierungszentrum der Niedersächsischen Staats- und Universitätsbibliothek Göttingen</dv:owner><dv:ownerLogo>http://gdz.sub.uni-goettingen.de/logo_gdz_dfgv.png</dv:ownerLogo><dv:ownerSiteURL>http://gdz.sub.uni-goettingen.de</dv:ownerSiteURL><dv:ownerContact/></dv:rights></mets:xmlData></mets:mdWrap></mets:rightsMD><mets:digiprovMD ID=\"DIGIPROV\"><mets:mdWrap MDTYPE=\"OTHER\" MIMETYPE=\"text/xml\" OTHERMDTYPE=\"DVLINKS\"><mets:xmlData><dv:links xmlns:dv=\"http://dfg-viewer.de/\"><dv:reference>http://opac.sub.uni-goettingen.de/DB=1/PPN?PPN=143342827</dv:reference><dv:presentation>http://resolver.sub.uni-goettingen.de/purl?PPN143342827</dv:presentation></dv:links></mets:xmlData></mets:mdWrap></mets:digiprovMD></mets:amdSec><mets:structMap TYPE=\"LOGICAL\"><mets:div ADMID=\"AMD\" DMDID=\"DMDLOG_0000\" ID=\"LOG_0000\" LABEL=\"Vite De' Pittori, Scultori Ed Architetti Napoletani\" TYPE=\"MultiVolumeWork\"><mets:div ID=\"LOG_0002\" TYPE=\"volume\" LABEL=\"Vite De' Pittori, Scultori Ed Architetti Napoletani : non mai date alla luce da autore alcuno ... 1\"><mets:mptr LOCTYPE=\"URL\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:href=\"http://gdz.sub.uni-goettingen.de/mets/PPN659413833.xml\"/></mets:div></mets:div></mets:structMap></mets:mets>\n",
		"items": [
			{
				"itemType": "document",
				"title": "Vite De' Pittori, Scultori Ed Architetti Napoletani: Non mai date alla luce da Autore alcuno ...",
				"creators": [
					{
						"firstName": "Bernardo",
						"lastName": "Dominici",
						"creatorType": "author"
					}
				],
				"date": "1742",
				"language": "it",
				"publisher": "Ricciardi",
				"attachments": [],
				"tags": [
					{
						"tag": "domivite"
					}
				],
				"notes": [],
				"seeAlso": []
			}
		]
	},
	{
		"type": "import",
		"input": "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<!-- This METS file was created on Mon Mar 19 14:28:59 CET 2012 using the UGH Metadata Library: ugh.fileformats.mets.MetsModsImportExport (version 1.9-20100505) -->\n<mets:mets xmlns:mets=\"http://www.loc.gov/METS/\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-3.xsd http://www.loc.gov/METS/ http://www.loc.gov/standards/mets/version17/mets.v1-7.xsd\"><mets:dmdSec ID=\"DMDLOG_0001\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:classification authority=\"GDZ\">Antiquitates_und_Archaeologia</mods:classification><mods:classification authority=\"GDZ\">Antiquitates_und_Archaeologia_ARCHAEO18</mods:classification><mods:relatedItem type=\"host\"><mods:recordInfo><mods:recordIdentifier source=\"gbv-ppn\">PPN137363532</mods:recordIdentifier></mods:recordInfo></mods:relatedItem><mods:recordInfo><mods:recordIdentifier source=\"gbv-ppn\">PPN137363540</mods:recordIdentifier></mods:recordInfo><mods:identifier type=\"PPNanalog\">PPN137363540</mods:identifier><mods:titleInfo><mods:title>Accurata, E Succinta Descrizione Topografica Delle Antichitá Di Roma</mods:title></mods:titleInfo><mods:part order=\"10\" type=\"host\"><mods:detail><mods:number>1</mods:number></mods:detail></mods:part><mods:language><mods:languageTerm authority=\"iso639-2b\" type=\"code\">la</mods:languageTerm></mods:language><mods:originInfo><mods:place><mods:placeTerm type=\"text\">Roma</mods:placeTerm></mods:place><mods:dateIssued encoding=\"w3cdtf\" keyDate=\"yes\">1763</mods:dateIssued><mods:publisher>Bernabò, e Lazzarini</mods:publisher></mods:originInfo><mods:name type=\"personal\"><mods:role><mods:roleTerm authority=\"marcrelator\" type=\"code\">aut</mods:roleTerm></mods:role><mods:namePart type=\"family\">Venuti</mods:namePart><mods:namePart type=\"given\">Ridolfino</mods:namePart><mods:displayForm>Venuti, Ridolfino</mods:displayForm></mods:name><mods:physicalDescription><mods:extent>[1] Bl., XXXVI, 143, [1] S., 59 Bl., [1] gef. Bl.</mods:extent></mods:physicalDescription></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0002\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Clemens Papa XIII. Ad futuram rei memoriam.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0003\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Avvertimento Al Lettore.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0004\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Indice De' Capi Della Parte Prima.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0005\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Approvazione.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0006\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Introduzione Alla Topografia Di Roma.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0007\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Carta Topografica Delle Antichità Di Roma [...]</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0008\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Primo Del Monte Palatino.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0009\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 1 [- Taf.] 14</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0010\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Primo Del Monte Palatino.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0011\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Secondo Del Foro Romano.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0012\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 15 [- Taf.] 16</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0013\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Terzo Del Campidoglio.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0014\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 17 [- Taf.] 22</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0015\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Quarto  Fori di Cesare, d'Augusto, di Nerva, e di Trajano.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0016\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 23 [-Taf.] 25</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0017\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Quinto Del Colle Quirinale.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0018\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 26 [- Taf.] 33</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0019\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Sesto Monte Viminale.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0020\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Settimo. Colle Esquilino.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0021\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 34 [- Taf.] 55</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0022\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>Capo Ottavo Del Monte Celio.</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDLOG_0023\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:titleInfo><mods:title>[Taf.] 56 [- Taf.] 59</mods:title></mods:titleInfo></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:dmdSec ID=\"DMDPHYS_0000\"><mets:mdWrap MDTYPE=\"MODS\"><mets:xmlData><mods:mods xmlns:mods=\"http://www.loc.gov/mods/v3\"><mods:location><mods:physicalLocation type=\"shelfmark\">4 ARCH I, 3892:1</mods:physicalLocation></mods:location></mods:mods></mets:xmlData></mets:mdWrap></mets:dmdSec><mets:amdSec ID=\"AMD\"><mets:rightsMD ID=\"RIGHTS\"><mets:mdWrap MDTYPE=\"OTHER\" MIMETYPE=\"text/xml\" OTHERMDTYPE=\"DVRIGHTS\"><mets:xmlData><dv:rights xmlns:dv=\"http://dfg-viewer.de/\"><dv:owner>Digitalisierungszentrum der Niedersächsischen Staats- und Universitätsbibliothek Göttingen</dv:owner><dv:ownerLogo>http://gdz.sub.uni-goettingen.de/logo_gdz_dfgv.png</dv:ownerLogo><dv:ownerSiteURL>http://gdz.sub.uni-goettingen.de</dv:ownerSiteURL><dv:ownerContact/></dv:rights></mets:xmlData></mets:mdWrap></mets:rightsMD><mets:digiprovMD ID=\"DIGIPROV\"><mets:mdWrap MDTYPE=\"OTHER\" MIMETYPE=\"text/xml\" OTHERMDTYPE=\"DVLINKS\"><mets:xmlData><dv:links xmlns:dv=\"http://dfg-viewer.de/\"><dv:reference>http://opac.sub.uni-goettingen.de/DB=1/PPN?PPN=137363532</dv:reference><dv:presentation>http://resolver.sub.uni-goettingen.de/purl?PPN137363540</dv:presentation></dv:links></mets:xmlData></mets:mdWrap></mets:digiprovMD></mets:amdSec><mets:fileSec><mets:fileGrp USE=\"MIN\"><mets:file ID=\"FILE_0000_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000001.jpg\"/></mets:file><mets:file ID=\"FILE_0001_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000002.jpg\"/></mets:file><mets:file ID=\"FILE_0002_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000003.jpg\"/></mets:file><mets:file ID=\"FILE_0003_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000004.jpg\"/></mets:file><mets:file ID=\"FILE_0004_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000005.jpg\"/></mets:file><mets:file ID=\"FILE_0005_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000006.jpg\"/></mets:file><mets:file ID=\"FILE_0006_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000007.jpg\"/></mets:file><mets:file ID=\"FILE_0007_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000008.jpg\"/></mets:file><mets:file ID=\"FILE_0008_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000009.jpg\"/></mets:file><mets:file ID=\"FILE_0009_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000010.jpg\"/></mets:file><mets:file ID=\"FILE_0010_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000011.jpg\"/></mets:file><mets:file ID=\"FILE_0011_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000012.jpg\"/></mets:file><mets:file ID=\"FILE_0012_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000013.jpg\"/></mets:file><mets:file ID=\"FILE_0013_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000014.jpg\"/></mets:file><mets:file ID=\"FILE_0014_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000015.jpg\"/></mets:file><mets:file ID=\"FILE_0015_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000016.jpg\"/></mets:file><mets:file ID=\"FILE_0016_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000017.jpg\"/></mets:file><mets:file ID=\"FILE_0017_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000018.jpg\"/></mets:file><mets:file ID=\"FILE_0018_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000019.jpg\"/></mets:file><mets:file ID=\"FILE_0019_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000020.jpg\"/></mets:file><mets:file ID=\"FILE_0020_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000021.jpg\"/></mets:file><mets:file ID=\"FILE_0021_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000022.jpg\"/></mets:file><mets:file ID=\"FILE_0022_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000023.jpg\"/></mets:file><mets:file ID=\"FILE_0023_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000024.jpg\"/></mets:file><mets:file ID=\"FILE_0024_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000025.jpg\"/></mets:file><mets:file ID=\"FILE_0025_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000026.jpg\"/></mets:file><mets:file ID=\"FILE_0026_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000027.jpg\"/></mets:file><mets:file ID=\"FILE_0027_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000028.jpg\"/></mets:file><mets:file ID=\"FILE_0028_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000029.jpg\"/></mets:file><mets:file ID=\"FILE_0029_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000030.jpg\"/></mets:file><mets:file ID=\"FILE_0030_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000031.jpg\"/></mets:file><mets:file ID=\"FILE_0031_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000032.jpg\"/></mets:file><mets:file ID=\"FILE_0032_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000033.jpg\"/></mets:file><mets:file ID=\"FILE_0033_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000034.jpg\"/></mets:file><mets:file ID=\"FILE_0034_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000035.jpg\"/></mets:file><mets:file ID=\"FILE_0035_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000036.jpg\"/></mets:file><mets:file ID=\"FILE_0036_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000037.jpg\"/></mets:file><mets:file ID=\"FILE_0037_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000038.jpg\"/></mets:file><mets:file ID=\"FILE_0038_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000039.jpg\"/></mets:file><mets:file ID=\"FILE_0039_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000040.jpg\"/></mets:file><mets:file ID=\"FILE_0040_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000041.jpg\"/></mets:file><mets:file ID=\"FILE_0041_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000042.jpg\"/></mets:file><mets:file ID=\"FILE_0042_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000043.jpg\"/></mets:file><mets:file ID=\"FILE_0043_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000044.jpg\"/></mets:file><mets:file ID=\"FILE_0044_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000045.jpg\"/></mets:file><mets:file ID=\"FILE_0045_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000046.jpg\"/></mets:file><mets:file ID=\"FILE_0046_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000047.jpg\"/></mets:file><mets:file ID=\"FILE_0047_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000048.jpg\"/></mets:file><mets:file ID=\"FILE_0048_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000049.jpg\"/></mets:file><mets:file ID=\"FILE_0049_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000050.jpg\"/></mets:file><mets:file ID=\"FILE_0050_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000051.jpg\"/></mets:file><mets:file ID=\"FILE_0051_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000052.jpg\"/></mets:file><mets:file ID=\"FILE_0052_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000053.jpg\"/></mets:file><mets:file ID=\"FILE_0053_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000054.jpg\"/></mets:file><mets:file ID=\"FILE_0054_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000055.jpg\"/></mets:file><mets:file ID=\"FILE_0055_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000056.jpg\"/></mets:file><mets:file ID=\"FILE_0056_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000057.jpg\"/></mets:file><mets:file ID=\"FILE_0057_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000058.jpg\"/></mets:file><mets:file ID=\"FILE_0058_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000059.jpg\"/></mets:file><mets:file ID=\"FILE_0059_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000060.jpg\"/></mets:file><mets:file ID=\"FILE_0060_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000061.jpg\"/></mets:file><mets:file ID=\"FILE_0061_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000062.jpg\"/></mets:file><mets:file ID=\"FILE_0062_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000063.jpg\"/></mets:file><mets:file ID=\"FILE_0063_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000064.jpg\"/></mets:file><mets:file ID=\"FILE_0064_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000065.jpg\"/></mets:file><mets:file ID=\"FILE_0065_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000066.jpg\"/></mets:file><mets:file ID=\"FILE_0066_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000067.jpg\"/></mets:file><mets:file ID=\"FILE_0067_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000068.jpg\"/></mets:file><mets:file ID=\"FILE_0068_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000069.jpg\"/></mets:file><mets:file ID=\"FILE_0069_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000070.jpg\"/></mets:file><mets:file ID=\"FILE_0070_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000071.jpg\"/></mets:file><mets:file ID=\"FILE_0071_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000072.jpg\"/></mets:file><mets:file ID=\"FILE_0072_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000073.jpg\"/></mets:file><mets:file ID=\"FILE_0073_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000074.jpg\"/></mets:file><mets:file ID=\"FILE_0074_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000075.jpg\"/></mets:file><mets:file ID=\"FILE_0075_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000076.jpg\"/></mets:file><mets:file ID=\"FILE_0076_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000077.jpg\"/></mets:file><mets:file ID=\"FILE_0077_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000078.jpg\"/></mets:file><mets:file ID=\"FILE_0078_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000079.jpg\"/></mets:file><mets:file ID=\"FILE_0079_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000080.jpg\"/></mets:file><mets:file ID=\"FILE_0080_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000081.jpg\"/></mets:file><mets:file ID=\"FILE_0081_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000082.jpg\"/></mets:file><mets:file ID=\"FILE_0082_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000083.jpg\"/></mets:file><mets:file ID=\"FILE_0083_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000084.jpg\"/></mets:file><mets:file ID=\"FILE_0084_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000085.jpg\"/></mets:file><mets:file ID=\"FILE_0085_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000086.jpg\"/></mets:file><mets:file ID=\"FILE_0086_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000087.jpg\"/></mets:file><mets:file ID=\"FILE_0087_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000088.jpg\"/></mets:file><mets:file ID=\"FILE_0088_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000089.jpg\"/></mets:file><mets:file ID=\"FILE_0089_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000090.jpg\"/></mets:file><mets:file ID=\"FILE_0090_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000091.jpg\"/></mets:file><mets:file ID=\"FILE_0091_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000092.jpg\"/></mets:file><mets:file ID=\"FILE_0092_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000093.jpg\"/></mets:file><mets:file ID=\"FILE_0093_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000094.jpg\"/></mets:file><mets:file ID=\"FILE_0094_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000095.jpg\"/></mets:file><mets:file ID=\"FILE_0095_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000096.jpg\"/></mets:file><mets:file ID=\"FILE_0096_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000097.jpg\"/></mets:file><mets:file ID=\"FILE_0097_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000098.jpg\"/></mets:file><mets:file ID=\"FILE_0098_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000099.jpg\"/></mets:file><mets:file ID=\"FILE_0099_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000100.jpg\"/></mets:file><mets:file ID=\"FILE_0100_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000101.jpg\"/></mets:file><mets:file ID=\"FILE_0101_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000102.jpg\"/></mets:file><mets:file ID=\"FILE_0102_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000103.jpg\"/></mets:file><mets:file ID=\"FILE_0103_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000104.jpg\"/></mets:file><mets:file ID=\"FILE_0104_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000105.jpg\"/></mets:file><mets:file ID=\"FILE_0105_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000106.jpg\"/></mets:file><mets:file ID=\"FILE_0106_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000107.jpg\"/></mets:file><mets:file ID=\"FILE_0107_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000108.jpg\"/></mets:file><mets:file ID=\"FILE_0108_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000109.jpg\"/></mets:file><mets:file ID=\"FILE_0109_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000110.jpg\"/></mets:file><mets:file ID=\"FILE_0110_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000111.jpg\"/></mets:file><mets:file ID=\"FILE_0111_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000112.jpg\"/></mets:file><mets:file ID=\"FILE_0112_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000113.jpg\"/></mets:file><mets:file ID=\"FILE_0113_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000114.jpg\"/></mets:file><mets:file ID=\"FILE_0114_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000115.jpg\"/></mets:file><mets:file ID=\"FILE_0115_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000116.jpg\"/></mets:file><mets:file ID=\"FILE_0116_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000117.jpg\"/></mets:file><mets:file ID=\"FILE_0117_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000118.jpg\"/></mets:file><mets:file ID=\"FILE_0118_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000119.jpg\"/></mets:file><mets:file ID=\"FILE_0119_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000120.jpg\"/></mets:file><mets:file ID=\"FILE_0120_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000121.jpg\"/></mets:file><mets:file ID=\"FILE_0121_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000122.jpg\"/></mets:file><mets:file ID=\"FILE_0122_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000123.jpg\"/></mets:file><mets:file ID=\"FILE_0123_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000124.jpg\"/></mets:file><mets:file ID=\"FILE_0124_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000125.jpg\"/></mets:file><mets:file ID=\"FILE_0125_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000126.jpg\"/></mets:file><mets:file ID=\"FILE_0126_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000127.jpg\"/></mets:file><mets:file ID=\"FILE_0127_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000128.jpg\"/></mets:file><mets:file ID=\"FILE_0128_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000129.jpg\"/></mets:file><mets:file ID=\"FILE_0129_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000130.jpg\"/></mets:file><mets:file ID=\"FILE_0130_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000131.jpg\"/></mets:file><mets:file ID=\"FILE_0131_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000132.jpg\"/></mets:file><mets:file ID=\"FILE_0132_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000133.jpg\"/></mets:file><mets:file ID=\"FILE_0133_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000134.jpg\"/></mets:file><mets:file ID=\"FILE_0134_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000135.jpg\"/></mets:file><mets:file ID=\"FILE_0135_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000136.jpg\"/></mets:file><mets:file ID=\"FILE_0136_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000137.jpg\"/></mets:file><mets:file ID=\"FILE_0137_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000138.jpg\"/></mets:file><mets:file ID=\"FILE_0138_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000139.jpg\"/></mets:file><mets:file ID=\"FILE_0139_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000140.jpg\"/></mets:file><mets:file ID=\"FILE_0140_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000141.jpg\"/></mets:file><mets:file ID=\"FILE_0141_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000142.jpg\"/></mets:file><mets:file ID=\"FILE_0142_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000143.jpg\"/></mets:file><mets:file ID=\"FILE_0143_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000144.jpg\"/></mets:file><mets:file ID=\"FILE_0144_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000145.jpg\"/></mets:file><mets:file ID=\"FILE_0145_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000146.jpg\"/></mets:file><mets:file ID=\"FILE_0146_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000147.jpg\"/></mets:file><mets:file ID=\"FILE_0147_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000148.jpg\"/></mets:file><mets:file ID=\"FILE_0148_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000149.jpg\"/></mets:file><mets:file ID=\"FILE_0149_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000150.jpg\"/></mets:file><mets:file ID=\"FILE_0150_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000151.jpg\"/></mets:file><mets:file ID=\"FILE_0151_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000152.jpg\"/></mets:file><mets:file ID=\"FILE_0152_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000153.jpg\"/></mets:file><mets:file ID=\"FILE_0153_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000154.jpg\"/></mets:file><mets:file ID=\"FILE_0154_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000155.jpg\"/></mets:file><mets:file ID=\"FILE_0155_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000156.jpg\"/></mets:file><mets:file ID=\"FILE_0156_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000157.jpg\"/></mets:file><mets:file ID=\"FILE_0157_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000158.jpg\"/></mets:file><mets:file ID=\"FILE_0158_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000159.jpg\"/></mets:file><mets:file ID=\"FILE_0159_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000160.jpg\"/></mets:file><mets:file ID=\"FILE_0160_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000161.jpg\"/></mets:file><mets:file ID=\"FILE_0161_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000162.jpg\"/></mets:file><mets:file ID=\"FILE_0162_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000163.jpg\"/></mets:file><mets:file ID=\"FILE_0163_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000164.jpg\"/></mets:file><mets:file ID=\"FILE_0164_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000165.jpg\"/></mets:file><mets:file ID=\"FILE_0165_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000166.jpg\"/></mets:file><mets:file ID=\"FILE_0166_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000167.jpg\"/></mets:file><mets:file ID=\"FILE_0167_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000168.jpg\"/></mets:file><mets:file ID=\"FILE_0168_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000169.jpg\"/></mets:file><mets:file ID=\"FILE_0169_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000170.jpg\"/></mets:file><mets:file ID=\"FILE_0170_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000171.jpg\"/></mets:file><mets:file ID=\"FILE_0171_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000172.jpg\"/></mets:file><mets:file ID=\"FILE_0172_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000173.jpg\"/></mets:file><mets:file ID=\"FILE_0173_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000174.jpg\"/></mets:file><mets:file ID=\"FILE_0174_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000175.jpg\"/></mets:file><mets:file ID=\"FILE_0175_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000176.jpg\"/></mets:file><mets:file ID=\"FILE_0176_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000177.jpg\"/></mets:file><mets:file ID=\"FILE_0177_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000178.jpg\"/></mets:file><mets:file ID=\"FILE_0178_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000179.jpg\"/></mets:file><mets:file ID=\"FILE_0179_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000180.jpg\"/></mets:file><mets:file ID=\"FILE_0180_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000181.jpg\"/></mets:file><mets:file ID=\"FILE_0181_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000182.jpg\"/></mets:file><mets:file ID=\"FILE_0182_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000183.jpg\"/></mets:file><mets:file ID=\"FILE_0183_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000184.jpg\"/></mets:file><mets:file ID=\"FILE_0184_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000185.jpg\"/></mets:file><mets:file ID=\"FILE_0185_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000186.jpg\"/></mets:file><mets:file ID=\"FILE_0186_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000187.jpg\"/></mets:file><mets:file ID=\"FILE_0187_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000188.jpg\"/></mets:file><mets:file ID=\"FILE_0188_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000189.jpg\"/></mets:file><mets:file ID=\"FILE_0189_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000190.jpg\"/></mets:file><mets:file ID=\"FILE_0190_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000191.jpg\"/></mets:file><mets:file ID=\"FILE_0191_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000192.jpg\"/></mets:file><mets:file ID=\"FILE_0192_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000193.jpg\"/></mets:file><mets:file ID=\"FILE_0193_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000194.jpg\"/></mets:file><mets:file ID=\"FILE_0194_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000195.jpg\"/></mets:file><mets:file ID=\"FILE_0195_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000196.jpg\"/></mets:file><mets:file ID=\"FILE_0196_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000197.jpg\"/></mets:file><mets:file ID=\"FILE_0197_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000198.jpg\"/></mets:file><mets:file ID=\"FILE_0198_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000199.jpg\"/></mets:file><mets:file ID=\"FILE_0199_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000200.jpg\"/></mets:file><mets:file ID=\"FILE_0200_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000201.jpg\"/></mets:file><mets:file ID=\"FILE_0201_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000202.jpg\"/></mets:file><mets:file ID=\"FILE_0202_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000203.jpg\"/></mets:file><mets:file ID=\"FILE_0203_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000204.jpg\"/></mets:file><mets:file ID=\"FILE_0204_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000205.jpg\"/></mets:file><mets:file ID=\"FILE_0205_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000206.jpg\"/></mets:file><mets:file ID=\"FILE_0206_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000207.jpg\"/></mets:file><mets:file ID=\"FILE_0207_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000208.jpg\"/></mets:file><mets:file ID=\"FILE_0208_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000209.jpg\"/></mets:file><mets:file ID=\"FILE_0209_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000210.jpg\"/></mets:file><mets:file ID=\"FILE_0210_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000211.jpg\"/></mets:file><mets:file ID=\"FILE_0211_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000212.jpg\"/></mets:file><mets:file ID=\"FILE_0212_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000213.jpg\"/></mets:file><mets:file ID=\"FILE_0213_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000214.jpg\"/></mets:file><mets:file ID=\"FILE_0214_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000215.jpg\"/></mets:file><mets:file ID=\"FILE_0215_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000216.jpg\"/></mets:file><mets:file ID=\"FILE_0216_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000217.jpg\"/></mets:file><mets:file ID=\"FILE_0217_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000218.jpg\"/></mets:file><mets:file ID=\"FILE_0218_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000219.jpg\"/></mets:file><mets:file ID=\"FILE_0219_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000220.jpg\"/></mets:file><mets:file ID=\"FILE_0220_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000221.jpg\"/></mets:file><mets:file ID=\"FILE_0221_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000222.jpg\"/></mets:file><mets:file ID=\"FILE_0222_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000223.jpg\"/></mets:file><mets:file ID=\"FILE_0223_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000224.jpg\"/></mets:file><mets:file ID=\"FILE_0224_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000225.jpg\"/></mets:file><mets:file ID=\"FILE_0225_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000226.jpg\"/></mets:file><mets:file ID=\"FILE_0226_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000227.jpg\"/></mets:file><mets:file ID=\"FILE_0227_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000228.jpg\"/></mets:file><mets:file ID=\"FILE_0228_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000229.jpg\"/></mets:file><mets:file ID=\"FILE_0229_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000230.jpg\"/></mets:file><mets:file ID=\"FILE_0230_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000231.jpg\"/></mets:file><mets:file ID=\"FILE_0231_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000232.jpg\"/></mets:file><mets:file ID=\"FILE_0232_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000233.jpg\"/></mets:file><mets:file ID=\"FILE_0233_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000234.jpg\"/></mets:file><mets:file ID=\"FILE_0234_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000235.jpg\"/></mets:file><mets:file ID=\"FILE_0235_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000236.jpg\"/></mets:file><mets:file ID=\"FILE_0236_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000237.jpg\"/></mets:file><mets:file ID=\"FILE_0237_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000238.jpg\"/></mets:file><mets:file ID=\"FILE_0238_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000239.jpg\"/></mets:file><mets:file ID=\"FILE_0239_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000240.jpg\"/></mets:file><mets:file ID=\"FILE_0240_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000241.jpg\"/></mets:file><mets:file ID=\"FILE_0241_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000242.jpg\"/></mets:file><mets:file ID=\"FILE_0242_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000243.jpg\"/></mets:file><mets:file ID=\"FILE_0243_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000244.jpg\"/></mets:file><mets:file ID=\"FILE_0244_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000245.jpg\"/></mets:file><mets:file ID=\"FILE_0245_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000246.jpg\"/></mets:file><mets:file ID=\"FILE_0246_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000247.jpg\"/></mets:file><mets:file ID=\"FILE_0247_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000248.jpg\"/></mets:file><mets:file ID=\"FILE_0248_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000249.jpg\"/></mets:file><mets:file ID=\"FILE_0249_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000250.jpg\"/></mets:file><mets:file ID=\"FILE_0250_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000251.jpg\"/></mets:file><mets:file ID=\"FILE_0251_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000252.jpg\"/></mets:file><mets:file ID=\"FILE_0252_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000253.jpg\"/></mets:file><mets:file ID=\"FILE_0253_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000254.jpg\"/></mets:file><mets:file ID=\"FILE_0254_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000255.jpg\"/></mets:file><mets:file ID=\"FILE_0255_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000256.jpg\"/></mets:file><mets:file ID=\"FILE_0256_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000257.jpg\"/></mets:file><mets:file ID=\"FILE_0257_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000258.jpg\"/></mets:file><mets:file ID=\"FILE_0258_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000259.jpg\"/></mets:file><mets:file ID=\"FILE_0259_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000260.jpg\"/></mets:file><mets:file ID=\"FILE_0260_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000261.jpg\"/></mets:file><mets:file ID=\"FILE_0261_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000262.jpg\"/></mets:file><mets:file ID=\"FILE_0262_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000263.jpg\"/></mets:file><mets:file ID=\"FILE_0263_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000264.jpg\"/></mets:file><mets:file ID=\"FILE_0264_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000265.jpg\"/></mets:file><mets:file ID=\"FILE_0265_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000266.jpg\"/></mets:file><mets:file ID=\"FILE_0266_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000267.jpg\"/></mets:file><mets:file ID=\"FILE_0267_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000268.jpg\"/></mets:file><mets:file ID=\"FILE_0268_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000269.jpg\"/></mets:file><mets:file ID=\"FILE_0269_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000270.jpg\"/></mets:file><mets:file ID=\"FILE_0270_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000271.jpg\"/></mets:file><mets:file ID=\"FILE_0271_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000272.jpg\"/></mets:file><mets:file ID=\"FILE_0272_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000273.jpg\"/></mets:file><mets:file ID=\"FILE_0273_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000274.jpg\"/></mets:file><mets:file ID=\"FILE_0274_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000275.jpg\"/></mets:file><mets:file ID=\"FILE_0275_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000276.jpg\"/></mets:file><mets:file ID=\"FILE_0276_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000277.jpg\"/></mets:file><mets:file ID=\"FILE_0277_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000278.jpg\"/></mets:file><mets:file ID=\"FILE_0278_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000279.jpg\"/></mets:file><mets:file ID=\"FILE_0279_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000280.jpg\"/></mets:file><mets:file ID=\"FILE_0280_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000281.jpg\"/></mets:file><mets:file ID=\"FILE_0281_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000282.jpg\"/></mets:file><mets:file ID=\"FILE_0282_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000283.jpg\"/></mets:file><mets:file ID=\"FILE_0283_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000284.jpg\"/></mets:file><mets:file ID=\"FILE_0284_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000285.jpg\"/></mets:file><mets:file ID=\"FILE_0285_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000286.jpg\"/></mets:file><mets:file ID=\"FILE_0286_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000287.jpg\"/></mets:file><mets:file ID=\"FILE_0287_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000288.jpg\"/></mets:file><mets:file ID=\"FILE_0288_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000289.jpg\"/></mets:file><mets:file ID=\"FILE_0289_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000290.jpg\"/></mets:file><mets:file ID=\"FILE_0290_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000291.jpg\"/></mets:file><mets:file ID=\"FILE_0291_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000292.jpg\"/></mets:file><mets:file ID=\"FILE_0292_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000293.jpg\"/></mets:file><mets:file ID=\"FILE_0293_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000294.jpg\"/></mets:file><mets:file ID=\"FILE_0294_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000295.jpg\"/></mets:file><mets:file ID=\"FILE_0295_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000296.jpg\"/></mets:file><mets:file ID=\"FILE_0296_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000297.jpg\"/></mets:file><mets:file ID=\"FILE_0297_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000298.jpg\"/></mets:file><mets:file ID=\"FILE_0298_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000299.jpg\"/></mets:file><mets:file ID=\"FILE_0299_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000300.jpg\"/></mets:file><mets:file ID=\"FILE_0300_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000301.jpg\"/></mets:file><mets:file ID=\"FILE_0301_MIN\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/800/0/00000302.jpg\"/></mets:file></mets:fileGrp><mets:fileGrp USE=\"DEFAULT\"><mets:file ID=\"FILE_0000_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000001.jpg\"/></mets:file><mets:file ID=\"FILE_0001_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000002.jpg\"/></mets:file><mets:file ID=\"FILE_0002_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000003.jpg\"/></mets:file><mets:file ID=\"FILE_0003_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000004.jpg\"/></mets:file><mets:file ID=\"FILE_0004_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000005.jpg\"/></mets:file><mets:file ID=\"FILE_0005_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000006.jpg\"/></mets:file><mets:file ID=\"FILE_0006_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000007.jpg\"/></mets:file><mets:file ID=\"FILE_0007_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000008.jpg\"/></mets:file><mets:file ID=\"FILE_0008_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000009.jpg\"/></mets:file><mets:file ID=\"FILE_0009_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000010.jpg\"/></mets:file><mets:file ID=\"FILE_0010_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000011.jpg\"/></mets:file><mets:file ID=\"FILE_0011_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000012.jpg\"/></mets:file><mets:file ID=\"FILE_0012_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000013.jpg\"/></mets:file><mets:file ID=\"FILE_0013_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000014.jpg\"/></mets:file><mets:file ID=\"FILE_0014_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000015.jpg\"/></mets:file><mets:file ID=\"FILE_0015_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000016.jpg\"/></mets:file><mets:file ID=\"FILE_0016_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000017.jpg\"/></mets:file><mets:file ID=\"FILE_0017_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000018.jpg\"/></mets:file><mets:file ID=\"FILE_0018_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000019.jpg\"/></mets:file><mets:file ID=\"FILE_0019_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000020.jpg\"/></mets:file><mets:file ID=\"FILE_0020_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000021.jpg\"/></mets:file><mets:file ID=\"FILE_0021_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000022.jpg\"/></mets:file><mets:file ID=\"FILE_0022_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000023.jpg\"/></mets:file><mets:file ID=\"FILE_0023_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000024.jpg\"/></mets:file><mets:file ID=\"FILE_0024_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000025.jpg\"/></mets:file><mets:file ID=\"FILE_0025_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000026.jpg\"/></mets:file><mets:file ID=\"FILE_0026_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000027.jpg\"/></mets:file><mets:file ID=\"FILE_0027_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000028.jpg\"/></mets:file><mets:file ID=\"FILE_0028_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000029.jpg\"/></mets:file><mets:file ID=\"FILE_0029_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000030.jpg\"/></mets:file><mets:file ID=\"FILE_0030_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000031.jpg\"/></mets:file><mets:file ID=\"FILE_0031_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000032.jpg\"/></mets:file><mets:file ID=\"FILE_0032_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000033.jpg\"/></mets:file><mets:file ID=\"FILE_0033_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000034.jpg\"/></mets:file><mets:file ID=\"FILE_0034_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000035.jpg\"/></mets:file><mets:file ID=\"FILE_0035_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000036.jpg\"/></mets:file><mets:file ID=\"FILE_0036_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000037.jpg\"/></mets:file><mets:file ID=\"FILE_0037_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000038.jpg\"/></mets:file><mets:file ID=\"FILE_0038_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000039.jpg\"/></mets:file><mets:file ID=\"FILE_0039_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000040.jpg\"/></mets:file><mets:file ID=\"FILE_0040_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000041.jpg\"/></mets:file><mets:file ID=\"FILE_0041_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000042.jpg\"/></mets:file><mets:file ID=\"FILE_0042_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000043.jpg\"/></mets:file><mets:file ID=\"FILE_0043_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000044.jpg\"/></mets:file><mets:file ID=\"FILE_0044_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000045.jpg\"/></mets:file><mets:file ID=\"FILE_0045_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000046.jpg\"/></mets:file><mets:file ID=\"FILE_0046_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000047.jpg\"/></mets:file><mets:file ID=\"FILE_0047_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000048.jpg\"/></mets:file><mets:file ID=\"FILE_0048_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000049.jpg\"/></mets:file><mets:file ID=\"FILE_0049_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000050.jpg\"/></mets:file><mets:file ID=\"FILE_0050_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000051.jpg\"/></mets:file><mets:file ID=\"FILE_0051_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000052.jpg\"/></mets:file><mets:file ID=\"FILE_0052_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000053.jpg\"/></mets:file><mets:file ID=\"FILE_0053_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000054.jpg\"/></mets:file><mets:file ID=\"FILE_0054_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000055.jpg\"/></mets:file><mets:file ID=\"FILE_0055_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000056.jpg\"/></mets:file><mets:file ID=\"FILE_0056_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000057.jpg\"/></mets:file><mets:file ID=\"FILE_0057_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000058.jpg\"/></mets:file><mets:file ID=\"FILE_0058_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000059.jpg\"/></mets:file><mets:file ID=\"FILE_0059_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000060.jpg\"/></mets:file><mets:file ID=\"FILE_0060_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000061.jpg\"/></mets:file><mets:file ID=\"FILE_0061_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000062.jpg\"/></mets:file><mets:file ID=\"FILE_0062_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000063.jpg\"/></mets:file><mets:file ID=\"FILE_0063_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000064.jpg\"/></mets:file><mets:file ID=\"FILE_0064_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000065.jpg\"/></mets:file><mets:file ID=\"FILE_0065_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000066.jpg\"/></mets:file><mets:file ID=\"FILE_0066_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000067.jpg\"/></mets:file><mets:file ID=\"FILE_0067_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000068.jpg\"/></mets:file><mets:file ID=\"FILE_0068_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000069.jpg\"/></mets:file><mets:file ID=\"FILE_0069_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000070.jpg\"/></mets:file><mets:file ID=\"FILE_0070_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000071.jpg\"/></mets:file><mets:file ID=\"FILE_0071_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000072.jpg\"/></mets:file><mets:file ID=\"FILE_0072_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000073.jpg\"/></mets:file><mets:file ID=\"FILE_0073_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000074.jpg\"/></mets:file><mets:file ID=\"FILE_0074_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000075.jpg\"/></mets:file><mets:file ID=\"FILE_0075_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000076.jpg\"/></mets:file><mets:file ID=\"FILE_0076_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000077.jpg\"/></mets:file><mets:file ID=\"FILE_0077_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000078.jpg\"/></mets:file><mets:file ID=\"FILE_0078_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000079.jpg\"/></mets:file><mets:file ID=\"FILE_0079_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000080.jpg\"/></mets:file><mets:file ID=\"FILE_0080_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000081.jpg\"/></mets:file><mets:file ID=\"FILE_0081_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000082.jpg\"/></mets:file><mets:file ID=\"FILE_0082_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000083.jpg\"/></mets:file><mets:file ID=\"FILE_0083_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000084.jpg\"/></mets:file><mets:file ID=\"FILE_0084_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000085.jpg\"/></mets:file><mets:file ID=\"FILE_0085_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000086.jpg\"/></mets:file><mets:file ID=\"FILE_0086_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000087.jpg\"/></mets:file><mets:file ID=\"FILE_0087_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000088.jpg\"/></mets:file><mets:file ID=\"FILE_0088_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000089.jpg\"/></mets:file><mets:file ID=\"FILE_0089_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000090.jpg\"/></mets:file><mets:file ID=\"FILE_0090_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000091.jpg\"/></mets:file><mets:file ID=\"FILE_0091_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000092.jpg\"/></mets:file><mets:file ID=\"FILE_0092_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000093.jpg\"/></mets:file><mets:file ID=\"FILE_0093_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000094.jpg\"/></mets:file><mets:file ID=\"FILE_0094_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000095.jpg\"/></mets:file><mets:file ID=\"FILE_0095_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000096.jpg\"/></mets:file><mets:file ID=\"FILE_0096_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000097.jpg\"/></mets:file><mets:file ID=\"FILE_0097_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000098.jpg\"/></mets:file><mets:file ID=\"FILE_0098_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000099.jpg\"/></mets:file><mets:file ID=\"FILE_0099_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000100.jpg\"/></mets:file><mets:file ID=\"FILE_0100_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000101.jpg\"/></mets:file><mets:file ID=\"FILE_0101_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000102.jpg\"/></mets:file><mets:file ID=\"FILE_0102_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000103.jpg\"/></mets:file><mets:file ID=\"FILE_0103_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000104.jpg\"/></mets:file><mets:file ID=\"FILE_0104_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000105.jpg\"/></mets:file><mets:file ID=\"FILE_0105_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000106.jpg\"/></mets:file><mets:file ID=\"FILE_0106_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000107.jpg\"/></mets:file><mets:file ID=\"FILE_0107_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000108.jpg\"/></mets:file><mets:file ID=\"FILE_0108_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000109.jpg\"/></mets:file><mets:file ID=\"FILE_0109_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000110.jpg\"/></mets:file><mets:file ID=\"FILE_0110_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000111.jpg\"/></mets:file><mets:file ID=\"FILE_0111_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000112.jpg\"/></mets:file><mets:file ID=\"FILE_0112_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000113.jpg\"/></mets:file><mets:file ID=\"FILE_0113_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000114.jpg\"/></mets:file><mets:file ID=\"FILE_0114_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000115.jpg\"/></mets:file><mets:file ID=\"FILE_0115_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000116.jpg\"/></mets:file><mets:file ID=\"FILE_0116_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000117.jpg\"/></mets:file><mets:file ID=\"FILE_0117_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000118.jpg\"/></mets:file><mets:file ID=\"FILE_0118_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000119.jpg\"/></mets:file><mets:file ID=\"FILE_0119_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000120.jpg\"/></mets:file><mets:file ID=\"FILE_0120_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000121.jpg\"/></mets:file><mets:file ID=\"FILE_0121_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000122.jpg\"/></mets:file><mets:file ID=\"FILE_0122_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000123.jpg\"/></mets:file><mets:file ID=\"FILE_0123_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000124.jpg\"/></mets:file><mets:file ID=\"FILE_0124_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000125.jpg\"/></mets:file><mets:file ID=\"FILE_0125_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000126.jpg\"/></mets:file><mets:file ID=\"FILE_0126_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000127.jpg\"/></mets:file><mets:file ID=\"FILE_0127_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000128.jpg\"/></mets:file><mets:file ID=\"FILE_0128_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000129.jpg\"/></mets:file><mets:file ID=\"FILE_0129_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000130.jpg\"/></mets:file><mets:file ID=\"FILE_0130_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000131.jpg\"/></mets:file><mets:file ID=\"FILE_0131_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000132.jpg\"/></mets:file><mets:file ID=\"FILE_0132_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000133.jpg\"/></mets:file><mets:file ID=\"FILE_0133_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000134.jpg\"/></mets:file><mets:file ID=\"FILE_0134_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000135.jpg\"/></mets:file><mets:file ID=\"FILE_0135_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000136.jpg\"/></mets:file><mets:file ID=\"FILE_0136_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000137.jpg\"/></mets:file><mets:file ID=\"FILE_0137_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000138.jpg\"/></mets:file><mets:file ID=\"FILE_0138_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000139.jpg\"/></mets:file><mets:file ID=\"FILE_0139_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000140.jpg\"/></mets:file><mets:file ID=\"FILE_0140_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000141.jpg\"/></mets:file><mets:file ID=\"FILE_0141_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000142.jpg\"/></mets:file><mets:file ID=\"FILE_0142_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000143.jpg\"/></mets:file><mets:file ID=\"FILE_0143_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000144.jpg\"/></mets:file><mets:file ID=\"FILE_0144_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000145.jpg\"/></mets:file><mets:file ID=\"FILE_0145_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000146.jpg\"/></mets:file><mets:file ID=\"FILE_0146_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000147.jpg\"/></mets:file><mets:file ID=\"FILE_0147_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000148.jpg\"/></mets:file><mets:file ID=\"FILE_0148_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000149.jpg\"/></mets:file><mets:file ID=\"FILE_0149_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000150.jpg\"/></mets:file><mets:file ID=\"FILE_0150_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000151.jpg\"/></mets:file><mets:file ID=\"FILE_0151_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000152.jpg\"/></mets:file><mets:file ID=\"FILE_0152_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000153.jpg\"/></mets:file><mets:file ID=\"FILE_0153_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000154.jpg\"/></mets:file><mets:file ID=\"FILE_0154_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000155.jpg\"/></mets:file><mets:file ID=\"FILE_0155_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000156.jpg\"/></mets:file><mets:file ID=\"FILE_0156_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000157.jpg\"/></mets:file><mets:file ID=\"FILE_0157_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000158.jpg\"/></mets:file><mets:file ID=\"FILE_0158_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000159.jpg\"/></mets:file><mets:file ID=\"FILE_0159_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000160.jpg\"/></mets:file><mets:file ID=\"FILE_0160_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000161.jpg\"/></mets:file><mets:file ID=\"FILE_0161_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000162.jpg\"/></mets:file><mets:file ID=\"FILE_0162_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000163.jpg\"/></mets:file><mets:file ID=\"FILE_0163_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000164.jpg\"/></mets:file><mets:file ID=\"FILE_0164_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000165.jpg\"/></mets:file><mets:file ID=\"FILE_0165_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000166.jpg\"/></mets:file><mets:file ID=\"FILE_0166_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000167.jpg\"/></mets:file><mets:file ID=\"FILE_0167_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000168.jpg\"/></mets:file><mets:file ID=\"FILE_0168_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000169.jpg\"/></mets:file><mets:file ID=\"FILE_0169_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000170.jpg\"/></mets:file><mets:file ID=\"FILE_0170_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000171.jpg\"/></mets:file><mets:file ID=\"FILE_0171_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000172.jpg\"/></mets:file><mets:file ID=\"FILE_0172_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000173.jpg\"/></mets:file><mets:file ID=\"FILE_0173_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000174.jpg\"/></mets:file><mets:file ID=\"FILE_0174_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000175.jpg\"/></mets:file><mets:file ID=\"FILE_0175_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000176.jpg\"/></mets:file><mets:file ID=\"FILE_0176_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000177.jpg\"/></mets:file><mets:file ID=\"FILE_0177_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000178.jpg\"/></mets:file><mets:file ID=\"FILE_0178_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000179.jpg\"/></mets:file><mets:file ID=\"FILE_0179_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000180.jpg\"/></mets:file><mets:file ID=\"FILE_0180_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000181.jpg\"/></mets:file><mets:file ID=\"FILE_0181_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000182.jpg\"/></mets:file><mets:file ID=\"FILE_0182_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000183.jpg\"/></mets:file><mets:file ID=\"FILE_0183_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000184.jpg\"/></mets:file><mets:file ID=\"FILE_0184_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000185.jpg\"/></mets:file><mets:file ID=\"FILE_0185_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000186.jpg\"/></mets:file><mets:file ID=\"FILE_0186_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000187.jpg\"/></mets:file><mets:file ID=\"FILE_0187_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000188.jpg\"/></mets:file><mets:file ID=\"FILE_0188_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000189.jpg\"/></mets:file><mets:file ID=\"FILE_0189_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000190.jpg\"/></mets:file><mets:file ID=\"FILE_0190_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000191.jpg\"/></mets:file><mets:file ID=\"FILE_0191_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000192.jpg\"/></mets:file><mets:file ID=\"FILE_0192_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000193.jpg\"/></mets:file><mets:file ID=\"FILE_0193_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000194.jpg\"/></mets:file><mets:file ID=\"FILE_0194_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000195.jpg\"/></mets:file><mets:file ID=\"FILE_0195_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000196.jpg\"/></mets:file><mets:file ID=\"FILE_0196_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000197.jpg\"/></mets:file><mets:file ID=\"FILE_0197_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000198.jpg\"/></mets:file><mets:file ID=\"FILE_0198_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000199.jpg\"/></mets:file><mets:file ID=\"FILE_0199_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000200.jpg\"/></mets:file><mets:file ID=\"FILE_0200_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000201.jpg\"/></mets:file><mets:file ID=\"FILE_0201_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000202.jpg\"/></mets:file><mets:file ID=\"FILE_0202_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000203.jpg\"/></mets:file><mets:file ID=\"FILE_0203_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000204.jpg\"/></mets:file><mets:file ID=\"FILE_0204_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000205.jpg\"/></mets:file><mets:file ID=\"FILE_0205_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000206.jpg\"/></mets:file><mets:file ID=\"FILE_0206_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000207.jpg\"/></mets:file><mets:file ID=\"FILE_0207_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000208.jpg\"/></mets:file><mets:file ID=\"FILE_0208_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000209.jpg\"/></mets:file><mets:file ID=\"FILE_0209_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000210.jpg\"/></mets:file><mets:file ID=\"FILE_0210_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000211.jpg\"/></mets:file><mets:file ID=\"FILE_0211_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000212.jpg\"/></mets:file><mets:file ID=\"FILE_0212_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000213.jpg\"/></mets:file><mets:file ID=\"FILE_0213_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000214.jpg\"/></mets:file><mets:file ID=\"FILE_0214_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000215.jpg\"/></mets:file><mets:file ID=\"FILE_0215_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000216.jpg\"/></mets:file><mets:file ID=\"FILE_0216_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000217.jpg\"/></mets:file><mets:file ID=\"FILE_0217_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000218.jpg\"/></mets:file><mets:file ID=\"FILE_0218_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000219.jpg\"/></mets:file><mets:file ID=\"FILE_0219_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000220.jpg\"/></mets:file><mets:file ID=\"FILE_0220_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000221.jpg\"/></mets:file><mets:file ID=\"FILE_0221_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000222.jpg\"/></mets:file><mets:file ID=\"FILE_0222_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000223.jpg\"/></mets:file><mets:file ID=\"FILE_0223_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000224.jpg\"/></mets:file><mets:file ID=\"FILE_0224_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000225.jpg\"/></mets:file><mets:file ID=\"FILE_0225_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000226.jpg\"/></mets:file><mets:file ID=\"FILE_0226_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000227.jpg\"/></mets:file><mets:file ID=\"FILE_0227_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000228.jpg\"/></mets:file><mets:file ID=\"FILE_0228_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000229.jpg\"/></mets:file><mets:file ID=\"FILE_0229_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000230.jpg\"/></mets:file><mets:file ID=\"FILE_0230_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000231.jpg\"/></mets:file><mets:file ID=\"FILE_0231_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000232.jpg\"/></mets:file><mets:file ID=\"FILE_0232_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000233.jpg\"/></mets:file><mets:file ID=\"FILE_0233_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000234.jpg\"/></mets:file><mets:file ID=\"FILE_0234_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000235.jpg\"/></mets:file><mets:file ID=\"FILE_0235_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000236.jpg\"/></mets:file><mets:file ID=\"FILE_0236_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000237.jpg\"/></mets:file><mets:file ID=\"FILE_0237_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000238.jpg\"/></mets:file><mets:file ID=\"FILE_0238_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000239.jpg\"/></mets:file><mets:file ID=\"FILE_0239_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000240.jpg\"/></mets:file><mets:file ID=\"FILE_0240_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000241.jpg\"/></mets:file><mets:file ID=\"FILE_0241_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000242.jpg\"/></mets:file><mets:file ID=\"FILE_0242_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000243.jpg\"/></mets:file><mets:file ID=\"FILE_0243_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000244.jpg\"/></mets:file><mets:file ID=\"FILE_0244_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000245.jpg\"/></mets:file><mets:file ID=\"FILE_0245_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000246.jpg\"/></mets:file><mets:file ID=\"FILE_0246_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000247.jpg\"/></mets:file><mets:file ID=\"FILE_0247_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000248.jpg\"/></mets:file><mets:file ID=\"FILE_0248_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000249.jpg\"/></mets:file><mets:file ID=\"FILE_0249_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000250.jpg\"/></mets:file><mets:file ID=\"FILE_0250_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000251.jpg\"/></mets:file><mets:file ID=\"FILE_0251_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000252.jpg\"/></mets:file><mets:file ID=\"FILE_0252_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000253.jpg\"/></mets:file><mets:file ID=\"FILE_0253_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000254.jpg\"/></mets:file><mets:file ID=\"FILE_0254_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000255.jpg\"/></mets:file><mets:file ID=\"FILE_0255_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000256.jpg\"/></mets:file><mets:file ID=\"FILE_0256_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000257.jpg\"/></mets:file><mets:file ID=\"FILE_0257_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000258.jpg\"/></mets:file><mets:file ID=\"FILE_0258_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000259.jpg\"/></mets:file><mets:file ID=\"FILE_0259_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000260.jpg\"/></mets:file><mets:file ID=\"FILE_0260_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000261.jpg\"/></mets:file><mets:file ID=\"FILE_0261_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000262.jpg\"/></mets:file><mets:file ID=\"FILE_0262_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000263.jpg\"/></mets:file><mets:file ID=\"FILE_0263_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000264.jpg\"/></mets:file><mets:file ID=\"FILE_0264_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000265.jpg\"/></mets:file><mets:file ID=\"FILE_0265_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000266.jpg\"/></mets:file><mets:file ID=\"FILE_0266_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000267.jpg\"/></mets:file><mets:file ID=\"FILE_0267_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000268.jpg\"/></mets:file><mets:file ID=\"FILE_0268_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000269.jpg\"/></mets:file><mets:file ID=\"FILE_0269_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000270.jpg\"/></mets:file><mets:file ID=\"FILE_0270_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000271.jpg\"/></mets:file><mets:file ID=\"FILE_0271_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000272.jpg\"/></mets:file><mets:file ID=\"FILE_0272_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000273.jpg\"/></mets:file><mets:file ID=\"FILE_0273_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000274.jpg\"/></mets:file><mets:file ID=\"FILE_0274_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000275.jpg\"/></mets:file><mets:file ID=\"FILE_0275_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000276.jpg\"/></mets:file><mets:file ID=\"FILE_0276_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000277.jpg\"/></mets:file><mets:file ID=\"FILE_0277_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000278.jpg\"/></mets:file><mets:file ID=\"FILE_0278_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000279.jpg\"/></mets:file><mets:file ID=\"FILE_0279_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000280.jpg\"/></mets:file><mets:file ID=\"FILE_0280_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000281.jpg\"/></mets:file><mets:file ID=\"FILE_0281_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000282.jpg\"/></mets:file><mets:file ID=\"FILE_0282_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000283.jpg\"/></mets:file><mets:file ID=\"FILE_0283_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000284.jpg\"/></mets:file><mets:file ID=\"FILE_0284_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000285.jpg\"/></mets:file><mets:file ID=\"FILE_0285_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000286.jpg\"/></mets:file><mets:file ID=\"FILE_0286_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000287.jpg\"/></mets:file><mets:file ID=\"FILE_0287_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000288.jpg\"/></mets:file><mets:file ID=\"FILE_0288_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000289.jpg\"/></mets:file><mets:file ID=\"FILE_0289_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000290.jpg\"/></mets:file><mets:file ID=\"FILE_0290_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000291.jpg\"/></mets:file><mets:file ID=\"FILE_0291_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000292.jpg\"/></mets:file><mets:file ID=\"FILE_0292_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000293.jpg\"/></mets:file><mets:file ID=\"FILE_0293_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000294.jpg\"/></mets:file><mets:file ID=\"FILE_0294_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000295.jpg\"/></mets:file><mets:file ID=\"FILE_0295_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000296.jpg\"/></mets:file><mets:file ID=\"FILE_0296_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000297.jpg\"/></mets:file><mets:file ID=\"FILE_0297_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000298.jpg\"/></mets:file><mets:file ID=\"FILE_0298_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000299.jpg\"/></mets:file><mets:file ID=\"FILE_0299_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000300.jpg\"/></mets:file><mets:file ID=\"FILE_0300_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000301.jpg\"/></mets:file><mets:file ID=\"FILE_0301_DEFAULT\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1000/0/00000302.jpg\"/></mets:file></mets:fileGrp><mets:fileGrp USE=\"PRESENTATION\"><mets:file ID=\"FILE_0000_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000001.tif\"/></mets:file><mets:file ID=\"FILE_0001_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000002.tif\"/></mets:file><mets:file ID=\"FILE_0002_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000003.tif\"/></mets:file><mets:file ID=\"FILE_0003_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000004.tif\"/></mets:file><mets:file ID=\"FILE_0004_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000005.tif\"/></mets:file><mets:file ID=\"FILE_0005_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000006.tif\"/></mets:file><mets:file ID=\"FILE_0006_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000007.tif\"/></mets:file><mets:file ID=\"FILE_0007_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000008.tif\"/></mets:file><mets:file ID=\"FILE_0008_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000009.tif\"/></mets:file><mets:file ID=\"FILE_0009_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000010.tif\"/></mets:file><mets:file ID=\"FILE_0010_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000011.tif\"/></mets:file><mets:file ID=\"FILE_0011_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000012.tif\"/></mets:file><mets:file ID=\"FILE_0012_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000013.tif\"/></mets:file><mets:file ID=\"FILE_0013_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000014.tif\"/></mets:file><mets:file ID=\"FILE_0014_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000015.tif\"/></mets:file><mets:file ID=\"FILE_0015_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000016.tif\"/></mets:file><mets:file ID=\"FILE_0016_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000017.tif\"/></mets:file><mets:file ID=\"FILE_0017_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000018.tif\"/></mets:file><mets:file ID=\"FILE_0018_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000019.tif\"/></mets:file><mets:file ID=\"FILE_0019_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000020.tif\"/></mets:file><mets:file ID=\"FILE_0020_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000021.tif\"/></mets:file><mets:file ID=\"FILE_0021_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000022.tif\"/></mets:file><mets:file ID=\"FILE_0022_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000023.tif\"/></mets:file><mets:file ID=\"FILE_0023_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000024.tif\"/></mets:file><mets:file ID=\"FILE_0024_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000025.tif\"/></mets:file><mets:file ID=\"FILE_0025_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000026.tif\"/></mets:file><mets:file ID=\"FILE_0026_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000027.tif\"/></mets:file><mets:file ID=\"FILE_0027_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000028.tif\"/></mets:file><mets:file ID=\"FILE_0028_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000029.tif\"/></mets:file><mets:file ID=\"FILE_0029_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000030.tif\"/></mets:file><mets:file ID=\"FILE_0030_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000031.tif\"/></mets:file><mets:file ID=\"FILE_0031_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000032.tif\"/></mets:file><mets:file ID=\"FILE_0032_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000033.tif\"/></mets:file><mets:file ID=\"FILE_0033_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000034.tif\"/></mets:file><mets:file ID=\"FILE_0034_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000035.tif\"/></mets:file><mets:file ID=\"FILE_0035_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000036.tif\"/></mets:file><mets:file ID=\"FILE_0036_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000037.tif\"/></mets:file><mets:file ID=\"FILE_0037_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000038.tif\"/></mets:file><mets:file ID=\"FILE_0038_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000039.tif\"/></mets:file><mets:file ID=\"FILE_0039_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000040.tif\"/></mets:file><mets:file ID=\"FILE_0040_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000041.tif\"/></mets:file><mets:file ID=\"FILE_0041_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000042.tif\"/></mets:file><mets:file ID=\"FILE_0042_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000043.tif\"/></mets:file><mets:file ID=\"FILE_0043_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000044.tif\"/></mets:file><mets:file ID=\"FILE_0044_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000045.tif\"/></mets:file><mets:file ID=\"FILE_0045_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000046.tif\"/></mets:file><mets:file ID=\"FILE_0046_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000047.tif\"/></mets:file><mets:file ID=\"FILE_0047_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000048.tif\"/></mets:file><mets:file ID=\"FILE_0048_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000049.tif\"/></mets:file><mets:file ID=\"FILE_0049_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000050.tif\"/></mets:file><mets:file ID=\"FILE_0050_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000051.tif\"/></mets:file><mets:file ID=\"FILE_0051_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000052.tif\"/></mets:file><mets:file ID=\"FILE_0052_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000053.tif\"/></mets:file><mets:file ID=\"FILE_0053_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000054.tif\"/></mets:file><mets:file ID=\"FILE_0054_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000055.tif\"/></mets:file><mets:file ID=\"FILE_0055_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000056.tif\"/></mets:file><mets:file ID=\"FILE_0056_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000057.tif\"/></mets:file><mets:file ID=\"FILE_0057_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000058.tif\"/></mets:file><mets:file ID=\"FILE_0058_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000059.tif\"/></mets:file><mets:file ID=\"FILE_0059_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000060.tif\"/></mets:file><mets:file ID=\"FILE_0060_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000061.tif\"/></mets:file><mets:file ID=\"FILE_0061_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000062.tif\"/></mets:file><mets:file ID=\"FILE_0062_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000063.tif\"/></mets:file><mets:file ID=\"FILE_0063_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000064.tif\"/></mets:file><mets:file ID=\"FILE_0064_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000065.tif\"/></mets:file><mets:file ID=\"FILE_0065_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000066.tif\"/></mets:file><mets:file ID=\"FILE_0066_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000067.tif\"/></mets:file><mets:file ID=\"FILE_0067_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000068.tif\"/></mets:file><mets:file ID=\"FILE_0068_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000069.tif\"/></mets:file><mets:file ID=\"FILE_0069_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000070.tif\"/></mets:file><mets:file ID=\"FILE_0070_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000071.tif\"/></mets:file><mets:file ID=\"FILE_0071_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000072.tif\"/></mets:file><mets:file ID=\"FILE_0072_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000073.tif\"/></mets:file><mets:file ID=\"FILE_0073_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000074.tif\"/></mets:file><mets:file ID=\"FILE_0074_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000075.tif\"/></mets:file><mets:file ID=\"FILE_0075_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000076.tif\"/></mets:file><mets:file ID=\"FILE_0076_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000077.tif\"/></mets:file><mets:file ID=\"FILE_0077_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000078.tif\"/></mets:file><mets:file ID=\"FILE_0078_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000079.tif\"/></mets:file><mets:file ID=\"FILE_0079_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000080.tif\"/></mets:file><mets:file ID=\"FILE_0080_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000081.tif\"/></mets:file><mets:file ID=\"FILE_0081_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000082.tif\"/></mets:file><mets:file ID=\"FILE_0082_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000083.tif\"/></mets:file><mets:file ID=\"FILE_0083_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000084.tif\"/></mets:file><mets:file ID=\"FILE_0084_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000085.tif\"/></mets:file><mets:file ID=\"FILE_0085_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000086.tif\"/></mets:file><mets:file ID=\"FILE_0086_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000087.tif\"/></mets:file><mets:file ID=\"FILE_0087_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000088.tif\"/></mets:file><mets:file ID=\"FILE_0088_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000089.tif\"/></mets:file><mets:file ID=\"FILE_0089_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000090.tif\"/></mets:file><mets:file ID=\"FILE_0090_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000091.tif\"/></mets:file><mets:file ID=\"FILE_0091_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000092.tif\"/></mets:file><mets:file ID=\"FILE_0092_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000093.tif\"/></mets:file><mets:file ID=\"FILE_0093_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000094.tif\"/></mets:file><mets:file ID=\"FILE_0094_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000095.tif\"/></mets:file><mets:file ID=\"FILE_0095_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000096.tif\"/></mets:file><mets:file ID=\"FILE_0096_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000097.tif\"/></mets:file><mets:file ID=\"FILE_0097_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000098.tif\"/></mets:file><mets:file ID=\"FILE_0098_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000099.tif\"/></mets:file><mets:file ID=\"FILE_0099_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000100.tif\"/></mets:file><mets:file ID=\"FILE_0100_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000101.tif\"/></mets:file><mets:file ID=\"FILE_0101_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000102.tif\"/></mets:file><mets:file ID=\"FILE_0102_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000103.tif\"/></mets:file><mets:file ID=\"FILE_0103_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000104.tif\"/></mets:file><mets:file ID=\"FILE_0104_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000105.tif\"/></mets:file><mets:file ID=\"FILE_0105_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000106.tif\"/></mets:file><mets:file ID=\"FILE_0106_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000107.tif\"/></mets:file><mets:file ID=\"FILE_0107_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000108.tif\"/></mets:file><mets:file ID=\"FILE_0108_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000109.tif\"/></mets:file><mets:file ID=\"FILE_0109_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000110.tif\"/></mets:file><mets:file ID=\"FILE_0110_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000111.tif\"/></mets:file><mets:file ID=\"FILE_0111_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000112.tif\"/></mets:file><mets:file ID=\"FILE_0112_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000113.tif\"/></mets:file><mets:file ID=\"FILE_0113_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000114.tif\"/></mets:file><mets:file ID=\"FILE_0114_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000115.tif\"/></mets:file><mets:file ID=\"FILE_0115_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000116.tif\"/></mets:file><mets:file ID=\"FILE_0116_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000117.tif\"/></mets:file><mets:file ID=\"FILE_0117_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000118.tif\"/></mets:file><mets:file ID=\"FILE_0118_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000119.tif\"/></mets:file><mets:file ID=\"FILE_0119_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000120.tif\"/></mets:file><mets:file ID=\"FILE_0120_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000121.tif\"/></mets:file><mets:file ID=\"FILE_0121_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000122.tif\"/></mets:file><mets:file ID=\"FILE_0122_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000123.tif\"/></mets:file><mets:file ID=\"FILE_0123_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000124.tif\"/></mets:file><mets:file ID=\"FILE_0124_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000125.tif\"/></mets:file><mets:file ID=\"FILE_0125_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000126.tif\"/></mets:file><mets:file ID=\"FILE_0126_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000127.tif\"/></mets:file><mets:file ID=\"FILE_0127_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000128.tif\"/></mets:file><mets:file ID=\"FILE_0128_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000129.tif\"/></mets:file><mets:file ID=\"FILE_0129_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000130.tif\"/></mets:file><mets:file ID=\"FILE_0130_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000131.tif\"/></mets:file><mets:file ID=\"FILE_0131_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000132.tif\"/></mets:file><mets:file ID=\"FILE_0132_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000133.tif\"/></mets:file><mets:file ID=\"FILE_0133_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000134.tif\"/></mets:file><mets:file ID=\"FILE_0134_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000135.tif\"/></mets:file><mets:file ID=\"FILE_0135_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000136.tif\"/></mets:file><mets:file ID=\"FILE_0136_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000137.tif\"/></mets:file><mets:file ID=\"FILE_0137_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000138.tif\"/></mets:file><mets:file ID=\"FILE_0138_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000139.tif\"/></mets:file><mets:file ID=\"FILE_0139_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000140.tif\"/></mets:file><mets:file ID=\"FILE_0140_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000141.tif\"/></mets:file><mets:file ID=\"FILE_0141_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000142.tif\"/></mets:file><mets:file ID=\"FILE_0142_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000143.tif\"/></mets:file><mets:file ID=\"FILE_0143_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000144.tif\"/></mets:file><mets:file ID=\"FILE_0144_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000145.tif\"/></mets:file><mets:file ID=\"FILE_0145_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000146.tif\"/></mets:file><mets:file ID=\"FILE_0146_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000147.tif\"/></mets:file><mets:file ID=\"FILE_0147_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000148.tif\"/></mets:file><mets:file ID=\"FILE_0148_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000149.tif\"/></mets:file><mets:file ID=\"FILE_0149_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000150.tif\"/></mets:file><mets:file ID=\"FILE_0150_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000151.tif\"/></mets:file><mets:file ID=\"FILE_0151_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000152.tif\"/></mets:file><mets:file ID=\"FILE_0152_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000153.tif\"/></mets:file><mets:file ID=\"FILE_0153_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000154.tif\"/></mets:file><mets:file ID=\"FILE_0154_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000155.tif\"/></mets:file><mets:file ID=\"FILE_0155_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000156.tif\"/></mets:file><mets:file ID=\"FILE_0156_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000157.tif\"/></mets:file><mets:file ID=\"FILE_0157_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000158.tif\"/></mets:file><mets:file ID=\"FILE_0158_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000159.tif\"/></mets:file><mets:file ID=\"FILE_0159_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000160.tif\"/></mets:file><mets:file ID=\"FILE_0160_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000161.tif\"/></mets:file><mets:file ID=\"FILE_0161_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000162.tif\"/></mets:file><mets:file ID=\"FILE_0162_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000163.tif\"/></mets:file><mets:file ID=\"FILE_0163_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000164.tif\"/></mets:file><mets:file ID=\"FILE_0164_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000165.tif\"/></mets:file><mets:file ID=\"FILE_0165_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000166.tif\"/></mets:file><mets:file ID=\"FILE_0166_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000167.tif\"/></mets:file><mets:file ID=\"FILE_0167_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000168.tif\"/></mets:file><mets:file ID=\"FILE_0168_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000169.tif\"/></mets:file><mets:file ID=\"FILE_0169_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000170.tif\"/></mets:file><mets:file ID=\"FILE_0170_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000171.tif\"/></mets:file><mets:file ID=\"FILE_0171_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000172.tif\"/></mets:file><mets:file ID=\"FILE_0172_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000173.tif\"/></mets:file><mets:file ID=\"FILE_0173_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000174.tif\"/></mets:file><mets:file ID=\"FILE_0174_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000175.tif\"/></mets:file><mets:file ID=\"FILE_0175_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000176.tif\"/></mets:file><mets:file ID=\"FILE_0176_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000177.tif\"/></mets:file><mets:file ID=\"FILE_0177_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000178.tif\"/></mets:file><mets:file ID=\"FILE_0178_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000179.tif\"/></mets:file><mets:file ID=\"FILE_0179_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000180.tif\"/></mets:file><mets:file ID=\"FILE_0180_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000181.tif\"/></mets:file><mets:file ID=\"FILE_0181_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000182.tif\"/></mets:file><mets:file ID=\"FILE_0182_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000183.tif\"/></mets:file><mets:file ID=\"FILE_0183_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000184.tif\"/></mets:file><mets:file ID=\"FILE_0184_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000185.tif\"/></mets:file><mets:file ID=\"FILE_0185_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000186.tif\"/></mets:file><mets:file ID=\"FILE_0186_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000187.tif\"/></mets:file><mets:file ID=\"FILE_0187_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000188.tif\"/></mets:file><mets:file ID=\"FILE_0188_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000189.tif\"/></mets:file><mets:file ID=\"FILE_0189_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000190.tif\"/></mets:file><mets:file ID=\"FILE_0190_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000191.tif\"/></mets:file><mets:file ID=\"FILE_0191_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000192.tif\"/></mets:file><mets:file ID=\"FILE_0192_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000193.tif\"/></mets:file><mets:file ID=\"FILE_0193_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000194.tif\"/></mets:file><mets:file ID=\"FILE_0194_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000195.tif\"/></mets:file><mets:file ID=\"FILE_0195_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000196.tif\"/></mets:file><mets:file ID=\"FILE_0196_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000197.tif\"/></mets:file><mets:file ID=\"FILE_0197_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000198.tif\"/></mets:file><mets:file ID=\"FILE_0198_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000199.tif\"/></mets:file><mets:file ID=\"FILE_0199_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000200.tif\"/></mets:file><mets:file ID=\"FILE_0200_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000201.tif\"/></mets:file><mets:file ID=\"FILE_0201_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000202.tif\"/></mets:file><mets:file ID=\"FILE_0202_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000203.tif\"/></mets:file><mets:file ID=\"FILE_0203_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000204.tif\"/></mets:file><mets:file ID=\"FILE_0204_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000205.tif\"/></mets:file><mets:file ID=\"FILE_0205_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000206.tif\"/></mets:file><mets:file ID=\"FILE_0206_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000207.tif\"/></mets:file><mets:file ID=\"FILE_0207_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000208.tif\"/></mets:file><mets:file ID=\"FILE_0208_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000209.tif\"/></mets:file><mets:file ID=\"FILE_0209_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000210.tif\"/></mets:file><mets:file ID=\"FILE_0210_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000211.tif\"/></mets:file><mets:file ID=\"FILE_0211_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000212.tif\"/></mets:file><mets:file ID=\"FILE_0212_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000213.tif\"/></mets:file><mets:file ID=\"FILE_0213_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000214.tif\"/></mets:file><mets:file ID=\"FILE_0214_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000215.tif\"/></mets:file><mets:file ID=\"FILE_0215_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000216.tif\"/></mets:file><mets:file ID=\"FILE_0216_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000217.tif\"/></mets:file><mets:file ID=\"FILE_0217_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000218.tif\"/></mets:file><mets:file ID=\"FILE_0218_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000219.tif\"/></mets:file><mets:file ID=\"FILE_0219_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000220.tif\"/></mets:file><mets:file ID=\"FILE_0220_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000221.tif\"/></mets:file><mets:file ID=\"FILE_0221_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000222.tif\"/></mets:file><mets:file ID=\"FILE_0222_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000223.tif\"/></mets:file><mets:file ID=\"FILE_0223_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000224.tif\"/></mets:file><mets:file ID=\"FILE_0224_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000225.tif\"/></mets:file><mets:file ID=\"FILE_0225_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000226.tif\"/></mets:file><mets:file ID=\"FILE_0226_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000227.tif\"/></mets:file><mets:file ID=\"FILE_0227_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000228.tif\"/></mets:file><mets:file ID=\"FILE_0228_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000229.tif\"/></mets:file><mets:file ID=\"FILE_0229_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000230.tif\"/></mets:file><mets:file ID=\"FILE_0230_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000231.tif\"/></mets:file><mets:file ID=\"FILE_0231_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000232.tif\"/></mets:file><mets:file ID=\"FILE_0232_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000233.tif\"/></mets:file><mets:file ID=\"FILE_0233_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000234.tif\"/></mets:file><mets:file ID=\"FILE_0234_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000235.tif\"/></mets:file><mets:file ID=\"FILE_0235_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000236.tif\"/></mets:file><mets:file ID=\"FILE_0236_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000237.tif\"/></mets:file><mets:file ID=\"FILE_0237_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000238.tif\"/></mets:file><mets:file ID=\"FILE_0238_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000239.tif\"/></mets:file><mets:file ID=\"FILE_0239_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000240.tif\"/></mets:file><mets:file ID=\"FILE_0240_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000241.tif\"/></mets:file><mets:file ID=\"FILE_0241_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000242.tif\"/></mets:file><mets:file ID=\"FILE_0242_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000243.tif\"/></mets:file><mets:file ID=\"FILE_0243_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000244.tif\"/></mets:file><mets:file ID=\"FILE_0244_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000245.tif\"/></mets:file><mets:file ID=\"FILE_0245_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000246.tif\"/></mets:file><mets:file ID=\"FILE_0246_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000247.tif\"/></mets:file><mets:file ID=\"FILE_0247_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000248.tif\"/></mets:file><mets:file ID=\"FILE_0248_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000249.tif\"/></mets:file><mets:file ID=\"FILE_0249_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000250.tif\"/></mets:file><mets:file ID=\"FILE_0250_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000251.tif\"/></mets:file><mets:file ID=\"FILE_0251_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000252.tif\"/></mets:file><mets:file ID=\"FILE_0252_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000253.tif\"/></mets:file><mets:file ID=\"FILE_0253_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000254.tif\"/></mets:file><mets:file ID=\"FILE_0254_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000255.tif\"/></mets:file><mets:file ID=\"FILE_0255_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000256.tif\"/></mets:file><mets:file ID=\"FILE_0256_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000257.tif\"/></mets:file><mets:file ID=\"FILE_0257_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000258.tif\"/></mets:file><mets:file ID=\"FILE_0258_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000259.tif\"/></mets:file><mets:file ID=\"FILE_0259_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000260.tif\"/></mets:file><mets:file ID=\"FILE_0260_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000261.tif\"/></mets:file><mets:file ID=\"FILE_0261_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000262.tif\"/></mets:file><mets:file ID=\"FILE_0262_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000263.tif\"/></mets:file><mets:file ID=\"FILE_0263_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000264.tif\"/></mets:file><mets:file ID=\"FILE_0264_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000265.tif\"/></mets:file><mets:file ID=\"FILE_0265_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000266.tif\"/></mets:file><mets:file ID=\"FILE_0266_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000267.tif\"/></mets:file><mets:file ID=\"FILE_0267_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000268.tif\"/></mets:file><mets:file ID=\"FILE_0268_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000269.tif\"/></mets:file><mets:file ID=\"FILE_0269_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000270.tif\"/></mets:file><mets:file ID=\"FILE_0270_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000271.tif\"/></mets:file><mets:file ID=\"FILE_0271_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000272.tif\"/></mets:file><mets:file ID=\"FILE_0272_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000273.tif\"/></mets:file><mets:file ID=\"FILE_0273_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000274.tif\"/></mets:file><mets:file ID=\"FILE_0274_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000275.tif\"/></mets:file><mets:file ID=\"FILE_0275_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000276.tif\"/></mets:file><mets:file ID=\"FILE_0276_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000277.tif\"/></mets:file><mets:file ID=\"FILE_0277_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000278.tif\"/></mets:file><mets:file ID=\"FILE_0278_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000279.tif\"/></mets:file><mets:file ID=\"FILE_0279_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000280.tif\"/></mets:file><mets:file ID=\"FILE_0280_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000281.tif\"/></mets:file><mets:file ID=\"FILE_0281_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000282.tif\"/></mets:file><mets:file ID=\"FILE_0282_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000283.tif\"/></mets:file><mets:file ID=\"FILE_0283_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000284.tif\"/></mets:file><mets:file ID=\"FILE_0284_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000285.tif\"/></mets:file><mets:file ID=\"FILE_0285_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000286.tif\"/></mets:file><mets:file ID=\"FILE_0286_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000287.tif\"/></mets:file><mets:file ID=\"FILE_0287_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000288.tif\"/></mets:file><mets:file ID=\"FILE_0288_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000289.tif\"/></mets:file><mets:file ID=\"FILE_0289_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000290.tif\"/></mets:file><mets:file ID=\"FILE_0290_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000291.tif\"/></mets:file><mets:file ID=\"FILE_0291_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000292.tif\"/></mets:file><mets:file ID=\"FILE_0292_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000293.tif\"/></mets:file><mets:file ID=\"FILE_0293_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000294.tif\"/></mets:file><mets:file ID=\"FILE_0294_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000295.tif\"/></mets:file><mets:file ID=\"FILE_0295_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000296.tif\"/></mets:file><mets:file ID=\"FILE_0296_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000297.tif\"/></mets:file><mets:file ID=\"FILE_0297_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000298.tif\"/></mets:file><mets:file ID=\"FILE_0298_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000299.tif\"/></mets:file><mets:file ID=\"FILE_0299_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000300.tif\"/></mets:file><mets:file ID=\"FILE_0300_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000301.tif\"/></mets:file><mets:file ID=\"FILE_0301_PRESENTATION\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/tiff/PPN137363540/00000302.tif\"/></mets:file></mets:fileGrp><mets:fileGrp USE=\"MAX\"><mets:file ID=\"FILE_0000_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000001.jpg\"/></mets:file><mets:file ID=\"FILE_0001_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000002.jpg\"/></mets:file><mets:file ID=\"FILE_0002_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000003.jpg\"/></mets:file><mets:file ID=\"FILE_0003_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000004.jpg\"/></mets:file><mets:file ID=\"FILE_0004_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000005.jpg\"/></mets:file><mets:file ID=\"FILE_0005_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000006.jpg\"/></mets:file><mets:file ID=\"FILE_0006_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000007.jpg\"/></mets:file><mets:file ID=\"FILE_0007_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000008.jpg\"/></mets:file><mets:file ID=\"FILE_0008_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000009.jpg\"/></mets:file><mets:file ID=\"FILE_0009_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000010.jpg\"/></mets:file><mets:file ID=\"FILE_0010_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000011.jpg\"/></mets:file><mets:file ID=\"FILE_0011_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000012.jpg\"/></mets:file><mets:file ID=\"FILE_0012_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000013.jpg\"/></mets:file><mets:file ID=\"FILE_0013_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000014.jpg\"/></mets:file><mets:file ID=\"FILE_0014_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000015.jpg\"/></mets:file><mets:file ID=\"FILE_0015_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000016.jpg\"/></mets:file><mets:file ID=\"FILE_0016_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000017.jpg\"/></mets:file><mets:file ID=\"FILE_0017_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000018.jpg\"/></mets:file><mets:file ID=\"FILE_0018_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000019.jpg\"/></mets:file><mets:file ID=\"FILE_0019_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000020.jpg\"/></mets:file><mets:file ID=\"FILE_0020_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000021.jpg\"/></mets:file><mets:file ID=\"FILE_0021_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000022.jpg\"/></mets:file><mets:file ID=\"FILE_0022_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000023.jpg\"/></mets:file><mets:file ID=\"FILE_0023_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000024.jpg\"/></mets:file><mets:file ID=\"FILE_0024_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000025.jpg\"/></mets:file><mets:file ID=\"FILE_0025_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000026.jpg\"/></mets:file><mets:file ID=\"FILE_0026_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000027.jpg\"/></mets:file><mets:file ID=\"FILE_0027_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000028.jpg\"/></mets:file><mets:file ID=\"FILE_0028_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000029.jpg\"/></mets:file><mets:file ID=\"FILE_0029_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000030.jpg\"/></mets:file><mets:file ID=\"FILE_0030_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000031.jpg\"/></mets:file><mets:file ID=\"FILE_0031_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000032.jpg\"/></mets:file><mets:file ID=\"FILE_0032_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000033.jpg\"/></mets:file><mets:file ID=\"FILE_0033_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000034.jpg\"/></mets:file><mets:file ID=\"FILE_0034_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000035.jpg\"/></mets:file><mets:file ID=\"FILE_0035_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000036.jpg\"/></mets:file><mets:file ID=\"FILE_0036_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000037.jpg\"/></mets:file><mets:file ID=\"FILE_0037_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000038.jpg\"/></mets:file><mets:file ID=\"FILE_0038_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000039.jpg\"/></mets:file><mets:file ID=\"FILE_0039_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000040.jpg\"/></mets:file><mets:file ID=\"FILE_0040_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000041.jpg\"/></mets:file><mets:file ID=\"FILE_0041_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000042.jpg\"/></mets:file><mets:file ID=\"FILE_0042_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000043.jpg\"/></mets:file><mets:file ID=\"FILE_0043_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000044.jpg\"/></mets:file><mets:file ID=\"FILE_0044_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000045.jpg\"/></mets:file><mets:file ID=\"FILE_0045_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000046.jpg\"/></mets:file><mets:file ID=\"FILE_0046_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000047.jpg\"/></mets:file><mets:file ID=\"FILE_0047_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000048.jpg\"/></mets:file><mets:file ID=\"FILE_0048_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000049.jpg\"/></mets:file><mets:file ID=\"FILE_0049_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000050.jpg\"/></mets:file><mets:file ID=\"FILE_0050_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000051.jpg\"/></mets:file><mets:file ID=\"FILE_0051_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000052.jpg\"/></mets:file><mets:file ID=\"FILE_0052_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000053.jpg\"/></mets:file><mets:file ID=\"FILE_0053_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000054.jpg\"/></mets:file><mets:file ID=\"FILE_0054_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000055.jpg\"/></mets:file><mets:file ID=\"FILE_0055_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000056.jpg\"/></mets:file><mets:file ID=\"FILE_0056_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000057.jpg\"/></mets:file><mets:file ID=\"FILE_0057_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000058.jpg\"/></mets:file><mets:file ID=\"FILE_0058_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000059.jpg\"/></mets:file><mets:file ID=\"FILE_0059_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000060.jpg\"/></mets:file><mets:file ID=\"FILE_0060_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000061.jpg\"/></mets:file><mets:file ID=\"FILE_0061_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000062.jpg\"/></mets:file><mets:file ID=\"FILE_0062_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000063.jpg\"/></mets:file><mets:file ID=\"FILE_0063_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000064.jpg\"/></mets:file><mets:file ID=\"FILE_0064_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000065.jpg\"/></mets:file><mets:file ID=\"FILE_0065_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000066.jpg\"/></mets:file><mets:file ID=\"FILE_0066_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000067.jpg\"/></mets:file><mets:file ID=\"FILE_0067_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000068.jpg\"/></mets:file><mets:file ID=\"FILE_0068_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000069.jpg\"/></mets:file><mets:file ID=\"FILE_0069_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000070.jpg\"/></mets:file><mets:file ID=\"FILE_0070_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000071.jpg\"/></mets:file><mets:file ID=\"FILE_0071_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000072.jpg\"/></mets:file><mets:file ID=\"FILE_0072_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000073.jpg\"/></mets:file><mets:file ID=\"FILE_0073_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000074.jpg\"/></mets:file><mets:file ID=\"FILE_0074_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000075.jpg\"/></mets:file><mets:file ID=\"FILE_0075_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000076.jpg\"/></mets:file><mets:file ID=\"FILE_0076_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000077.jpg\"/></mets:file><mets:file ID=\"FILE_0077_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000078.jpg\"/></mets:file><mets:file ID=\"FILE_0078_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000079.jpg\"/></mets:file><mets:file ID=\"FILE_0079_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000080.jpg\"/></mets:file><mets:file ID=\"FILE_0080_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000081.jpg\"/></mets:file><mets:file ID=\"FILE_0081_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000082.jpg\"/></mets:file><mets:file ID=\"FILE_0082_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000083.jpg\"/></mets:file><mets:file ID=\"FILE_0083_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000084.jpg\"/></mets:file><mets:file ID=\"FILE_0084_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000085.jpg\"/></mets:file><mets:file ID=\"FILE_0085_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000086.jpg\"/></mets:file><mets:file ID=\"FILE_0086_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000087.jpg\"/></mets:file><mets:file ID=\"FILE_0087_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000088.jpg\"/></mets:file><mets:file ID=\"FILE_0088_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000089.jpg\"/></mets:file><mets:file ID=\"FILE_0089_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000090.jpg\"/></mets:file><mets:file ID=\"FILE_0090_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000091.jpg\"/></mets:file><mets:file ID=\"FILE_0091_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000092.jpg\"/></mets:file><mets:file ID=\"FILE_0092_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000093.jpg\"/></mets:file><mets:file ID=\"FILE_0093_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000094.jpg\"/></mets:file><mets:file ID=\"FILE_0094_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000095.jpg\"/></mets:file><mets:file ID=\"FILE_0095_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000096.jpg\"/></mets:file><mets:file ID=\"FILE_0096_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000097.jpg\"/></mets:file><mets:file ID=\"FILE_0097_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000098.jpg\"/></mets:file><mets:file ID=\"FILE_0098_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000099.jpg\"/></mets:file><mets:file ID=\"FILE_0099_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000100.jpg\"/></mets:file><mets:file ID=\"FILE_0100_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000101.jpg\"/></mets:file><mets:file ID=\"FILE_0101_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000102.jpg\"/></mets:file><mets:file ID=\"FILE_0102_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000103.jpg\"/></mets:file><mets:file ID=\"FILE_0103_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000104.jpg\"/></mets:file><mets:file ID=\"FILE_0104_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000105.jpg\"/></mets:file><mets:file ID=\"FILE_0105_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000106.jpg\"/></mets:file><mets:file ID=\"FILE_0106_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000107.jpg\"/></mets:file><mets:file ID=\"FILE_0107_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000108.jpg\"/></mets:file><mets:file ID=\"FILE_0108_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000109.jpg\"/></mets:file><mets:file ID=\"FILE_0109_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000110.jpg\"/></mets:file><mets:file ID=\"FILE_0110_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000111.jpg\"/></mets:file><mets:file ID=\"FILE_0111_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000112.jpg\"/></mets:file><mets:file ID=\"FILE_0112_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000113.jpg\"/></mets:file><mets:file ID=\"FILE_0113_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000114.jpg\"/></mets:file><mets:file ID=\"FILE_0114_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000115.jpg\"/></mets:file><mets:file ID=\"FILE_0115_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000116.jpg\"/></mets:file><mets:file ID=\"FILE_0116_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000117.jpg\"/></mets:file><mets:file ID=\"FILE_0117_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000118.jpg\"/></mets:file><mets:file ID=\"FILE_0118_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000119.jpg\"/></mets:file><mets:file ID=\"FILE_0119_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000120.jpg\"/></mets:file><mets:file ID=\"FILE_0120_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000121.jpg\"/></mets:file><mets:file ID=\"FILE_0121_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000122.jpg\"/></mets:file><mets:file ID=\"FILE_0122_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000123.jpg\"/></mets:file><mets:file ID=\"FILE_0123_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000124.jpg\"/></mets:file><mets:file ID=\"FILE_0124_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000125.jpg\"/></mets:file><mets:file ID=\"FILE_0125_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000126.jpg\"/></mets:file><mets:file ID=\"FILE_0126_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000127.jpg\"/></mets:file><mets:file ID=\"FILE_0127_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000128.jpg\"/></mets:file><mets:file ID=\"FILE_0128_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000129.jpg\"/></mets:file><mets:file ID=\"FILE_0129_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000130.jpg\"/></mets:file><mets:file ID=\"FILE_0130_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000131.jpg\"/></mets:file><mets:file ID=\"FILE_0131_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000132.jpg\"/></mets:file><mets:file ID=\"FILE_0132_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000133.jpg\"/></mets:file><mets:file ID=\"FILE_0133_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000134.jpg\"/></mets:file><mets:file ID=\"FILE_0134_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000135.jpg\"/></mets:file><mets:file ID=\"FILE_0135_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000136.jpg\"/></mets:file><mets:file ID=\"FILE_0136_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000137.jpg\"/></mets:file><mets:file ID=\"FILE_0137_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000138.jpg\"/></mets:file><mets:file ID=\"FILE_0138_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000139.jpg\"/></mets:file><mets:file ID=\"FILE_0139_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000140.jpg\"/></mets:file><mets:file ID=\"FILE_0140_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000141.jpg\"/></mets:file><mets:file ID=\"FILE_0141_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000142.jpg\"/></mets:file><mets:file ID=\"FILE_0142_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000143.jpg\"/></mets:file><mets:file ID=\"FILE_0143_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000144.jpg\"/></mets:file><mets:file ID=\"FILE_0144_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000145.jpg\"/></mets:file><mets:file ID=\"FILE_0145_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000146.jpg\"/></mets:file><mets:file ID=\"FILE_0146_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000147.jpg\"/></mets:file><mets:file ID=\"FILE_0147_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000148.jpg\"/></mets:file><mets:file ID=\"FILE_0148_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000149.jpg\"/></mets:file><mets:file ID=\"FILE_0149_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000150.jpg\"/></mets:file><mets:file ID=\"FILE_0150_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000151.jpg\"/></mets:file><mets:file ID=\"FILE_0151_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000152.jpg\"/></mets:file><mets:file ID=\"FILE_0152_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000153.jpg\"/></mets:file><mets:file ID=\"FILE_0153_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000154.jpg\"/></mets:file><mets:file ID=\"FILE_0154_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000155.jpg\"/></mets:file><mets:file ID=\"FILE_0155_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000156.jpg\"/></mets:file><mets:file ID=\"FILE_0156_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000157.jpg\"/></mets:file><mets:file ID=\"FILE_0157_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000158.jpg\"/></mets:file><mets:file ID=\"FILE_0158_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000159.jpg\"/></mets:file><mets:file ID=\"FILE_0159_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000160.jpg\"/></mets:file><mets:file ID=\"FILE_0160_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000161.jpg\"/></mets:file><mets:file ID=\"FILE_0161_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000162.jpg\"/></mets:file><mets:file ID=\"FILE_0162_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000163.jpg\"/></mets:file><mets:file ID=\"FILE_0163_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000164.jpg\"/></mets:file><mets:file ID=\"FILE_0164_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000165.jpg\"/></mets:file><mets:file ID=\"FILE_0165_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000166.jpg\"/></mets:file><mets:file ID=\"FILE_0166_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000167.jpg\"/></mets:file><mets:file ID=\"FILE_0167_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000168.jpg\"/></mets:file><mets:file ID=\"FILE_0168_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000169.jpg\"/></mets:file><mets:file ID=\"FILE_0169_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000170.jpg\"/></mets:file><mets:file ID=\"FILE_0170_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000171.jpg\"/></mets:file><mets:file ID=\"FILE_0171_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000172.jpg\"/></mets:file><mets:file ID=\"FILE_0172_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000173.jpg\"/></mets:file><mets:file ID=\"FILE_0173_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000174.jpg\"/></mets:file><mets:file ID=\"FILE_0174_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000175.jpg\"/></mets:file><mets:file ID=\"FILE_0175_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000176.jpg\"/></mets:file><mets:file ID=\"FILE_0176_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000177.jpg\"/></mets:file><mets:file ID=\"FILE_0177_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000178.jpg\"/></mets:file><mets:file ID=\"FILE_0178_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000179.jpg\"/></mets:file><mets:file ID=\"FILE_0179_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000180.jpg\"/></mets:file><mets:file ID=\"FILE_0180_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000181.jpg\"/></mets:file><mets:file ID=\"FILE_0181_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000182.jpg\"/></mets:file><mets:file ID=\"FILE_0182_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000183.jpg\"/></mets:file><mets:file ID=\"FILE_0183_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000184.jpg\"/></mets:file><mets:file ID=\"FILE_0184_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000185.jpg\"/></mets:file><mets:file ID=\"FILE_0185_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000186.jpg\"/></mets:file><mets:file ID=\"FILE_0186_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000187.jpg\"/></mets:file><mets:file ID=\"FILE_0187_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000188.jpg\"/></mets:file><mets:file ID=\"FILE_0188_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000189.jpg\"/></mets:file><mets:file ID=\"FILE_0189_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000190.jpg\"/></mets:file><mets:file ID=\"FILE_0190_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000191.jpg\"/></mets:file><mets:file ID=\"FILE_0191_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000192.jpg\"/></mets:file><mets:file ID=\"FILE_0192_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000193.jpg\"/></mets:file><mets:file ID=\"FILE_0193_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000194.jpg\"/></mets:file><mets:file ID=\"FILE_0194_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000195.jpg\"/></mets:file><mets:file ID=\"FILE_0195_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000196.jpg\"/></mets:file><mets:file ID=\"FILE_0196_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000197.jpg\"/></mets:file><mets:file ID=\"FILE_0197_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000198.jpg\"/></mets:file><mets:file ID=\"FILE_0198_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000199.jpg\"/></mets:file><mets:file ID=\"FILE_0199_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000200.jpg\"/></mets:file><mets:file ID=\"FILE_0200_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000201.jpg\"/></mets:file><mets:file ID=\"FILE_0201_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000202.jpg\"/></mets:file><mets:file ID=\"FILE_0202_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000203.jpg\"/></mets:file><mets:file ID=\"FILE_0203_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000204.jpg\"/></mets:file><mets:file ID=\"FILE_0204_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000205.jpg\"/></mets:file><mets:file ID=\"FILE_0205_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000206.jpg\"/></mets:file><mets:file ID=\"FILE_0206_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000207.jpg\"/></mets:file><mets:file ID=\"FILE_0207_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000208.jpg\"/></mets:file><mets:file ID=\"FILE_0208_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000209.jpg\"/></mets:file><mets:file ID=\"FILE_0209_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000210.jpg\"/></mets:file><mets:file ID=\"FILE_0210_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000211.jpg\"/></mets:file><mets:file ID=\"FILE_0211_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000212.jpg\"/></mets:file><mets:file ID=\"FILE_0212_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000213.jpg\"/></mets:file><mets:file ID=\"FILE_0213_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000214.jpg\"/></mets:file><mets:file ID=\"FILE_0214_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000215.jpg\"/></mets:file><mets:file ID=\"FILE_0215_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000216.jpg\"/></mets:file><mets:file ID=\"FILE_0216_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000217.jpg\"/></mets:file><mets:file ID=\"FILE_0217_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000218.jpg\"/></mets:file><mets:file ID=\"FILE_0218_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000219.jpg\"/></mets:file><mets:file ID=\"FILE_0219_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000220.jpg\"/></mets:file><mets:file ID=\"FILE_0220_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000221.jpg\"/></mets:file><mets:file ID=\"FILE_0221_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000222.jpg\"/></mets:file><mets:file ID=\"FILE_0222_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000223.jpg\"/></mets:file><mets:file ID=\"FILE_0223_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000224.jpg\"/></mets:file><mets:file ID=\"FILE_0224_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000225.jpg\"/></mets:file><mets:file ID=\"FILE_0225_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000226.jpg\"/></mets:file><mets:file ID=\"FILE_0226_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000227.jpg\"/></mets:file><mets:file ID=\"FILE_0227_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000228.jpg\"/></mets:file><mets:file ID=\"FILE_0228_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000229.jpg\"/></mets:file><mets:file ID=\"FILE_0229_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000230.jpg\"/></mets:file><mets:file ID=\"FILE_0230_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000231.jpg\"/></mets:file><mets:file ID=\"FILE_0231_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000232.jpg\"/></mets:file><mets:file ID=\"FILE_0232_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000233.jpg\"/></mets:file><mets:file ID=\"FILE_0233_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000234.jpg\"/></mets:file><mets:file ID=\"FILE_0234_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000235.jpg\"/></mets:file><mets:file ID=\"FILE_0235_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000236.jpg\"/></mets:file><mets:file ID=\"FILE_0236_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000237.jpg\"/></mets:file><mets:file ID=\"FILE_0237_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000238.jpg\"/></mets:file><mets:file ID=\"FILE_0238_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000239.jpg\"/></mets:file><mets:file ID=\"FILE_0239_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000240.jpg\"/></mets:file><mets:file ID=\"FILE_0240_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000241.jpg\"/></mets:file><mets:file ID=\"FILE_0241_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000242.jpg\"/></mets:file><mets:file ID=\"FILE_0242_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000243.jpg\"/></mets:file><mets:file ID=\"FILE_0243_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000244.jpg\"/></mets:file><mets:file ID=\"FILE_0244_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000245.jpg\"/></mets:file><mets:file ID=\"FILE_0245_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000246.jpg\"/></mets:file><mets:file ID=\"FILE_0246_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000247.jpg\"/></mets:file><mets:file ID=\"FILE_0247_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000248.jpg\"/></mets:file><mets:file ID=\"FILE_0248_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000249.jpg\"/></mets:file><mets:file ID=\"FILE_0249_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000250.jpg\"/></mets:file><mets:file ID=\"FILE_0250_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000251.jpg\"/></mets:file><mets:file ID=\"FILE_0251_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000252.jpg\"/></mets:file><mets:file ID=\"FILE_0252_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000253.jpg\"/></mets:file><mets:file ID=\"FILE_0253_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000254.jpg\"/></mets:file><mets:file ID=\"FILE_0254_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000255.jpg\"/></mets:file><mets:file ID=\"FILE_0255_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000256.jpg\"/></mets:file><mets:file ID=\"FILE_0256_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000257.jpg\"/></mets:file><mets:file ID=\"FILE_0257_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000258.jpg\"/></mets:file><mets:file ID=\"FILE_0258_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000259.jpg\"/></mets:file><mets:file ID=\"FILE_0259_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000260.jpg\"/></mets:file><mets:file ID=\"FILE_0260_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000261.jpg\"/></mets:file><mets:file ID=\"FILE_0261_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000262.jpg\"/></mets:file><mets:file ID=\"FILE_0262_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000263.jpg\"/></mets:file><mets:file ID=\"FILE_0263_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000264.jpg\"/></mets:file><mets:file ID=\"FILE_0264_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000265.jpg\"/></mets:file><mets:file ID=\"FILE_0265_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000266.jpg\"/></mets:file><mets:file ID=\"FILE_0266_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000267.jpg\"/></mets:file><mets:file ID=\"FILE_0267_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000268.jpg\"/></mets:file><mets:file ID=\"FILE_0268_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000269.jpg\"/></mets:file><mets:file ID=\"FILE_0269_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000270.jpg\"/></mets:file><mets:file ID=\"FILE_0270_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000271.jpg\"/></mets:file><mets:file ID=\"FILE_0271_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000272.jpg\"/></mets:file><mets:file ID=\"FILE_0272_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000273.jpg\"/></mets:file><mets:file ID=\"FILE_0273_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000274.jpg\"/></mets:file><mets:file ID=\"FILE_0274_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000275.jpg\"/></mets:file><mets:file ID=\"FILE_0275_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000276.jpg\"/></mets:file><mets:file ID=\"FILE_0276_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000277.jpg\"/></mets:file><mets:file ID=\"FILE_0277_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000278.jpg\"/></mets:file><mets:file ID=\"FILE_0278_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000279.jpg\"/></mets:file><mets:file ID=\"FILE_0279_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000280.jpg\"/></mets:file><mets:file ID=\"FILE_0280_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000281.jpg\"/></mets:file><mets:file ID=\"FILE_0281_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000282.jpg\"/></mets:file><mets:file ID=\"FILE_0282_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000283.jpg\"/></mets:file><mets:file ID=\"FILE_0283_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000284.jpg\"/></mets:file><mets:file ID=\"FILE_0284_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000285.jpg\"/></mets:file><mets:file ID=\"FILE_0285_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000286.jpg\"/></mets:file><mets:file ID=\"FILE_0286_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000287.jpg\"/></mets:file><mets:file ID=\"FILE_0287_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000288.jpg\"/></mets:file><mets:file ID=\"FILE_0288_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000289.jpg\"/></mets:file><mets:file ID=\"FILE_0289_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000290.jpg\"/></mets:file><mets:file ID=\"FILE_0290_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000291.jpg\"/></mets:file><mets:file ID=\"FILE_0291_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000292.jpg\"/></mets:file><mets:file ID=\"FILE_0292_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000293.jpg\"/></mets:file><mets:file ID=\"FILE_0293_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000294.jpg\"/></mets:file><mets:file ID=\"FILE_0294_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000295.jpg\"/></mets:file><mets:file ID=\"FILE_0295_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000296.jpg\"/></mets:file><mets:file ID=\"FILE_0296_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000297.jpg\"/></mets:file><mets:file ID=\"FILE_0297_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000298.jpg\"/></mets:file><mets:file ID=\"FILE_0298_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000299.jpg\"/></mets:file><mets:file ID=\"FILE_0299_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000300.jpg\"/></mets:file><mets:file ID=\"FILE_0300_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000301.jpg\"/></mets:file><mets:file ID=\"FILE_0301_MAX\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/1200/0/00000302.jpg\"/></mets:file></mets:fileGrp><mets:fileGrp USE=\"THUMBS\"><mets:file ID=\"FILE_0000_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000001.jpg\"/></mets:file><mets:file ID=\"FILE_0001_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000002.jpg\"/></mets:file><mets:file ID=\"FILE_0002_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000003.jpg\"/></mets:file><mets:file ID=\"FILE_0003_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000004.jpg\"/></mets:file><mets:file ID=\"FILE_0004_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000005.jpg\"/></mets:file><mets:file ID=\"FILE_0005_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000006.jpg\"/></mets:file><mets:file ID=\"FILE_0006_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000007.jpg\"/></mets:file><mets:file ID=\"FILE_0007_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000008.jpg\"/></mets:file><mets:file ID=\"FILE_0008_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000009.jpg\"/></mets:file><mets:file ID=\"FILE_0009_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000010.jpg\"/></mets:file><mets:file ID=\"FILE_0010_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000011.jpg\"/></mets:file><mets:file ID=\"FILE_0011_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000012.jpg\"/></mets:file><mets:file ID=\"FILE_0012_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000013.jpg\"/></mets:file><mets:file ID=\"FILE_0013_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000014.jpg\"/></mets:file><mets:file ID=\"FILE_0014_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000015.jpg\"/></mets:file><mets:file ID=\"FILE_0015_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000016.jpg\"/></mets:file><mets:file ID=\"FILE_0016_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000017.jpg\"/></mets:file><mets:file ID=\"FILE_0017_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000018.jpg\"/></mets:file><mets:file ID=\"FILE_0018_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000019.jpg\"/></mets:file><mets:file ID=\"FILE_0019_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000020.jpg\"/></mets:file><mets:file ID=\"FILE_0020_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000021.jpg\"/></mets:file><mets:file ID=\"FILE_0021_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000022.jpg\"/></mets:file><mets:file ID=\"FILE_0022_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000023.jpg\"/></mets:file><mets:file ID=\"FILE_0023_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000024.jpg\"/></mets:file><mets:file ID=\"FILE_0024_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000025.jpg\"/></mets:file><mets:file ID=\"FILE_0025_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000026.jpg\"/></mets:file><mets:file ID=\"FILE_0026_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000027.jpg\"/></mets:file><mets:file ID=\"FILE_0027_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000028.jpg\"/></mets:file><mets:file ID=\"FILE_0028_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000029.jpg\"/></mets:file><mets:file ID=\"FILE_0029_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000030.jpg\"/></mets:file><mets:file ID=\"FILE_0030_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000031.jpg\"/></mets:file><mets:file ID=\"FILE_0031_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000032.jpg\"/></mets:file><mets:file ID=\"FILE_0032_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000033.jpg\"/></mets:file><mets:file ID=\"FILE_0033_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000034.jpg\"/></mets:file><mets:file ID=\"FILE_0034_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000035.jpg\"/></mets:file><mets:file ID=\"FILE_0035_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000036.jpg\"/></mets:file><mets:file ID=\"FILE_0036_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000037.jpg\"/></mets:file><mets:file ID=\"FILE_0037_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000038.jpg\"/></mets:file><mets:file ID=\"FILE_0038_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000039.jpg\"/></mets:file><mets:file ID=\"FILE_0039_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000040.jpg\"/></mets:file><mets:file ID=\"FILE_0040_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000041.jpg\"/></mets:file><mets:file ID=\"FILE_0041_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000042.jpg\"/></mets:file><mets:file ID=\"FILE_0042_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000043.jpg\"/></mets:file><mets:file ID=\"FILE_0043_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000044.jpg\"/></mets:file><mets:file ID=\"FILE_0044_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000045.jpg\"/></mets:file><mets:file ID=\"FILE_0045_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000046.jpg\"/></mets:file><mets:file ID=\"FILE_0046_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000047.jpg\"/></mets:file><mets:file ID=\"FILE_0047_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000048.jpg\"/></mets:file><mets:file ID=\"FILE_0048_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000049.jpg\"/></mets:file><mets:file ID=\"FILE_0049_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000050.jpg\"/></mets:file><mets:file ID=\"FILE_0050_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000051.jpg\"/></mets:file><mets:file ID=\"FILE_0051_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000052.jpg\"/></mets:file><mets:file ID=\"FILE_0052_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000053.jpg\"/></mets:file><mets:file ID=\"FILE_0053_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000054.jpg\"/></mets:file><mets:file ID=\"FILE_0054_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000055.jpg\"/></mets:file><mets:file ID=\"FILE_0055_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000056.jpg\"/></mets:file><mets:file ID=\"FILE_0056_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000057.jpg\"/></mets:file><mets:file ID=\"FILE_0057_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000058.jpg\"/></mets:file><mets:file ID=\"FILE_0058_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000059.jpg\"/></mets:file><mets:file ID=\"FILE_0059_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000060.jpg\"/></mets:file><mets:file ID=\"FILE_0060_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000061.jpg\"/></mets:file><mets:file ID=\"FILE_0061_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000062.jpg\"/></mets:file><mets:file ID=\"FILE_0062_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000063.jpg\"/></mets:file><mets:file ID=\"FILE_0063_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000064.jpg\"/></mets:file><mets:file ID=\"FILE_0064_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000065.jpg\"/></mets:file><mets:file ID=\"FILE_0065_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000066.jpg\"/></mets:file><mets:file ID=\"FILE_0066_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000067.jpg\"/></mets:file><mets:file ID=\"FILE_0067_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000068.jpg\"/></mets:file><mets:file ID=\"FILE_0068_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000069.jpg\"/></mets:file><mets:file ID=\"FILE_0069_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000070.jpg\"/></mets:file><mets:file ID=\"FILE_0070_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000071.jpg\"/></mets:file><mets:file ID=\"FILE_0071_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000072.jpg\"/></mets:file><mets:file ID=\"FILE_0072_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000073.jpg\"/></mets:file><mets:file ID=\"FILE_0073_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000074.jpg\"/></mets:file><mets:file ID=\"FILE_0074_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000075.jpg\"/></mets:file><mets:file ID=\"FILE_0075_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000076.jpg\"/></mets:file><mets:file ID=\"FILE_0076_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000077.jpg\"/></mets:file><mets:file ID=\"FILE_0077_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000078.jpg\"/></mets:file><mets:file ID=\"FILE_0078_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000079.jpg\"/></mets:file><mets:file ID=\"FILE_0079_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000080.jpg\"/></mets:file><mets:file ID=\"FILE_0080_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000081.jpg\"/></mets:file><mets:file ID=\"FILE_0081_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000082.jpg\"/></mets:file><mets:file ID=\"FILE_0082_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000083.jpg\"/></mets:file><mets:file ID=\"FILE_0083_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000084.jpg\"/></mets:file><mets:file ID=\"FILE_0084_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000085.jpg\"/></mets:file><mets:file ID=\"FILE_0085_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000086.jpg\"/></mets:file><mets:file ID=\"FILE_0086_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000087.jpg\"/></mets:file><mets:file ID=\"FILE_0087_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000088.jpg\"/></mets:file><mets:file ID=\"FILE_0088_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000089.jpg\"/></mets:file><mets:file ID=\"FILE_0089_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000090.jpg\"/></mets:file><mets:file ID=\"FILE_0090_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000091.jpg\"/></mets:file><mets:file ID=\"FILE_0091_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000092.jpg\"/></mets:file><mets:file ID=\"FILE_0092_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000093.jpg\"/></mets:file><mets:file ID=\"FILE_0093_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000094.jpg\"/></mets:file><mets:file ID=\"FILE_0094_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000095.jpg\"/></mets:file><mets:file ID=\"FILE_0095_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000096.jpg\"/></mets:file><mets:file ID=\"FILE_0096_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000097.jpg\"/></mets:file><mets:file ID=\"FILE_0097_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000098.jpg\"/></mets:file><mets:file ID=\"FILE_0098_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000099.jpg\"/></mets:file><mets:file ID=\"FILE_0099_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000100.jpg\"/></mets:file><mets:file ID=\"FILE_0100_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000101.jpg\"/></mets:file><mets:file ID=\"FILE_0101_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000102.jpg\"/></mets:file><mets:file ID=\"FILE_0102_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000103.jpg\"/></mets:file><mets:file ID=\"FILE_0103_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000104.jpg\"/></mets:file><mets:file ID=\"FILE_0104_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000105.jpg\"/></mets:file><mets:file ID=\"FILE_0105_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000106.jpg\"/></mets:file><mets:file ID=\"FILE_0106_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000107.jpg\"/></mets:file><mets:file ID=\"FILE_0107_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000108.jpg\"/></mets:file><mets:file ID=\"FILE_0108_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000109.jpg\"/></mets:file><mets:file ID=\"FILE_0109_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000110.jpg\"/></mets:file><mets:file ID=\"FILE_0110_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000111.jpg\"/></mets:file><mets:file ID=\"FILE_0111_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000112.jpg\"/></mets:file><mets:file ID=\"FILE_0112_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000113.jpg\"/></mets:file><mets:file ID=\"FILE_0113_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000114.jpg\"/></mets:file><mets:file ID=\"FILE_0114_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000115.jpg\"/></mets:file><mets:file ID=\"FILE_0115_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000116.jpg\"/></mets:file><mets:file ID=\"FILE_0116_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000117.jpg\"/></mets:file><mets:file ID=\"FILE_0117_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000118.jpg\"/></mets:file><mets:file ID=\"FILE_0118_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000119.jpg\"/></mets:file><mets:file ID=\"FILE_0119_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000120.jpg\"/></mets:file><mets:file ID=\"FILE_0120_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000121.jpg\"/></mets:file><mets:file ID=\"FILE_0121_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000122.jpg\"/></mets:file><mets:file ID=\"FILE_0122_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000123.jpg\"/></mets:file><mets:file ID=\"FILE_0123_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000124.jpg\"/></mets:file><mets:file ID=\"FILE_0124_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000125.jpg\"/></mets:file><mets:file ID=\"FILE_0125_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000126.jpg\"/></mets:file><mets:file ID=\"FILE_0126_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000127.jpg\"/></mets:file><mets:file ID=\"FILE_0127_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000128.jpg\"/></mets:file><mets:file ID=\"FILE_0128_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000129.jpg\"/></mets:file><mets:file ID=\"FILE_0129_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000130.jpg\"/></mets:file><mets:file ID=\"FILE_0130_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000131.jpg\"/></mets:file><mets:file ID=\"FILE_0131_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000132.jpg\"/></mets:file><mets:file ID=\"FILE_0132_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000133.jpg\"/></mets:file><mets:file ID=\"FILE_0133_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000134.jpg\"/></mets:file><mets:file ID=\"FILE_0134_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000135.jpg\"/></mets:file><mets:file ID=\"FILE_0135_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000136.jpg\"/></mets:file><mets:file ID=\"FILE_0136_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000137.jpg\"/></mets:file><mets:file ID=\"FILE_0137_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000138.jpg\"/></mets:file><mets:file ID=\"FILE_0138_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000139.jpg\"/></mets:file><mets:file ID=\"FILE_0139_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000140.jpg\"/></mets:file><mets:file ID=\"FILE_0140_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000141.jpg\"/></mets:file><mets:file ID=\"FILE_0141_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000142.jpg\"/></mets:file><mets:file ID=\"FILE_0142_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000143.jpg\"/></mets:file><mets:file ID=\"FILE_0143_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000144.jpg\"/></mets:file><mets:file ID=\"FILE_0144_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000145.jpg\"/></mets:file><mets:file ID=\"FILE_0145_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000146.jpg\"/></mets:file><mets:file ID=\"FILE_0146_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000147.jpg\"/></mets:file><mets:file ID=\"FILE_0147_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000148.jpg\"/></mets:file><mets:file ID=\"FILE_0148_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000149.jpg\"/></mets:file><mets:file ID=\"FILE_0149_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000150.jpg\"/></mets:file><mets:file ID=\"FILE_0150_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000151.jpg\"/></mets:file><mets:file ID=\"FILE_0151_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000152.jpg\"/></mets:file><mets:file ID=\"FILE_0152_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000153.jpg\"/></mets:file><mets:file ID=\"FILE_0153_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000154.jpg\"/></mets:file><mets:file ID=\"FILE_0154_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000155.jpg\"/></mets:file><mets:file ID=\"FILE_0155_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000156.jpg\"/></mets:file><mets:file ID=\"FILE_0156_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000157.jpg\"/></mets:file><mets:file ID=\"FILE_0157_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000158.jpg\"/></mets:file><mets:file ID=\"FILE_0158_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000159.jpg\"/></mets:file><mets:file ID=\"FILE_0159_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000160.jpg\"/></mets:file><mets:file ID=\"FILE_0160_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000161.jpg\"/></mets:file><mets:file ID=\"FILE_0161_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000162.jpg\"/></mets:file><mets:file ID=\"FILE_0162_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000163.jpg\"/></mets:file><mets:file ID=\"FILE_0163_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000164.jpg\"/></mets:file><mets:file ID=\"FILE_0164_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000165.jpg\"/></mets:file><mets:file ID=\"FILE_0165_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000166.jpg\"/></mets:file><mets:file ID=\"FILE_0166_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000167.jpg\"/></mets:file><mets:file ID=\"FILE_0167_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000168.jpg\"/></mets:file><mets:file ID=\"FILE_0168_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000169.jpg\"/></mets:file><mets:file ID=\"FILE_0169_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000170.jpg\"/></mets:file><mets:file ID=\"FILE_0170_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000171.jpg\"/></mets:file><mets:file ID=\"FILE_0171_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000172.jpg\"/></mets:file><mets:file ID=\"FILE_0172_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000173.jpg\"/></mets:file><mets:file ID=\"FILE_0173_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000174.jpg\"/></mets:file><mets:file ID=\"FILE_0174_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000175.jpg\"/></mets:file><mets:file ID=\"FILE_0175_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000176.jpg\"/></mets:file><mets:file ID=\"FILE_0176_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000177.jpg\"/></mets:file><mets:file ID=\"FILE_0177_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000178.jpg\"/></mets:file><mets:file ID=\"FILE_0178_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000179.jpg\"/></mets:file><mets:file ID=\"FILE_0179_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000180.jpg\"/></mets:file><mets:file ID=\"FILE_0180_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000181.jpg\"/></mets:file><mets:file ID=\"FILE_0181_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000182.jpg\"/></mets:file><mets:file ID=\"FILE_0182_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000183.jpg\"/></mets:file><mets:file ID=\"FILE_0183_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000184.jpg\"/></mets:file><mets:file ID=\"FILE_0184_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000185.jpg\"/></mets:file><mets:file ID=\"FILE_0185_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000186.jpg\"/></mets:file><mets:file ID=\"FILE_0186_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000187.jpg\"/></mets:file><mets:file ID=\"FILE_0187_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000188.jpg\"/></mets:file><mets:file ID=\"FILE_0188_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000189.jpg\"/></mets:file><mets:file ID=\"FILE_0189_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000190.jpg\"/></mets:file><mets:file ID=\"FILE_0190_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000191.jpg\"/></mets:file><mets:file ID=\"FILE_0191_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000192.jpg\"/></mets:file><mets:file ID=\"FILE_0192_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000193.jpg\"/></mets:file><mets:file ID=\"FILE_0193_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000194.jpg\"/></mets:file><mets:file ID=\"FILE_0194_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000195.jpg\"/></mets:file><mets:file ID=\"FILE_0195_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000196.jpg\"/></mets:file><mets:file ID=\"FILE_0196_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000197.jpg\"/></mets:file><mets:file ID=\"FILE_0197_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000198.jpg\"/></mets:file><mets:file ID=\"FILE_0198_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000199.jpg\"/></mets:file><mets:file ID=\"FILE_0199_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000200.jpg\"/></mets:file><mets:file ID=\"FILE_0200_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000201.jpg\"/></mets:file><mets:file ID=\"FILE_0201_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000202.jpg\"/></mets:file><mets:file ID=\"FILE_0202_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000203.jpg\"/></mets:file><mets:file ID=\"FILE_0203_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000204.jpg\"/></mets:file><mets:file ID=\"FILE_0204_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000205.jpg\"/></mets:file><mets:file ID=\"FILE_0205_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000206.jpg\"/></mets:file><mets:file ID=\"FILE_0206_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000207.jpg\"/></mets:file><mets:file ID=\"FILE_0207_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000208.jpg\"/></mets:file><mets:file ID=\"FILE_0208_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000209.jpg\"/></mets:file><mets:file ID=\"FILE_0209_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000210.jpg\"/></mets:file><mets:file ID=\"FILE_0210_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000211.jpg\"/></mets:file><mets:file ID=\"FILE_0211_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000212.jpg\"/></mets:file><mets:file ID=\"FILE_0212_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000213.jpg\"/></mets:file><mets:file ID=\"FILE_0213_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000214.jpg\"/></mets:file><mets:file ID=\"FILE_0214_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000215.jpg\"/></mets:file><mets:file ID=\"FILE_0215_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000216.jpg\"/></mets:file><mets:file ID=\"FILE_0216_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000217.jpg\"/></mets:file><mets:file ID=\"FILE_0217_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000218.jpg\"/></mets:file><mets:file ID=\"FILE_0218_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000219.jpg\"/></mets:file><mets:file ID=\"FILE_0219_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000220.jpg\"/></mets:file><mets:file ID=\"FILE_0220_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000221.jpg\"/></mets:file><mets:file ID=\"FILE_0221_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000222.jpg\"/></mets:file><mets:file ID=\"FILE_0222_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000223.jpg\"/></mets:file><mets:file ID=\"FILE_0223_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000224.jpg\"/></mets:file><mets:file ID=\"FILE_0224_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000225.jpg\"/></mets:file><mets:file ID=\"FILE_0225_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000226.jpg\"/></mets:file><mets:file ID=\"FILE_0226_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000227.jpg\"/></mets:file><mets:file ID=\"FILE_0227_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000228.jpg\"/></mets:file><mets:file ID=\"FILE_0228_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000229.jpg\"/></mets:file><mets:file ID=\"FILE_0229_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000230.jpg\"/></mets:file><mets:file ID=\"FILE_0230_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000231.jpg\"/></mets:file><mets:file ID=\"FILE_0231_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000232.jpg\"/></mets:file><mets:file ID=\"FILE_0232_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000233.jpg\"/></mets:file><mets:file ID=\"FILE_0233_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000234.jpg\"/></mets:file><mets:file ID=\"FILE_0234_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000235.jpg\"/></mets:file><mets:file ID=\"FILE_0235_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000236.jpg\"/></mets:file><mets:file ID=\"FILE_0236_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000237.jpg\"/></mets:file><mets:file ID=\"FILE_0237_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000238.jpg\"/></mets:file><mets:file ID=\"FILE_0238_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000239.jpg\"/></mets:file><mets:file ID=\"FILE_0239_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000240.jpg\"/></mets:file><mets:file ID=\"FILE_0240_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000241.jpg\"/></mets:file><mets:file ID=\"FILE_0241_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000242.jpg\"/></mets:file><mets:file ID=\"FILE_0242_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000243.jpg\"/></mets:file><mets:file ID=\"FILE_0243_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000244.jpg\"/></mets:file><mets:file ID=\"FILE_0244_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000245.jpg\"/></mets:file><mets:file ID=\"FILE_0245_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000246.jpg\"/></mets:file><mets:file ID=\"FILE_0246_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000247.jpg\"/></mets:file><mets:file ID=\"FILE_0247_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000248.jpg\"/></mets:file><mets:file ID=\"FILE_0248_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000249.jpg\"/></mets:file><mets:file ID=\"FILE_0249_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000250.jpg\"/></mets:file><mets:file ID=\"FILE_0250_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000251.jpg\"/></mets:file><mets:file ID=\"FILE_0251_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000252.jpg\"/></mets:file><mets:file ID=\"FILE_0252_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000253.jpg\"/></mets:file><mets:file ID=\"FILE_0253_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000254.jpg\"/></mets:file><mets:file ID=\"FILE_0254_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000255.jpg\"/></mets:file><mets:file ID=\"FILE_0255_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000256.jpg\"/></mets:file><mets:file ID=\"FILE_0256_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000257.jpg\"/></mets:file><mets:file ID=\"FILE_0257_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000258.jpg\"/></mets:file><mets:file ID=\"FILE_0258_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000259.jpg\"/></mets:file><mets:file ID=\"FILE_0259_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000260.jpg\"/></mets:file><mets:file ID=\"FILE_0260_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000261.jpg\"/></mets:file><mets:file ID=\"FILE_0261_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000262.jpg\"/></mets:file><mets:file ID=\"FILE_0262_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000263.jpg\"/></mets:file><mets:file ID=\"FILE_0263_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000264.jpg\"/></mets:file><mets:file ID=\"FILE_0264_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000265.jpg\"/></mets:file><mets:file ID=\"FILE_0265_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000266.jpg\"/></mets:file><mets:file ID=\"FILE_0266_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000267.jpg\"/></mets:file><mets:file ID=\"FILE_0267_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000268.jpg\"/></mets:file><mets:file ID=\"FILE_0268_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000269.jpg\"/></mets:file><mets:file ID=\"FILE_0269_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000270.jpg\"/></mets:file><mets:file ID=\"FILE_0270_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000271.jpg\"/></mets:file><mets:file ID=\"FILE_0271_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000272.jpg\"/></mets:file><mets:file ID=\"FILE_0272_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000273.jpg\"/></mets:file><mets:file ID=\"FILE_0273_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000274.jpg\"/></mets:file><mets:file ID=\"FILE_0274_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000275.jpg\"/></mets:file><mets:file ID=\"FILE_0275_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000276.jpg\"/></mets:file><mets:file ID=\"FILE_0276_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000277.jpg\"/></mets:file><mets:file ID=\"FILE_0277_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000278.jpg\"/></mets:file><mets:file ID=\"FILE_0278_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000279.jpg\"/></mets:file><mets:file ID=\"FILE_0279_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000280.jpg\"/></mets:file><mets:file ID=\"FILE_0280_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000281.jpg\"/></mets:file><mets:file ID=\"FILE_0281_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000282.jpg\"/></mets:file><mets:file ID=\"FILE_0282_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000283.jpg\"/></mets:file><mets:file ID=\"FILE_0283_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000284.jpg\"/></mets:file><mets:file ID=\"FILE_0284_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000285.jpg\"/></mets:file><mets:file ID=\"FILE_0285_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000286.jpg\"/></mets:file><mets:file ID=\"FILE_0286_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000287.jpg\"/></mets:file><mets:file ID=\"FILE_0287_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000288.jpg\"/></mets:file><mets:file ID=\"FILE_0288_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000289.jpg\"/></mets:file><mets:file ID=\"FILE_0289_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000290.jpg\"/></mets:file><mets:file ID=\"FILE_0290_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000291.jpg\"/></mets:file><mets:file ID=\"FILE_0291_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000292.jpg\"/></mets:file><mets:file ID=\"FILE_0292_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000293.jpg\"/></mets:file><mets:file ID=\"FILE_0293_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000294.jpg\"/></mets:file><mets:file ID=\"FILE_0294_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000295.jpg\"/></mets:file><mets:file ID=\"FILE_0295_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000296.jpg\"/></mets:file><mets:file ID=\"FILE_0296_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000297.jpg\"/></mets:file><mets:file ID=\"FILE_0297_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000298.jpg\"/></mets:file><mets:file ID=\"FILE_0298_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000299.jpg\"/></mets:file><mets:file ID=\"FILE_0299_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000300.jpg\"/></mets:file><mets:file ID=\"FILE_0300_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000301.jpg\"/></mets:file><mets:file ID=\"FILE_0301_THUMBS\" MIMETYPE=\"image/jpeg\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz-srv1.sub.uni-goettingen.de/content/PPN137363540/150/0/00000302.jpg\"/></mets:file></mets:fileGrp><mets:fileGrp USE=\"LOCAL\"><mets:file ID=\"FILE_0000\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000001.tif\"/></mets:file><mets:file ID=\"FILE_0001\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000002.tif\"/></mets:file><mets:file ID=\"FILE_0002\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000003.tif\"/></mets:file><mets:file ID=\"FILE_0003\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000004.tif\"/></mets:file><mets:file ID=\"FILE_0004\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000005.tif\"/></mets:file><mets:file ID=\"FILE_0005\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000006.tif\"/></mets:file><mets:file ID=\"FILE_0006\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000007.tif\"/></mets:file><mets:file ID=\"FILE_0007\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000008.tif\"/></mets:file><mets:file ID=\"FILE_0008\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000009.tif\"/></mets:file><mets:file ID=\"FILE_0009\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000010.tif\"/></mets:file><mets:file ID=\"FILE_0010\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000011.tif\"/></mets:file><mets:file ID=\"FILE_0011\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000012.tif\"/></mets:file><mets:file ID=\"FILE_0012\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000013.tif\"/></mets:file><mets:file ID=\"FILE_0013\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000014.tif\"/></mets:file><mets:file ID=\"FILE_0014\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000015.tif\"/></mets:file><mets:file ID=\"FILE_0015\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000016.tif\"/></mets:file><mets:file ID=\"FILE_0016\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000017.tif\"/></mets:file><mets:file ID=\"FILE_0017\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000018.tif\"/></mets:file><mets:file ID=\"FILE_0018\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000019.tif\"/></mets:file><mets:file ID=\"FILE_0019\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000020.tif\"/></mets:file><mets:file ID=\"FILE_0020\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000021.tif\"/></mets:file><mets:file ID=\"FILE_0021\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000022.tif\"/></mets:file><mets:file ID=\"FILE_0022\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000023.tif\"/></mets:file><mets:file ID=\"FILE_0023\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000024.tif\"/></mets:file><mets:file ID=\"FILE_0024\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000025.tif\"/></mets:file><mets:file ID=\"FILE_0025\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000026.tif\"/></mets:file><mets:file ID=\"FILE_0026\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000027.tif\"/></mets:file><mets:file ID=\"FILE_0027\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000028.tif\"/></mets:file><mets:file ID=\"FILE_0028\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000029.tif\"/></mets:file><mets:file ID=\"FILE_0029\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000030.tif\"/></mets:file><mets:file ID=\"FILE_0030\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000031.tif\"/></mets:file><mets:file ID=\"FILE_0031\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000032.tif\"/></mets:file><mets:file ID=\"FILE_0032\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000033.tif\"/></mets:file><mets:file ID=\"FILE_0033\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000034.tif\"/></mets:file><mets:file ID=\"FILE_0034\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000035.tif\"/></mets:file><mets:file ID=\"FILE_0035\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000036.tif\"/></mets:file><mets:file ID=\"FILE_0036\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000037.tif\"/></mets:file><mets:file ID=\"FILE_0037\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000038.tif\"/></mets:file><mets:file ID=\"FILE_0038\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000039.tif\"/></mets:file><mets:file ID=\"FILE_0039\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000040.tif\"/></mets:file><mets:file ID=\"FILE_0040\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000041.tif\"/></mets:file><mets:file ID=\"FILE_0041\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000042.tif\"/></mets:file><mets:file ID=\"FILE_0042\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000043.tif\"/></mets:file><mets:file ID=\"FILE_0043\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000044.tif\"/></mets:file><mets:file ID=\"FILE_0044\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000045.tif\"/></mets:file><mets:file ID=\"FILE_0045\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000046.tif\"/></mets:file><mets:file ID=\"FILE_0046\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000047.tif\"/></mets:file><mets:file ID=\"FILE_0047\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000048.tif\"/></mets:file><mets:file ID=\"FILE_0048\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000049.tif\"/></mets:file><mets:file ID=\"FILE_0049\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000050.tif\"/></mets:file><mets:file ID=\"FILE_0050\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000051.tif\"/></mets:file><mets:file ID=\"FILE_0051\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000052.tif\"/></mets:file><mets:file ID=\"FILE_0052\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000053.tif\"/></mets:file><mets:file ID=\"FILE_0053\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000054.tif\"/></mets:file><mets:file ID=\"FILE_0054\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000055.tif\"/></mets:file><mets:file ID=\"FILE_0055\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000056.tif\"/></mets:file><mets:file ID=\"FILE_0056\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000057.tif\"/></mets:file><mets:file ID=\"FILE_0057\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000058.tif\"/></mets:file><mets:file ID=\"FILE_0058\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000059.tif\"/></mets:file><mets:file ID=\"FILE_0059\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000060.tif\"/></mets:file><mets:file ID=\"FILE_0060\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000061.tif\"/></mets:file><mets:file ID=\"FILE_0061\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000062.tif\"/></mets:file><mets:file ID=\"FILE_0062\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000063.tif\"/></mets:file><mets:file ID=\"FILE_0063\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000064.tif\"/></mets:file><mets:file ID=\"FILE_0064\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000065.tif\"/></mets:file><mets:file ID=\"FILE_0065\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000066.tif\"/></mets:file><mets:file ID=\"FILE_0066\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000067.tif\"/></mets:file><mets:file ID=\"FILE_0067\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000068.tif\"/></mets:file><mets:file ID=\"FILE_0068\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000069.tif\"/></mets:file><mets:file ID=\"FILE_0069\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000070.tif\"/></mets:file><mets:file ID=\"FILE_0070\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000071.tif\"/></mets:file><mets:file ID=\"FILE_0071\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000072.tif\"/></mets:file><mets:file ID=\"FILE_0072\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000073.tif\"/></mets:file><mets:file ID=\"FILE_0073\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000074.tif\"/></mets:file><mets:file ID=\"FILE_0074\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000075.tif\"/></mets:file><mets:file ID=\"FILE_0075\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000076.tif\"/></mets:file><mets:file ID=\"FILE_0076\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000077.tif\"/></mets:file><mets:file ID=\"FILE_0077\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000078.tif\"/></mets:file><mets:file ID=\"FILE_0078\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000079.tif\"/></mets:file><mets:file ID=\"FILE_0079\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000080.tif\"/></mets:file><mets:file ID=\"FILE_0080\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000081.tif\"/></mets:file><mets:file ID=\"FILE_0081\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000082.tif\"/></mets:file><mets:file ID=\"FILE_0082\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000083.tif\"/></mets:file><mets:file ID=\"FILE_0083\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000084.tif\"/></mets:file><mets:file ID=\"FILE_0084\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000085.tif\"/></mets:file><mets:file ID=\"FILE_0085\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000086.tif\"/></mets:file><mets:file ID=\"FILE_0086\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000087.tif\"/></mets:file><mets:file ID=\"FILE_0087\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000088.tif\"/></mets:file><mets:file ID=\"FILE_0088\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000089.tif\"/></mets:file><mets:file ID=\"FILE_0089\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000090.tif\"/></mets:file><mets:file ID=\"FILE_0090\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000091.tif\"/></mets:file><mets:file ID=\"FILE_0091\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000092.tif\"/></mets:file><mets:file ID=\"FILE_0092\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000093.tif\"/></mets:file><mets:file ID=\"FILE_0093\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000094.tif\"/></mets:file><mets:file ID=\"FILE_0094\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000095.tif\"/></mets:file><mets:file ID=\"FILE_0095\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000096.tif\"/></mets:file><mets:file ID=\"FILE_0096\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000097.tif\"/></mets:file><mets:file ID=\"FILE_0097\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000098.tif\"/></mets:file><mets:file ID=\"FILE_0098\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000099.tif\"/></mets:file><mets:file ID=\"FILE_0099\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000100.tif\"/></mets:file><mets:file ID=\"FILE_0100\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000101.tif\"/></mets:file><mets:file ID=\"FILE_0101\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000102.tif\"/></mets:file><mets:file ID=\"FILE_0102\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000103.tif\"/></mets:file><mets:file ID=\"FILE_0103\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000104.tif\"/></mets:file><mets:file ID=\"FILE_0104\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000105.tif\"/></mets:file><mets:file ID=\"FILE_0105\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000106.tif\"/></mets:file><mets:file ID=\"FILE_0106\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000107.tif\"/></mets:file><mets:file ID=\"FILE_0107\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000108.tif\"/></mets:file><mets:file ID=\"FILE_0108\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000109.tif\"/></mets:file><mets:file ID=\"FILE_0109\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000110.tif\"/></mets:file><mets:file ID=\"FILE_0110\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000111.tif\"/></mets:file><mets:file ID=\"FILE_0111\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000112.tif\"/></mets:file><mets:file ID=\"FILE_0112\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000113.tif\"/></mets:file><mets:file ID=\"FILE_0113\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000114.tif\"/></mets:file><mets:file ID=\"FILE_0114\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000115.tif\"/></mets:file><mets:file ID=\"FILE_0115\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000116.tif\"/></mets:file><mets:file ID=\"FILE_0116\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000117.tif\"/></mets:file><mets:file ID=\"FILE_0117\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000118.tif\"/></mets:file><mets:file ID=\"FILE_0118\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000119.tif\"/></mets:file><mets:file ID=\"FILE_0119\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000120.tif\"/></mets:file><mets:file ID=\"FILE_0120\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000121.tif\"/></mets:file><mets:file ID=\"FILE_0121\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000122.tif\"/></mets:file><mets:file ID=\"FILE_0122\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000123.tif\"/></mets:file><mets:file ID=\"FILE_0123\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000124.tif\"/></mets:file><mets:file ID=\"FILE_0124\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000125.tif\"/></mets:file><mets:file ID=\"FILE_0125\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000126.tif\"/></mets:file><mets:file ID=\"FILE_0126\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000127.tif\"/></mets:file><mets:file ID=\"FILE_0127\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000128.tif\"/></mets:file><mets:file ID=\"FILE_0128\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000129.tif\"/></mets:file><mets:file ID=\"FILE_0129\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000130.tif\"/></mets:file><mets:file ID=\"FILE_0130\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000131.tif\"/></mets:file><mets:file ID=\"FILE_0131\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000132.tif\"/></mets:file><mets:file ID=\"FILE_0132\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000133.tif\"/></mets:file><mets:file ID=\"FILE_0133\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000134.tif\"/></mets:file><mets:file ID=\"FILE_0134\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000135.tif\"/></mets:file><mets:file ID=\"FILE_0135\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000136.tif\"/></mets:file><mets:file ID=\"FILE_0136\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000137.tif\"/></mets:file><mets:file ID=\"FILE_0137\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000138.tif\"/></mets:file><mets:file ID=\"FILE_0138\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000139.tif\"/></mets:file><mets:file ID=\"FILE_0139\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000140.tif\"/></mets:file><mets:file ID=\"FILE_0140\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000141.tif\"/></mets:file><mets:file ID=\"FILE_0141\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000142.tif\"/></mets:file><mets:file ID=\"FILE_0142\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000143.tif\"/></mets:file><mets:file ID=\"FILE_0143\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000144.tif\"/></mets:file><mets:file ID=\"FILE_0144\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000145.tif\"/></mets:file><mets:file ID=\"FILE_0145\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000146.tif\"/></mets:file><mets:file ID=\"FILE_0146\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000147.tif\"/></mets:file><mets:file ID=\"FILE_0147\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000148.tif\"/></mets:file><mets:file ID=\"FILE_0148\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000149.tif\"/></mets:file><mets:file ID=\"FILE_0149\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000150.tif\"/></mets:file><mets:file ID=\"FILE_0150\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000151.tif\"/></mets:file><mets:file ID=\"FILE_0151\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000152.tif\"/></mets:file><mets:file ID=\"FILE_0152\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000153.tif\"/></mets:file><mets:file ID=\"FILE_0153\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000154.tif\"/></mets:file><mets:file ID=\"FILE_0154\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000155.tif\"/></mets:file><mets:file ID=\"FILE_0155\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000156.tif\"/></mets:file><mets:file ID=\"FILE_0156\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000157.tif\"/></mets:file><mets:file ID=\"FILE_0157\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000158.tif\"/></mets:file><mets:file ID=\"FILE_0158\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000159.tif\"/></mets:file><mets:file ID=\"FILE_0159\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000160.tif\"/></mets:file><mets:file ID=\"FILE_0160\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000161.tif\"/></mets:file><mets:file ID=\"FILE_0161\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000162.tif\"/></mets:file><mets:file ID=\"FILE_0162\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000163.tif\"/></mets:file><mets:file ID=\"FILE_0163\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000164.tif\"/></mets:file><mets:file ID=\"FILE_0164\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000165.tif\"/></mets:file><mets:file ID=\"FILE_0165\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000166.tif\"/></mets:file><mets:file ID=\"FILE_0166\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000167.tif\"/></mets:file><mets:file ID=\"FILE_0167\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000168.tif\"/></mets:file><mets:file ID=\"FILE_0168\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000169.tif\"/></mets:file><mets:file ID=\"FILE_0169\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000170.tif\"/></mets:file><mets:file ID=\"FILE_0170\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000171.tif\"/></mets:file><mets:file ID=\"FILE_0171\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000172.tif\"/></mets:file><mets:file ID=\"FILE_0172\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000173.tif\"/></mets:file><mets:file ID=\"FILE_0173\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000174.tif\"/></mets:file><mets:file ID=\"FILE_0174\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000175.tif\"/></mets:file><mets:file ID=\"FILE_0175\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000176.tif\"/></mets:file><mets:file ID=\"FILE_0176\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000177.tif\"/></mets:file><mets:file ID=\"FILE_0177\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000178.tif\"/></mets:file><mets:file ID=\"FILE_0178\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000179.tif\"/></mets:file><mets:file ID=\"FILE_0179\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000180.tif\"/></mets:file><mets:file ID=\"FILE_0180\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000181.tif\"/></mets:file><mets:file ID=\"FILE_0181\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000182.tif\"/></mets:file><mets:file ID=\"FILE_0182\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000183.tif\"/></mets:file><mets:file ID=\"FILE_0183\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000184.tif\"/></mets:file><mets:file ID=\"FILE_0184\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000185.tif\"/></mets:file><mets:file ID=\"FILE_0185\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000186.tif\"/></mets:file><mets:file ID=\"FILE_0186\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000187.tif\"/></mets:file><mets:file ID=\"FILE_0187\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000188.tif\"/></mets:file><mets:file ID=\"FILE_0188\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000189.tif\"/></mets:file><mets:file ID=\"FILE_0189\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000190.tif\"/></mets:file><mets:file ID=\"FILE_0190\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000191.tif\"/></mets:file><mets:file ID=\"FILE_0191\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000192.tif\"/></mets:file><mets:file ID=\"FILE_0192\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000193.tif\"/></mets:file><mets:file ID=\"FILE_0193\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000194.tif\"/></mets:file><mets:file ID=\"FILE_0194\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000195.tif\"/></mets:file><mets:file ID=\"FILE_0195\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000196.tif\"/></mets:file><mets:file ID=\"FILE_0196\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000197.tif\"/></mets:file><mets:file ID=\"FILE_0197\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000198.tif\"/></mets:file><mets:file ID=\"FILE_0198\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000199.tif\"/></mets:file><mets:file ID=\"FILE_0199\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000200.tif\"/></mets:file><mets:file ID=\"FILE_0200\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000201.tif\"/></mets:file><mets:file ID=\"FILE_0201\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000202.tif\"/></mets:file><mets:file ID=\"FILE_0202\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000203.tif\"/></mets:file><mets:file ID=\"FILE_0203\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000204.tif\"/></mets:file><mets:file ID=\"FILE_0204\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000205.tif\"/></mets:file><mets:file ID=\"FILE_0205\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000206.tif\"/></mets:file><mets:file ID=\"FILE_0206\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000207.tif\"/></mets:file><mets:file ID=\"FILE_0207\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000208.tif\"/></mets:file><mets:file ID=\"FILE_0208\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000209.tif\"/></mets:file><mets:file ID=\"FILE_0209\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000210.tif\"/></mets:file><mets:file ID=\"FILE_0210\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000211.tif\"/></mets:file><mets:file ID=\"FILE_0211\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000212.tif\"/></mets:file><mets:file ID=\"FILE_0212\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000213.tif\"/></mets:file><mets:file ID=\"FILE_0213\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000214.tif\"/></mets:file><mets:file ID=\"FILE_0214\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000215.tif\"/></mets:file><mets:file ID=\"FILE_0215\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000216.tif\"/></mets:file><mets:file ID=\"FILE_0216\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000217.tif\"/></mets:file><mets:file ID=\"FILE_0217\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000218.tif\"/></mets:file><mets:file ID=\"FILE_0218\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000219.tif\"/></mets:file><mets:file ID=\"FILE_0219\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000220.tif\"/></mets:file><mets:file ID=\"FILE_0220\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000221.tif\"/></mets:file><mets:file ID=\"FILE_0221\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000222.tif\"/></mets:file><mets:file ID=\"FILE_0222\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000223.tif\"/></mets:file><mets:file ID=\"FILE_0223\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000224.tif\"/></mets:file><mets:file ID=\"FILE_0224\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000225.tif\"/></mets:file><mets:file ID=\"FILE_0225\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000226.tif\"/></mets:file><mets:file ID=\"FILE_0226\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000227.tif\"/></mets:file><mets:file ID=\"FILE_0227\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000228.tif\"/></mets:file><mets:file ID=\"FILE_0228\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000229.tif\"/></mets:file><mets:file ID=\"FILE_0229\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000230.tif\"/></mets:file><mets:file ID=\"FILE_0230\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000231.tif\"/></mets:file><mets:file ID=\"FILE_0231\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000232.tif\"/></mets:file><mets:file ID=\"FILE_0232\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000233.tif\"/></mets:file><mets:file ID=\"FILE_0233\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000234.tif\"/></mets:file><mets:file ID=\"FILE_0234\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000235.tif\"/></mets:file><mets:file ID=\"FILE_0235\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000236.tif\"/></mets:file><mets:file ID=\"FILE_0236\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000237.tif\"/></mets:file><mets:file ID=\"FILE_0237\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000238.tif\"/></mets:file><mets:file ID=\"FILE_0238\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000239.tif\"/></mets:file><mets:file ID=\"FILE_0239\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000240.tif\"/></mets:file><mets:file ID=\"FILE_0240\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000241.tif\"/></mets:file><mets:file ID=\"FILE_0241\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000242.tif\"/></mets:file><mets:file ID=\"FILE_0242\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000243.tif\"/></mets:file><mets:file ID=\"FILE_0243\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000244.tif\"/></mets:file><mets:file ID=\"FILE_0244\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000245.tif\"/></mets:file><mets:file ID=\"FILE_0245\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000246.tif\"/></mets:file><mets:file ID=\"FILE_0246\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000247.tif\"/></mets:file><mets:file ID=\"FILE_0247\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000248.tif\"/></mets:file><mets:file ID=\"FILE_0248\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000249.tif\"/></mets:file><mets:file ID=\"FILE_0249\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000250.tif\"/></mets:file><mets:file ID=\"FILE_0250\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000251.tif\"/></mets:file><mets:file ID=\"FILE_0251\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000252.tif\"/></mets:file><mets:file ID=\"FILE_0252\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000253.tif\"/></mets:file><mets:file ID=\"FILE_0253\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000254.tif\"/></mets:file><mets:file ID=\"FILE_0254\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000255.tif\"/></mets:file><mets:file ID=\"FILE_0255\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000256.tif\"/></mets:file><mets:file ID=\"FILE_0256\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000257.tif\"/></mets:file><mets:file ID=\"FILE_0257\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000258.tif\"/></mets:file><mets:file ID=\"FILE_0258\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000259.tif\"/></mets:file><mets:file ID=\"FILE_0259\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000260.tif\"/></mets:file><mets:file ID=\"FILE_0260\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000261.tif\"/></mets:file><mets:file ID=\"FILE_0261\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000262.tif\"/></mets:file><mets:file ID=\"FILE_0262\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000263.tif\"/></mets:file><mets:file ID=\"FILE_0263\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000264.tif\"/></mets:file><mets:file ID=\"FILE_0264\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000265.tif\"/></mets:file><mets:file ID=\"FILE_0265\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000266.tif\"/></mets:file><mets:file ID=\"FILE_0266\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000267.tif\"/></mets:file><mets:file ID=\"FILE_0267\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000268.tif\"/></mets:file><mets:file ID=\"FILE_0268\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000269.tif\"/></mets:file><mets:file ID=\"FILE_0269\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000270.tif\"/></mets:file><mets:file ID=\"FILE_0270\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000271.tif\"/></mets:file><mets:file ID=\"FILE_0271\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000272.tif\"/></mets:file><mets:file ID=\"FILE_0272\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000273.tif\"/></mets:file><mets:file ID=\"FILE_0273\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000274.tif\"/></mets:file><mets:file ID=\"FILE_0274\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000275.tif\"/></mets:file><mets:file ID=\"FILE_0275\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000276.tif\"/></mets:file><mets:file ID=\"FILE_0276\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000277.tif\"/></mets:file><mets:file ID=\"FILE_0277\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000278.tif\"/></mets:file><mets:file ID=\"FILE_0278\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000279.tif\"/></mets:file><mets:file ID=\"FILE_0279\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000280.tif\"/></mets:file><mets:file ID=\"FILE_0280\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000281.tif\"/></mets:file><mets:file ID=\"FILE_0281\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000282.tif\"/></mets:file><mets:file ID=\"FILE_0282\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000283.tif\"/></mets:file><mets:file ID=\"FILE_0283\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000284.tif\"/></mets:file><mets:file ID=\"FILE_0284\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000285.tif\"/></mets:file><mets:file ID=\"FILE_0285\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000286.tif\"/></mets:file><mets:file ID=\"FILE_0286\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000287.tif\"/></mets:file><mets:file ID=\"FILE_0287\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000288.tif\"/></mets:file><mets:file ID=\"FILE_0288\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000289.tif\"/></mets:file><mets:file ID=\"FILE_0289\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000290.tif\"/></mets:file><mets:file ID=\"FILE_0290\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000291.tif\"/></mets:file><mets:file ID=\"FILE_0291\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000292.tif\"/></mets:file><mets:file ID=\"FILE_0292\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000293.tif\"/></mets:file><mets:file ID=\"FILE_0293\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000294.tif\"/></mets:file><mets:file ID=\"FILE_0294\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000295.tif\"/></mets:file><mets:file ID=\"FILE_0295\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000296.tif\"/></mets:file><mets:file ID=\"FILE_0296\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000297.tif\"/></mets:file><mets:file ID=\"FILE_0297\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000298.tif\"/></mets:file><mets:file ID=\"FILE_0298\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000299.tif\"/></mets:file><mets:file ID=\"FILE_0299\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000300.tif\"/></mets:file><mets:file ID=\"FILE_0300\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000301.tif\"/></mets:file><mets:file ID=\"FILE_0301\" MIMETYPE=\"image/tiff\"><mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file:///san/goobi/rusdml/Metadaten/31327/images/venuaccu_PPN665547242_0001_tif/00000302.tif\"/></mets:file></mets:fileGrp></mets:fileSec><mets:structMap TYPE=\"LOGICAL\"><mets:div ID=\"LOG_0002\" LABEL=\"Accurata, E Succinta Descrizione Topografica Delle Antichitá Di Roma\" TYPE=\"MultiVolumeWork\"><mets:mptr xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"http://gdz.sub.uni-goettingen.de/mets/PPN137363532 .xml\"/><mets:div ADMID=\"AMD\" DMDID=\"DMDLOG_0001\" ID=\"LOG_0003\" LABEL=\"Accurata, E Succinta Descrizione Topografica Delle Antichitá Di Roma\" TYPE=\"Volume\"><mets:div ID=\"LOG_0004\" TYPE=\"TitlePage\"/><mets:div ID=\"LOG_0005\" TYPE=\"TitlePage\"/><mets:div DMDID=\"DMDLOG_0002\" ID=\"LOG_0006\" LABEL=\"Clemens Papa XIII. Ad futuram rei memoriam.\" TYPE=\"Dedication\"/><mets:div DMDID=\"DMDLOG_0003\" ID=\"LOG_0007\" LABEL=\"Avvertimento Al Lettore.\" TYPE=\"Preface\"/><mets:div DMDID=\"DMDLOG_0004\" ID=\"LOG_0008\" LABEL=\"Indice De' Capi Della Parte Prima.\" TYPE=\"TableOfContents\"/><mets:div DMDID=\"DMDLOG_0005\" ID=\"LOG_0009\" LABEL=\"Approvazione.\" TYPE=\"OtherDocStrct\"/><mets:div DMDID=\"DMDLOG_0006\" ID=\"LOG_0010\" LABEL=\"Introduzione Alla Topografia Di Roma.\" TYPE=\"Introduction\"/><mets:div DMDID=\"DMDLOG_0007\" ID=\"LOG_0011\" LABEL=\"Carta Topografica Delle Antichità Di Roma [...]\" TYPE=\"Figure\"/><mets:div DMDID=\"DMDLOG_0008\" ID=\"LOG_0012\" LABEL=\"Capo Primo Del Monte Palatino.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0009\" ID=\"LOG_0013\" LABEL=\"[Taf.] 1 [- Taf.] 14\" TYPE=\"Figure\"/><mets:div DMDID=\"DMDLOG_0010\" ID=\"LOG_0014\" LABEL=\"Capo Primo Del Monte Palatino.\" TYPE=\"Chapter\"/></mets:div><mets:div DMDID=\"DMDLOG_0011\" ID=\"LOG_0015\" LABEL=\"Capo Secondo Del Foro Romano.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0012\" ID=\"LOG_0016\" LABEL=\"[Taf.] 15 [- Taf.] 16\" TYPE=\"Figure\"/></mets:div><mets:div DMDID=\"DMDLOG_0013\" ID=\"LOG_0017\" LABEL=\"Capo Terzo Del Campidoglio.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0014\" ID=\"LOG_0018\" LABEL=\"[Taf.] 17 [- Taf.] 22\" TYPE=\"Figure\"/></mets:div><mets:div DMDID=\"DMDLOG_0015\" ID=\"LOG_0019\" LABEL=\"Capo Quarto  Fori di Cesare, d'Augusto, di Nerva, e di Trajano.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0016\" ID=\"LOG_0020\" LABEL=\"[Taf.] 23 [-Taf.] 25\" TYPE=\"Figure\"/></mets:div><mets:div DMDID=\"DMDLOG_0017\" ID=\"LOG_0021\" LABEL=\"Capo Quinto Del Colle Quirinale.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0018\" ID=\"LOG_0022\" LABEL=\"[Taf.] 26 [- Taf.] 33\" TYPE=\"Figure\"/></mets:div><mets:div DMDID=\"DMDLOG_0019\" ID=\"LOG_0023\" LABEL=\"Capo Sesto Monte Viminale.\" TYPE=\"Chapter\"/><mets:div DMDID=\"DMDLOG_0020\" ID=\"LOG_0024\" LABEL=\"Capo Settimo. Colle Esquilino.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0021\" ID=\"LOG_0025\" LABEL=\"[Taf.] 34 [- Taf.] 55\" TYPE=\"Figure\"/></mets:div><mets:div DMDID=\"DMDLOG_0022\" ID=\"LOG_0026\" LABEL=\"Capo Ottavo Del Monte Celio.\" TYPE=\"Chapter\"><mets:div DMDID=\"DMDLOG_0023\" ID=\"LOG_0027\" LABEL=\"[Taf.] 56 [- Taf.] 59\" TYPE=\"Figure\"/></mets:div></mets:div></mets:div></mets:structMap><mets:structMap TYPE=\"PHYSICAL\"><mets:div DMDID=\"DMDPHYS_0000\" ID=\"PHYS_0000\" TYPE=\"physSequence\"><mets:div ID=\"PHYS_0001\" ORDER=\"1\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0000_MIN\"/><mets:fptr FILEID=\"FILE_0000_DEFAULT\"/><mets:fptr FILEID=\"FILE_0000_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0000_MAX\"/><mets:fptr FILEID=\"FILE_0000_THUMBS\"/><mets:fptr FILEID=\"FILE_0000\"/></mets:div><mets:div ID=\"PHYS_0002\" ORDER=\"2\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0001_MIN\"/><mets:fptr FILEID=\"FILE_0001_DEFAULT\"/><mets:fptr FILEID=\"FILE_0001_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0001_MAX\"/><mets:fptr FILEID=\"FILE_0001_THUMBS\"/><mets:fptr FILEID=\"FILE_0001\"/></mets:div><mets:div ID=\"PHYS_0003\" ORDER=\"3\" ORDERLABEL=\"[I]\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0002_MIN\"/><mets:fptr FILEID=\"FILE_0002_DEFAULT\"/><mets:fptr FILEID=\"FILE_0002_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0002_MAX\"/><mets:fptr FILEID=\"FILE_0002_THUMBS\"/><mets:fptr FILEID=\"FILE_0002\"/></mets:div><mets:div ID=\"PHYS_0004\" ORDER=\"4\" ORDERLABEL=\"[II]\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0003_MIN\"/><mets:fptr FILEID=\"FILE_0003_DEFAULT\"/><mets:fptr FILEID=\"FILE_0003_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0003_MAX\"/><mets:fptr FILEID=\"FILE_0003_THUMBS\"/><mets:fptr FILEID=\"FILE_0003\"/></mets:div><mets:div ID=\"PHYS_0005\" ORDER=\"5\" ORDERLABEL=\"[III]\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0004_MIN\"/><mets:fptr FILEID=\"FILE_0004_DEFAULT\"/><mets:fptr FILEID=\"FILE_0004_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0004_MAX\"/><mets:fptr FILEID=\"FILE_0004_THUMBS\"/><mets:fptr FILEID=\"FILE_0004\"/></mets:div><mets:div ID=\"PHYS_0006\" ORDER=\"6\" ORDERLABEL=\"[IV]\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0005_MIN\"/><mets:fptr FILEID=\"FILE_0005_DEFAULT\"/><mets:fptr FILEID=\"FILE_0005_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0005_MAX\"/><mets:fptr FILEID=\"FILE_0005_THUMBS\"/><mets:fptr FILEID=\"FILE_0005\"/></mets:div><mets:div ID=\"PHYS_0007\" ORDER=\"7\" ORDERLABEL=\"V\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0006_MIN\"/><mets:fptr FILEID=\"FILE_0006_DEFAULT\"/><mets:fptr FILEID=\"FILE_0006_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0006_MAX\"/><mets:fptr FILEID=\"FILE_0006_THUMBS\"/><mets:fptr FILEID=\"FILE_0006\"/></mets:div><mets:div ID=\"PHYS_0008\" ORDER=\"8\" ORDERLABEL=\"VI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0007_MIN\"/><mets:fptr FILEID=\"FILE_0007_DEFAULT\"/><mets:fptr FILEID=\"FILE_0007_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0007_MAX\"/><mets:fptr FILEID=\"FILE_0007_THUMBS\"/><mets:fptr FILEID=\"FILE_0007\"/></mets:div><mets:div ID=\"PHYS_0009\" ORDER=\"9\" ORDERLABEL=\"VII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0008_MIN\"/><mets:fptr FILEID=\"FILE_0008_DEFAULT\"/><mets:fptr FILEID=\"FILE_0008_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0008_MAX\"/><mets:fptr FILEID=\"FILE_0008_THUMBS\"/><mets:fptr FILEID=\"FILE_0008\"/></mets:div><mets:div ID=\"PHYS_0010\" ORDER=\"10\" ORDERLABEL=\"VIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0009_MIN\"/><mets:fptr FILEID=\"FILE_0009_DEFAULT\"/><mets:fptr FILEID=\"FILE_0009_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0009_MAX\"/><mets:fptr FILEID=\"FILE_0009_THUMBS\"/><mets:fptr FILEID=\"FILE_0009\"/></mets:div><mets:div ID=\"PHYS_0011\" ORDER=\"11\" ORDERLABEL=\"IX\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0010_MIN\"/><mets:fptr FILEID=\"FILE_0010_DEFAULT\"/><mets:fptr FILEID=\"FILE_0010_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0010_MAX\"/><mets:fptr FILEID=\"FILE_0010_THUMBS\"/><mets:fptr FILEID=\"FILE_0010\"/></mets:div><mets:div ID=\"PHYS_0012\" ORDER=\"12\" ORDERLABEL=\"X\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0011_MIN\"/><mets:fptr FILEID=\"FILE_0011_DEFAULT\"/><mets:fptr FILEID=\"FILE_0011_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0011_MAX\"/><mets:fptr FILEID=\"FILE_0011_THUMBS\"/><mets:fptr FILEID=\"FILE_0011\"/></mets:div><mets:div ID=\"PHYS_0013\" ORDER=\"13\" ORDERLABEL=\"XI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0012_MIN\"/><mets:fptr FILEID=\"FILE_0012_DEFAULT\"/><mets:fptr FILEID=\"FILE_0012_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0012_MAX\"/><mets:fptr FILEID=\"FILE_0012_THUMBS\"/><mets:fptr FILEID=\"FILE_0012\"/></mets:div><mets:div ID=\"PHYS_0014\" ORDER=\"14\" ORDERLABEL=\"XII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0013_MIN\"/><mets:fptr FILEID=\"FILE_0013_DEFAULT\"/><mets:fptr FILEID=\"FILE_0013_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0013_MAX\"/><mets:fptr FILEID=\"FILE_0013_THUMBS\"/><mets:fptr FILEID=\"FILE_0013\"/></mets:div><mets:div ID=\"PHYS_0015\" ORDER=\"15\" ORDERLABEL=\"XIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0014_MIN\"/><mets:fptr FILEID=\"FILE_0014_DEFAULT\"/><mets:fptr FILEID=\"FILE_0014_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0014_MAX\"/><mets:fptr FILEID=\"FILE_0014_THUMBS\"/><mets:fptr FILEID=\"FILE_0014\"/></mets:div><mets:div ID=\"PHYS_0016\" ORDER=\"16\" ORDERLABEL=\"XIV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0015_MIN\"/><mets:fptr FILEID=\"FILE_0015_DEFAULT\"/><mets:fptr FILEID=\"FILE_0015_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0015_MAX\"/><mets:fptr FILEID=\"FILE_0015_THUMBS\"/><mets:fptr FILEID=\"FILE_0015\"/></mets:div><mets:div ID=\"PHYS_0017\" ORDER=\"17\" ORDERLABEL=\"XV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0016_MIN\"/><mets:fptr FILEID=\"FILE_0016_DEFAULT\"/><mets:fptr FILEID=\"FILE_0016_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0016_MAX\"/><mets:fptr FILEID=\"FILE_0016_THUMBS\"/><mets:fptr FILEID=\"FILE_0016\"/></mets:div><mets:div ID=\"PHYS_0018\" ORDER=\"18\" ORDERLABEL=\"XVI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0017_MIN\"/><mets:fptr FILEID=\"FILE_0017_DEFAULT\"/><mets:fptr FILEID=\"FILE_0017_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0017_MAX\"/><mets:fptr FILEID=\"FILE_0017_THUMBS\"/><mets:fptr FILEID=\"FILE_0017\"/></mets:div><mets:div ID=\"PHYS_0019\" ORDER=\"19\" ORDERLABEL=\"XVII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0018_MIN\"/><mets:fptr FILEID=\"FILE_0018_DEFAULT\"/><mets:fptr FILEID=\"FILE_0018_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0018_MAX\"/><mets:fptr FILEID=\"FILE_0018_THUMBS\"/><mets:fptr FILEID=\"FILE_0018\"/></mets:div><mets:div ID=\"PHYS_0020\" ORDER=\"20\" ORDERLABEL=\"XVIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0019_MIN\"/><mets:fptr FILEID=\"FILE_0019_DEFAULT\"/><mets:fptr FILEID=\"FILE_0019_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0019_MAX\"/><mets:fptr FILEID=\"FILE_0019_THUMBS\"/><mets:fptr FILEID=\"FILE_0019\"/></mets:div><mets:div ID=\"PHYS_0021\" ORDER=\"21\" ORDERLABEL=\"XIX\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0020_MIN\"/><mets:fptr FILEID=\"FILE_0020_DEFAULT\"/><mets:fptr FILEID=\"FILE_0020_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0020_MAX\"/><mets:fptr FILEID=\"FILE_0020_THUMBS\"/><mets:fptr FILEID=\"FILE_0020\"/></mets:div><mets:div ID=\"PHYS_0022\" ORDER=\"22\" ORDERLABEL=\"XX\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0021_MIN\"/><mets:fptr FILEID=\"FILE_0021_DEFAULT\"/><mets:fptr FILEID=\"FILE_0021_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0021_MAX\"/><mets:fptr FILEID=\"FILE_0021_THUMBS\"/><mets:fptr FILEID=\"FILE_0021\"/></mets:div><mets:div ID=\"PHYS_0023\" ORDER=\"23\" ORDERLABEL=\"XXI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0022_MIN\"/><mets:fptr FILEID=\"FILE_0022_DEFAULT\"/><mets:fptr FILEID=\"FILE_0022_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0022_MAX\"/><mets:fptr FILEID=\"FILE_0022_THUMBS\"/><mets:fptr FILEID=\"FILE_0022\"/></mets:div><mets:div ID=\"PHYS_0024\" ORDER=\"24\" ORDERLABEL=\"XXII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0023_MIN\"/><mets:fptr FILEID=\"FILE_0023_DEFAULT\"/><mets:fptr FILEID=\"FILE_0023_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0023_MAX\"/><mets:fptr FILEID=\"FILE_0023_THUMBS\"/><mets:fptr FILEID=\"FILE_0023\"/></mets:div><mets:div ID=\"PHYS_0025\" ORDER=\"25\" ORDERLABEL=\"XXIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0024_MIN\"/><mets:fptr FILEID=\"FILE_0024_DEFAULT\"/><mets:fptr FILEID=\"FILE_0024_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0024_MAX\"/><mets:fptr FILEID=\"FILE_0024_THUMBS\"/><mets:fptr FILEID=\"FILE_0024\"/></mets:div><mets:div ID=\"PHYS_0026\" ORDER=\"26\" ORDERLABEL=\"XXIV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0025_MIN\"/><mets:fptr FILEID=\"FILE_0025_DEFAULT\"/><mets:fptr FILEID=\"FILE_0025_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0025_MAX\"/><mets:fptr FILEID=\"FILE_0025_THUMBS\"/><mets:fptr FILEID=\"FILE_0025\"/></mets:div><mets:div ID=\"PHYS_0027\" ORDER=\"27\" ORDERLABEL=\"XXV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0026_MIN\"/><mets:fptr FILEID=\"FILE_0026_DEFAULT\"/><mets:fptr FILEID=\"FILE_0026_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0026_MAX\"/><mets:fptr FILEID=\"FILE_0026_THUMBS\"/><mets:fptr FILEID=\"FILE_0026\"/></mets:div><mets:div ID=\"PHYS_0028\" ORDER=\"28\" ORDERLABEL=\"XXVI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0027_MIN\"/><mets:fptr FILEID=\"FILE_0027_DEFAULT\"/><mets:fptr FILEID=\"FILE_0027_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0027_MAX\"/><mets:fptr FILEID=\"FILE_0027_THUMBS\"/><mets:fptr FILEID=\"FILE_0027\"/></mets:div><mets:div ID=\"PHYS_0029\" ORDER=\"29\" ORDERLABEL=\"XXVII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0028_MIN\"/><mets:fptr FILEID=\"FILE_0028_DEFAULT\"/><mets:fptr FILEID=\"FILE_0028_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0028_MAX\"/><mets:fptr FILEID=\"FILE_0028_THUMBS\"/><mets:fptr FILEID=\"FILE_0028\"/></mets:div><mets:div ID=\"PHYS_0030\" ORDER=\"30\" ORDERLABEL=\"XXVIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0029_MIN\"/><mets:fptr FILEID=\"FILE_0029_DEFAULT\"/><mets:fptr FILEID=\"FILE_0029_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0029_MAX\"/><mets:fptr FILEID=\"FILE_0029_THUMBS\"/><mets:fptr FILEID=\"FILE_0029\"/></mets:div><mets:div ID=\"PHYS_0031\" ORDER=\"31\" ORDERLABEL=\"XXIX\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0030_MIN\"/><mets:fptr FILEID=\"FILE_0030_DEFAULT\"/><mets:fptr FILEID=\"FILE_0030_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0030_MAX\"/><mets:fptr FILEID=\"FILE_0030_THUMBS\"/><mets:fptr FILEID=\"FILE_0030\"/></mets:div><mets:div ID=\"PHYS_0032\" ORDER=\"32\" ORDERLABEL=\"XXX\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0031_MIN\"/><mets:fptr FILEID=\"FILE_0031_DEFAULT\"/><mets:fptr FILEID=\"FILE_0031_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0031_MAX\"/><mets:fptr FILEID=\"FILE_0031_THUMBS\"/><mets:fptr FILEID=\"FILE_0031\"/></mets:div><mets:div ID=\"PHYS_0033\" ORDER=\"33\" ORDERLABEL=\"XXXI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0032_MIN\"/><mets:fptr FILEID=\"FILE_0032_DEFAULT\"/><mets:fptr FILEID=\"FILE_0032_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0032_MAX\"/><mets:fptr FILEID=\"FILE_0032_THUMBS\"/><mets:fptr FILEID=\"FILE_0032\"/></mets:div><mets:div ID=\"PHYS_0034\" ORDER=\"34\" ORDERLABEL=\"XXXII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0033_MIN\"/><mets:fptr FILEID=\"FILE_0033_DEFAULT\"/><mets:fptr FILEID=\"FILE_0033_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0033_MAX\"/><mets:fptr FILEID=\"FILE_0033_THUMBS\"/><mets:fptr FILEID=\"FILE_0033\"/></mets:div><mets:div ID=\"PHYS_0035\" ORDER=\"35\" ORDERLABEL=\"XXXIII\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0034_MIN\"/><mets:fptr FILEID=\"FILE_0034_DEFAULT\"/><mets:fptr FILEID=\"FILE_0034_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0034_MAX\"/><mets:fptr FILEID=\"FILE_0034_THUMBS\"/><mets:fptr FILEID=\"FILE_0034\"/></mets:div><mets:div ID=\"PHYS_0036\" ORDER=\"36\" ORDERLABEL=\"XXXIV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0035_MIN\"/><mets:fptr FILEID=\"FILE_0035_DEFAULT\"/><mets:fptr FILEID=\"FILE_0035_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0035_MAX\"/><mets:fptr FILEID=\"FILE_0035_THUMBS\"/><mets:fptr FILEID=\"FILE_0035\"/></mets:div><mets:div ID=\"PHYS_0037\" ORDER=\"37\" ORDERLABEL=\"XXXV\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0036_MIN\"/><mets:fptr FILEID=\"FILE_0036_DEFAULT\"/><mets:fptr FILEID=\"FILE_0036_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0036_MAX\"/><mets:fptr FILEID=\"FILE_0036_THUMBS\"/><mets:fptr FILEID=\"FILE_0036\"/></mets:div><mets:div ID=\"PHYS_0038\" ORDER=\"38\" ORDERLABEL=\"XXXVI\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0037_MIN\"/><mets:fptr FILEID=\"FILE_0037_DEFAULT\"/><mets:fptr FILEID=\"FILE_0037_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0037_MAX\"/><mets:fptr FILEID=\"FILE_0037_THUMBS\"/><mets:fptr FILEID=\"FILE_0037\"/></mets:div><mets:div ID=\"PHYS_0039\" ORDER=\"39\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0038_MIN\"/><mets:fptr FILEID=\"FILE_0038_DEFAULT\"/><mets:fptr FILEID=\"FILE_0038_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0038_MAX\"/><mets:fptr FILEID=\"FILE_0038_THUMBS\"/><mets:fptr FILEID=\"FILE_0038\"/></mets:div><mets:div ID=\"PHYS_0040\" ORDER=\"40\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0039_MIN\"/><mets:fptr FILEID=\"FILE_0039_DEFAULT\"/><mets:fptr FILEID=\"FILE_0039_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0039_MAX\"/><mets:fptr FILEID=\"FILE_0039_THUMBS\"/><mets:fptr FILEID=\"FILE_0039\"/></mets:div><mets:div ID=\"PHYS_0041\" ORDER=\"41\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0040_MIN\"/><mets:fptr FILEID=\"FILE_0040_DEFAULT\"/><mets:fptr FILEID=\"FILE_0040_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0040_MAX\"/><mets:fptr FILEID=\"FILE_0040_THUMBS\"/><mets:fptr FILEID=\"FILE_0040\"/></mets:div><mets:div ID=\"PHYS_0042\" ORDER=\"42\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0041_MIN\"/><mets:fptr FILEID=\"FILE_0041_DEFAULT\"/><mets:fptr FILEID=\"FILE_0041_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0041_MAX\"/><mets:fptr FILEID=\"FILE_0041_THUMBS\"/><mets:fptr FILEID=\"FILE_0041\"/></mets:div><mets:div ID=\"PHYS_0043\" ORDER=\"43\" ORDERLABEL=\"[1]\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0042_MIN\"/><mets:fptr FILEID=\"FILE_0042_DEFAULT\"/><mets:fptr FILEID=\"FILE_0042_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0042_MAX\"/><mets:fptr FILEID=\"FILE_0042_THUMBS\"/><mets:fptr FILEID=\"FILE_0042\"/></mets:div><mets:div ID=\"PHYS_0044\" ORDER=\"44\" ORDERLABEL=\"2\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0043_MIN\"/><mets:fptr FILEID=\"FILE_0043_DEFAULT\"/><mets:fptr FILEID=\"FILE_0043_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0043_MAX\"/><mets:fptr FILEID=\"FILE_0043_THUMBS\"/><mets:fptr FILEID=\"FILE_0043\"/></mets:div><mets:div ID=\"PHYS_0045\" ORDER=\"45\" ORDERLABEL=\"3\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0044_MIN\"/><mets:fptr FILEID=\"FILE_0044_DEFAULT\"/><mets:fptr FILEID=\"FILE_0044_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0044_MAX\"/><mets:fptr FILEID=\"FILE_0044_THUMBS\"/><mets:fptr FILEID=\"FILE_0044\"/></mets:div><mets:div ID=\"PHYS_0046\" ORDER=\"46\" ORDERLABEL=\"4\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0045_MIN\"/><mets:fptr FILEID=\"FILE_0045_DEFAULT\"/><mets:fptr FILEID=\"FILE_0045_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0045_MAX\"/><mets:fptr FILEID=\"FILE_0045_THUMBS\"/><mets:fptr FILEID=\"FILE_0045\"/></mets:div><mets:div ID=\"PHYS_0047\" ORDER=\"47\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0046_MIN\"/><mets:fptr FILEID=\"FILE_0046_DEFAULT\"/><mets:fptr FILEID=\"FILE_0046_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0046_MAX\"/><mets:fptr FILEID=\"FILE_0046_THUMBS\"/><mets:fptr FILEID=\"FILE_0046\"/></mets:div><mets:div ID=\"PHYS_0048\" ORDER=\"48\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0047_MIN\"/><mets:fptr FILEID=\"FILE_0047_DEFAULT\"/><mets:fptr FILEID=\"FILE_0047_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0047_MAX\"/><mets:fptr FILEID=\"FILE_0047_THUMBS\"/><mets:fptr FILEID=\"FILE_0047\"/></mets:div><mets:div ID=\"PHYS_0049\" ORDER=\"49\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0048_MIN\"/><mets:fptr FILEID=\"FILE_0048_DEFAULT\"/><mets:fptr FILEID=\"FILE_0048_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0048_MAX\"/><mets:fptr FILEID=\"FILE_0048_THUMBS\"/><mets:fptr FILEID=\"FILE_0048\"/></mets:div><mets:div ID=\"PHYS_0050\" ORDER=\"50\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0049_MIN\"/><mets:fptr FILEID=\"FILE_0049_DEFAULT\"/><mets:fptr FILEID=\"FILE_0049_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0049_MAX\"/><mets:fptr FILEID=\"FILE_0049_THUMBS\"/><mets:fptr FILEID=\"FILE_0049\"/></mets:div><mets:div ID=\"PHYS_0051\" ORDER=\"51\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0050_MIN\"/><mets:fptr FILEID=\"FILE_0050_DEFAULT\"/><mets:fptr FILEID=\"FILE_0050_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0050_MAX\"/><mets:fptr FILEID=\"FILE_0050_THUMBS\"/><mets:fptr FILEID=\"FILE_0050\"/></mets:div><mets:div ID=\"PHYS_0052\" ORDER=\"52\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0051_MIN\"/><mets:fptr FILEID=\"FILE_0051_DEFAULT\"/><mets:fptr FILEID=\"FILE_0051_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0051_MAX\"/><mets:fptr FILEID=\"FILE_0051_THUMBS\"/><mets:fptr FILEID=\"FILE_0051\"/></mets:div><mets:div ID=\"PHYS_0053\" ORDER=\"53\" ORDERLABEL=\"5\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0052_MIN\"/><mets:fptr FILEID=\"FILE_0052_DEFAULT\"/><mets:fptr FILEID=\"FILE_0052_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0052_MAX\"/><mets:fptr FILEID=\"FILE_0052_THUMBS\"/><mets:fptr FILEID=\"FILE_0052\"/></mets:div><mets:div ID=\"PHYS_0054\" ORDER=\"54\" ORDERLABEL=\"6\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0053_MIN\"/><mets:fptr FILEID=\"FILE_0053_DEFAULT\"/><mets:fptr FILEID=\"FILE_0053_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0053_MAX\"/><mets:fptr FILEID=\"FILE_0053_THUMBS\"/><mets:fptr FILEID=\"FILE_0053\"/></mets:div><mets:div ID=\"PHYS_0055\" ORDER=\"55\" ORDERLABEL=\"7\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0054_MIN\"/><mets:fptr FILEID=\"FILE_0054_DEFAULT\"/><mets:fptr FILEID=\"FILE_0054_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0054_MAX\"/><mets:fptr FILEID=\"FILE_0054_THUMBS\"/><mets:fptr FILEID=\"FILE_0054\"/></mets:div><mets:div ID=\"PHYS_0056\" ORDER=\"56\" ORDERLABEL=\"8\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0055_MIN\"/><mets:fptr FILEID=\"FILE_0055_DEFAULT\"/><mets:fptr FILEID=\"FILE_0055_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0055_MAX\"/><mets:fptr FILEID=\"FILE_0055_THUMBS\"/><mets:fptr FILEID=\"FILE_0055\"/></mets:div><mets:div ID=\"PHYS_0057\" ORDER=\"57\" ORDERLABEL=\"9\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0056_MIN\"/><mets:fptr FILEID=\"FILE_0056_DEFAULT\"/><mets:fptr FILEID=\"FILE_0056_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0056_MAX\"/><mets:fptr FILEID=\"FILE_0056_THUMBS\"/><mets:fptr FILEID=\"FILE_0056\"/></mets:div><mets:div ID=\"PHYS_0058\" ORDER=\"58\" ORDERLABEL=\"10\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0057_MIN\"/><mets:fptr FILEID=\"FILE_0057_DEFAULT\"/><mets:fptr FILEID=\"FILE_0057_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0057_MAX\"/><mets:fptr FILEID=\"FILE_0057_THUMBS\"/><mets:fptr FILEID=\"FILE_0057\"/></mets:div><mets:div ID=\"PHYS_0059\" ORDER=\"59\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0058_MIN\"/><mets:fptr FILEID=\"FILE_0058_DEFAULT\"/><mets:fptr FILEID=\"FILE_0058_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0058_MAX\"/><mets:fptr FILEID=\"FILE_0058_THUMBS\"/><mets:fptr FILEID=\"FILE_0058\"/></mets:div><mets:div ID=\"PHYS_0060\" ORDER=\"60\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0059_MIN\"/><mets:fptr FILEID=\"FILE_0059_DEFAULT\"/><mets:fptr FILEID=\"FILE_0059_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0059_MAX\"/><mets:fptr FILEID=\"FILE_0059_THUMBS\"/><mets:fptr FILEID=\"FILE_0059\"/></mets:div><mets:div ID=\"PHYS_0061\" ORDER=\"61\" ORDERLABEL=\"11\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0060_MIN\"/><mets:fptr FILEID=\"FILE_0060_DEFAULT\"/><mets:fptr FILEID=\"FILE_0060_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0060_MAX\"/><mets:fptr FILEID=\"FILE_0060_THUMBS\"/><mets:fptr FILEID=\"FILE_0060\"/></mets:div><mets:div ID=\"PHYS_0062\" ORDER=\"62\" ORDERLABEL=\"12\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0061_MIN\"/><mets:fptr FILEID=\"FILE_0061_DEFAULT\"/><mets:fptr FILEID=\"FILE_0061_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0061_MAX\"/><mets:fptr FILEID=\"FILE_0061_THUMBS\"/><mets:fptr FILEID=\"FILE_0061\"/></mets:div><mets:div ID=\"PHYS_0063\" ORDER=\"63\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0062_MIN\"/><mets:fptr FILEID=\"FILE_0062_DEFAULT\"/><mets:fptr FILEID=\"FILE_0062_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0062_MAX\"/><mets:fptr FILEID=\"FILE_0062_THUMBS\"/><mets:fptr FILEID=\"FILE_0062\"/></mets:div><mets:div ID=\"PHYS_0064\" ORDER=\"64\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0063_MIN\"/><mets:fptr FILEID=\"FILE_0063_DEFAULT\"/><mets:fptr FILEID=\"FILE_0063_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0063_MAX\"/><mets:fptr FILEID=\"FILE_0063_THUMBS\"/><mets:fptr FILEID=\"FILE_0063\"/></mets:div><mets:div ID=\"PHYS_0065\" ORDER=\"65\" ORDERLABEL=\"13\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0064_MIN\"/><mets:fptr FILEID=\"FILE_0064_DEFAULT\"/><mets:fptr FILEID=\"FILE_0064_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0064_MAX\"/><mets:fptr FILEID=\"FILE_0064_THUMBS\"/><mets:fptr FILEID=\"FILE_0064\"/></mets:div><mets:div ID=\"PHYS_0066\" ORDER=\"66\" ORDERLABEL=\"14\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0065_MIN\"/><mets:fptr FILEID=\"FILE_0065_DEFAULT\"/><mets:fptr FILEID=\"FILE_0065_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0065_MAX\"/><mets:fptr FILEID=\"FILE_0065_THUMBS\"/><mets:fptr FILEID=\"FILE_0065\"/></mets:div><mets:div ID=\"PHYS_0067\" ORDER=\"67\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0066_MIN\"/><mets:fptr FILEID=\"FILE_0066_DEFAULT\"/><mets:fptr FILEID=\"FILE_0066_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0066_MAX\"/><mets:fptr FILEID=\"FILE_0066_THUMBS\"/><mets:fptr FILEID=\"FILE_0066\"/></mets:div><mets:div ID=\"PHYS_0068\" ORDER=\"68\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0067_MIN\"/><mets:fptr FILEID=\"FILE_0067_DEFAULT\"/><mets:fptr FILEID=\"FILE_0067_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0067_MAX\"/><mets:fptr FILEID=\"FILE_0067_THUMBS\"/><mets:fptr FILEID=\"FILE_0067\"/></mets:div><mets:div ID=\"PHYS_0069\" ORDER=\"69\" ORDERLABEL=\"15\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0068_MIN\"/><mets:fptr FILEID=\"FILE_0068_DEFAULT\"/><mets:fptr FILEID=\"FILE_0068_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0068_MAX\"/><mets:fptr FILEID=\"FILE_0068_THUMBS\"/><mets:fptr FILEID=\"FILE_0068\"/></mets:div><mets:div ID=\"PHYS_0070\" ORDER=\"70\" ORDERLABEL=\"16\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0069_MIN\"/><mets:fptr FILEID=\"FILE_0069_DEFAULT\"/><mets:fptr FILEID=\"FILE_0069_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0069_MAX\"/><mets:fptr FILEID=\"FILE_0069_THUMBS\"/><mets:fptr FILEID=\"FILE_0069\"/></mets:div><mets:div ID=\"PHYS_0071\" ORDER=\"71\" ORDERLABEL=\"17\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0070_MIN\"/><mets:fptr FILEID=\"FILE_0070_DEFAULT\"/><mets:fptr FILEID=\"FILE_0070_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0070_MAX\"/><mets:fptr FILEID=\"FILE_0070_THUMBS\"/><mets:fptr FILEID=\"FILE_0070\"/></mets:div><mets:div ID=\"PHYS_0072\" ORDER=\"72\" ORDERLABEL=\"18\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0071_MIN\"/><mets:fptr FILEID=\"FILE_0071_DEFAULT\"/><mets:fptr FILEID=\"FILE_0071_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0071_MAX\"/><mets:fptr FILEID=\"FILE_0071_THUMBS\"/><mets:fptr FILEID=\"FILE_0071\"/></mets:div><mets:div ID=\"PHYS_0073\" ORDER=\"73\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0072_MIN\"/><mets:fptr FILEID=\"FILE_0072_DEFAULT\"/><mets:fptr FILEID=\"FILE_0072_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0072_MAX\"/><mets:fptr FILEID=\"FILE_0072_THUMBS\"/><mets:fptr FILEID=\"FILE_0072\"/></mets:div><mets:div ID=\"PHYS_0074\" ORDER=\"74\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0073_MIN\"/><mets:fptr FILEID=\"FILE_0073_DEFAULT\"/><mets:fptr FILEID=\"FILE_0073_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0073_MAX\"/><mets:fptr FILEID=\"FILE_0073_THUMBS\"/><mets:fptr FILEID=\"FILE_0073\"/></mets:div><mets:div ID=\"PHYS_0075\" ORDER=\"75\" ORDERLABEL=\"19\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0074_MIN\"/><mets:fptr FILEID=\"FILE_0074_DEFAULT\"/><mets:fptr FILEID=\"FILE_0074_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0074_MAX\"/><mets:fptr FILEID=\"FILE_0074_THUMBS\"/><mets:fptr FILEID=\"FILE_0074\"/></mets:div><mets:div ID=\"PHYS_0076\" ORDER=\"76\" ORDERLABEL=\"20\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0075_MIN\"/><mets:fptr FILEID=\"FILE_0075_DEFAULT\"/><mets:fptr FILEID=\"FILE_0075_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0075_MAX\"/><mets:fptr FILEID=\"FILE_0075_THUMBS\"/><mets:fptr FILEID=\"FILE_0075\"/></mets:div><mets:div ID=\"PHYS_0077\" ORDER=\"77\" ORDERLABEL=\"21\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0076_MIN\"/><mets:fptr FILEID=\"FILE_0076_DEFAULT\"/><mets:fptr FILEID=\"FILE_0076_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0076_MAX\"/><mets:fptr FILEID=\"FILE_0076_THUMBS\"/><mets:fptr FILEID=\"FILE_0076\"/></mets:div><mets:div ID=\"PHYS_0078\" ORDER=\"78\" ORDERLABEL=\"22\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0077_MIN\"/><mets:fptr FILEID=\"FILE_0077_DEFAULT\"/><mets:fptr FILEID=\"FILE_0077_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0077_MAX\"/><mets:fptr FILEID=\"FILE_0077_THUMBS\"/><mets:fptr FILEID=\"FILE_0077\"/></mets:div><mets:div ID=\"PHYS_0079\" ORDER=\"79\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0078_MIN\"/><mets:fptr FILEID=\"FILE_0078_DEFAULT\"/><mets:fptr FILEID=\"FILE_0078_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0078_MAX\"/><mets:fptr FILEID=\"FILE_0078_THUMBS\"/><mets:fptr FILEID=\"FILE_0078\"/></mets:div><mets:div ID=\"PHYS_0080\" ORDER=\"80\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0079_MIN\"/><mets:fptr FILEID=\"FILE_0079_DEFAULT\"/><mets:fptr FILEID=\"FILE_0079_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0079_MAX\"/><mets:fptr FILEID=\"FILE_0079_THUMBS\"/><mets:fptr FILEID=\"FILE_0079\"/></mets:div><mets:div ID=\"PHYS_0081\" ORDER=\"81\" ORDERLABEL=\"23\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0080_MIN\"/><mets:fptr FILEID=\"FILE_0080_DEFAULT\"/><mets:fptr FILEID=\"FILE_0080_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0080_MAX\"/><mets:fptr FILEID=\"FILE_0080_THUMBS\"/><mets:fptr FILEID=\"FILE_0080\"/></mets:div><mets:div ID=\"PHYS_0082\" ORDER=\"82\" ORDERLABEL=\"24\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0081_MIN\"/><mets:fptr FILEID=\"FILE_0081_DEFAULT\"/><mets:fptr FILEID=\"FILE_0081_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0081_MAX\"/><mets:fptr FILEID=\"FILE_0081_THUMBS\"/><mets:fptr FILEID=\"FILE_0081\"/></mets:div><mets:div ID=\"PHYS_0083\" ORDER=\"83\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0082_MIN\"/><mets:fptr FILEID=\"FILE_0082_DEFAULT\"/><mets:fptr FILEID=\"FILE_0082_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0082_MAX\"/><mets:fptr FILEID=\"FILE_0082_THUMBS\"/><mets:fptr FILEID=\"FILE_0082\"/></mets:div><mets:div ID=\"PHYS_0084\" ORDER=\"84\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0083_MIN\"/><mets:fptr FILEID=\"FILE_0083_DEFAULT\"/><mets:fptr FILEID=\"FILE_0083_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0083_MAX\"/><mets:fptr FILEID=\"FILE_0083_THUMBS\"/><mets:fptr FILEID=\"FILE_0083\"/></mets:div><mets:div ID=\"PHYS_0085\" ORDER=\"85\" ORDERLABEL=\"25\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0084_MIN\"/><mets:fptr FILEID=\"FILE_0084_DEFAULT\"/><mets:fptr FILEID=\"FILE_0084_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0084_MAX\"/><mets:fptr FILEID=\"FILE_0084_THUMBS\"/><mets:fptr FILEID=\"FILE_0084\"/></mets:div><mets:div ID=\"PHYS_0086\" ORDER=\"86\" ORDERLABEL=\"26\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0085_MIN\"/><mets:fptr FILEID=\"FILE_0085_DEFAULT\"/><mets:fptr FILEID=\"FILE_0085_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0085_MAX\"/><mets:fptr FILEID=\"FILE_0085_THUMBS\"/><mets:fptr FILEID=\"FILE_0085\"/></mets:div><mets:div ID=\"PHYS_0087\" ORDER=\"87\" ORDERLABEL=\"27\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0086_MIN\"/><mets:fptr FILEID=\"FILE_0086_DEFAULT\"/><mets:fptr FILEID=\"FILE_0086_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0086_MAX\"/><mets:fptr FILEID=\"FILE_0086_THUMBS\"/><mets:fptr FILEID=\"FILE_0086\"/></mets:div><mets:div ID=\"PHYS_0088\" ORDER=\"88\" ORDERLABEL=\"28\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0087_MIN\"/><mets:fptr FILEID=\"FILE_0087_DEFAULT\"/><mets:fptr FILEID=\"FILE_0087_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0087_MAX\"/><mets:fptr FILEID=\"FILE_0087_THUMBS\"/><mets:fptr FILEID=\"FILE_0087\"/></mets:div><mets:div ID=\"PHYS_0089\" ORDER=\"89\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0088_MIN\"/><mets:fptr FILEID=\"FILE_0088_DEFAULT\"/><mets:fptr FILEID=\"FILE_0088_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0088_MAX\"/><mets:fptr FILEID=\"FILE_0088_THUMBS\"/><mets:fptr FILEID=\"FILE_0088\"/></mets:div><mets:div ID=\"PHYS_0090\" ORDER=\"90\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0089_MIN\"/><mets:fptr FILEID=\"FILE_0089_DEFAULT\"/><mets:fptr FILEID=\"FILE_0089_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0089_MAX\"/><mets:fptr FILEID=\"FILE_0089_THUMBS\"/><mets:fptr FILEID=\"FILE_0089\"/></mets:div><mets:div ID=\"PHYS_0091\" ORDER=\"91\" ORDERLABEL=\"29\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0090_MIN\"/><mets:fptr FILEID=\"FILE_0090_DEFAULT\"/><mets:fptr FILEID=\"FILE_0090_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0090_MAX\"/><mets:fptr FILEID=\"FILE_0090_THUMBS\"/><mets:fptr FILEID=\"FILE_0090\"/></mets:div><mets:div ID=\"PHYS_0092\" ORDER=\"92\" ORDERLABEL=\"30\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0091_MIN\"/><mets:fptr FILEID=\"FILE_0091_DEFAULT\"/><mets:fptr FILEID=\"FILE_0091_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0091_MAX\"/><mets:fptr FILEID=\"FILE_0091_THUMBS\"/><mets:fptr FILEID=\"FILE_0091\"/></mets:div><mets:div ID=\"PHYS_0093\" ORDER=\"93\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0092_MIN\"/><mets:fptr FILEID=\"FILE_0092_DEFAULT\"/><mets:fptr FILEID=\"FILE_0092_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0092_MAX\"/><mets:fptr FILEID=\"FILE_0092_THUMBS\"/><mets:fptr FILEID=\"FILE_0092\"/></mets:div><mets:div ID=\"PHYS_0094\" ORDER=\"94\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0093_MIN\"/><mets:fptr FILEID=\"FILE_0093_DEFAULT\"/><mets:fptr FILEID=\"FILE_0093_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0093_MAX\"/><mets:fptr FILEID=\"FILE_0093_THUMBS\"/><mets:fptr FILEID=\"FILE_0093\"/></mets:div><mets:div ID=\"PHYS_0095\" ORDER=\"95\" ORDERLABEL=\"31\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0094_MIN\"/><mets:fptr FILEID=\"FILE_0094_DEFAULT\"/><mets:fptr FILEID=\"FILE_0094_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0094_MAX\"/><mets:fptr FILEID=\"FILE_0094_THUMBS\"/><mets:fptr FILEID=\"FILE_0094\"/></mets:div><mets:div ID=\"PHYS_0096\" ORDER=\"96\" ORDERLABEL=\"32\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0095_MIN\"/><mets:fptr FILEID=\"FILE_0095_DEFAULT\"/><mets:fptr FILEID=\"FILE_0095_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0095_MAX\"/><mets:fptr FILEID=\"FILE_0095_THUMBS\"/><mets:fptr FILEID=\"FILE_0095\"/></mets:div><mets:div ID=\"PHYS_0097\" ORDER=\"97\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0096_MIN\"/><mets:fptr FILEID=\"FILE_0096_DEFAULT\"/><mets:fptr FILEID=\"FILE_0096_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0096_MAX\"/><mets:fptr FILEID=\"FILE_0096_THUMBS\"/><mets:fptr FILEID=\"FILE_0096\"/></mets:div><mets:div ID=\"PHYS_0098\" ORDER=\"98\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0097_MIN\"/><mets:fptr FILEID=\"FILE_0097_DEFAULT\"/><mets:fptr FILEID=\"FILE_0097_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0097_MAX\"/><mets:fptr FILEID=\"FILE_0097_THUMBS\"/><mets:fptr FILEID=\"FILE_0097\"/></mets:div><mets:div ID=\"PHYS_0099\" ORDER=\"99\" ORDERLABEL=\"33\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0098_MIN\"/><mets:fptr FILEID=\"FILE_0098_DEFAULT\"/><mets:fptr FILEID=\"FILE_0098_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0098_MAX\"/><mets:fptr FILEID=\"FILE_0098_THUMBS\"/><mets:fptr FILEID=\"FILE_0098\"/></mets:div><mets:div ID=\"PHYS_0100\" ORDER=\"100\" ORDERLABEL=\"34\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0099_MIN\"/><mets:fptr FILEID=\"FILE_0099_DEFAULT\"/><mets:fptr FILEID=\"FILE_0099_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0099_MAX\"/><mets:fptr FILEID=\"FILE_0099_THUMBS\"/><mets:fptr FILEID=\"FILE_0099\"/></mets:div><mets:div ID=\"PHYS_0101\" ORDER=\"101\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0100_MIN\"/><mets:fptr FILEID=\"FILE_0100_DEFAULT\"/><mets:fptr FILEID=\"FILE_0100_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0100_MAX\"/><mets:fptr FILEID=\"FILE_0100_THUMBS\"/><mets:fptr FILEID=\"FILE_0100\"/></mets:div><mets:div ID=\"PHYS_0102\" ORDER=\"102\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0101_MIN\"/><mets:fptr FILEID=\"FILE_0101_DEFAULT\"/><mets:fptr FILEID=\"FILE_0101_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0101_MAX\"/><mets:fptr FILEID=\"FILE_0101_THUMBS\"/><mets:fptr FILEID=\"FILE_0101\"/></mets:div><mets:div ID=\"PHYS_0103\" ORDER=\"103\" ORDERLABEL=\"35\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0102_MIN\"/><mets:fptr FILEID=\"FILE_0102_DEFAULT\"/><mets:fptr FILEID=\"FILE_0102_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0102_MAX\"/><mets:fptr FILEID=\"FILE_0102_THUMBS\"/><mets:fptr FILEID=\"FILE_0102\"/></mets:div><mets:div ID=\"PHYS_0104\" ORDER=\"104\" ORDERLABEL=\"36\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0103_MIN\"/><mets:fptr FILEID=\"FILE_0103_DEFAULT\"/><mets:fptr FILEID=\"FILE_0103_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0103_MAX\"/><mets:fptr FILEID=\"FILE_0103_THUMBS\"/><mets:fptr FILEID=\"FILE_0103\"/></mets:div><mets:div ID=\"PHYS_0105\" ORDER=\"105\" ORDERLABEL=\"37\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0104_MIN\"/><mets:fptr FILEID=\"FILE_0104_DEFAULT\"/><mets:fptr FILEID=\"FILE_0104_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0104_MAX\"/><mets:fptr FILEID=\"FILE_0104_THUMBS\"/><mets:fptr FILEID=\"FILE_0104\"/></mets:div><mets:div ID=\"PHYS_0106\" ORDER=\"106\" ORDERLABEL=\"38\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0105_MIN\"/><mets:fptr FILEID=\"FILE_0105_DEFAULT\"/><mets:fptr FILEID=\"FILE_0105_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0105_MAX\"/><mets:fptr FILEID=\"FILE_0105_THUMBS\"/><mets:fptr FILEID=\"FILE_0105\"/></mets:div><mets:div ID=\"PHYS_0107\" ORDER=\"107\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0106_MIN\"/><mets:fptr FILEID=\"FILE_0106_DEFAULT\"/><mets:fptr FILEID=\"FILE_0106_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0106_MAX\"/><mets:fptr FILEID=\"FILE_0106_THUMBS\"/><mets:fptr FILEID=\"FILE_0106\"/></mets:div><mets:div ID=\"PHYS_0108\" ORDER=\"108\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0107_MIN\"/><mets:fptr FILEID=\"FILE_0107_DEFAULT\"/><mets:fptr FILEID=\"FILE_0107_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0107_MAX\"/><mets:fptr FILEID=\"FILE_0107_THUMBS\"/><mets:fptr FILEID=\"FILE_0107\"/></mets:div><mets:div ID=\"PHYS_0109\" ORDER=\"109\" ORDERLABEL=\"39\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0108_MIN\"/><mets:fptr FILEID=\"FILE_0108_DEFAULT\"/><mets:fptr FILEID=\"FILE_0108_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0108_MAX\"/><mets:fptr FILEID=\"FILE_0108_THUMBS\"/><mets:fptr FILEID=\"FILE_0108\"/></mets:div><mets:div ID=\"PHYS_0110\" ORDER=\"110\" ORDERLABEL=\"40\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0109_MIN\"/><mets:fptr FILEID=\"FILE_0109_DEFAULT\"/><mets:fptr FILEID=\"FILE_0109_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0109_MAX\"/><mets:fptr FILEID=\"FILE_0109_THUMBS\"/><mets:fptr FILEID=\"FILE_0109\"/></mets:div><mets:div ID=\"PHYS_0111\" ORDER=\"111\" ORDERLABEL=\"41\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0110_MIN\"/><mets:fptr FILEID=\"FILE_0110_DEFAULT\"/><mets:fptr FILEID=\"FILE_0110_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0110_MAX\"/><mets:fptr FILEID=\"FILE_0110_THUMBS\"/><mets:fptr FILEID=\"FILE_0110\"/></mets:div><mets:div ID=\"PHYS_0112\" ORDER=\"112\" ORDERLABEL=\"42\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0111_MIN\"/><mets:fptr FILEID=\"FILE_0111_DEFAULT\"/><mets:fptr FILEID=\"FILE_0111_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0111_MAX\"/><mets:fptr FILEID=\"FILE_0111_THUMBS\"/><mets:fptr FILEID=\"FILE_0111\"/></mets:div><mets:div ID=\"PHYS_0113\" ORDER=\"113\" ORDERLABEL=\"43\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0112_MIN\"/><mets:fptr FILEID=\"FILE_0112_DEFAULT\"/><mets:fptr FILEID=\"FILE_0112_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0112_MAX\"/><mets:fptr FILEID=\"FILE_0112_THUMBS\"/><mets:fptr FILEID=\"FILE_0112\"/></mets:div><mets:div ID=\"PHYS_0114\" ORDER=\"114\" ORDERLABEL=\"44\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0113_MIN\"/><mets:fptr FILEID=\"FILE_0113_DEFAULT\"/><mets:fptr FILEID=\"FILE_0113_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0113_MAX\"/><mets:fptr FILEID=\"FILE_0113_THUMBS\"/><mets:fptr FILEID=\"FILE_0113\"/></mets:div><mets:div ID=\"PHYS_0115\" ORDER=\"115\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0114_MIN\"/><mets:fptr FILEID=\"FILE_0114_DEFAULT\"/><mets:fptr FILEID=\"FILE_0114_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0114_MAX\"/><mets:fptr FILEID=\"FILE_0114_THUMBS\"/><mets:fptr FILEID=\"FILE_0114\"/></mets:div><mets:div ID=\"PHYS_0116\" ORDER=\"116\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0115_MIN\"/><mets:fptr FILEID=\"FILE_0115_DEFAULT\"/><mets:fptr FILEID=\"FILE_0115_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0115_MAX\"/><mets:fptr FILEID=\"FILE_0115_THUMBS\"/><mets:fptr FILEID=\"FILE_0115\"/></mets:div><mets:div ID=\"PHYS_0117\" ORDER=\"117\" ORDERLABEL=\"45\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0116_MIN\"/><mets:fptr FILEID=\"FILE_0116_DEFAULT\"/><mets:fptr FILEID=\"FILE_0116_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0116_MAX\"/><mets:fptr FILEID=\"FILE_0116_THUMBS\"/><mets:fptr FILEID=\"FILE_0116\"/></mets:div><mets:div ID=\"PHYS_0118\" ORDER=\"118\" ORDERLABEL=\"46\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0117_MIN\"/><mets:fptr FILEID=\"FILE_0117_DEFAULT\"/><mets:fptr FILEID=\"FILE_0117_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0117_MAX\"/><mets:fptr FILEID=\"FILE_0117_THUMBS\"/><mets:fptr FILEID=\"FILE_0117\"/></mets:div><mets:div ID=\"PHYS_0119\" ORDER=\"119\" ORDERLABEL=\"47\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0118_MIN\"/><mets:fptr FILEID=\"FILE_0118_DEFAULT\"/><mets:fptr FILEID=\"FILE_0118_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0118_MAX\"/><mets:fptr FILEID=\"FILE_0118_THUMBS\"/><mets:fptr FILEID=\"FILE_0118\"/></mets:div><mets:div ID=\"PHYS_0120\" ORDER=\"120\" ORDERLABEL=\"48\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0119_MIN\"/><mets:fptr FILEID=\"FILE_0119_DEFAULT\"/><mets:fptr FILEID=\"FILE_0119_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0119_MAX\"/><mets:fptr FILEID=\"FILE_0119_THUMBS\"/><mets:fptr FILEID=\"FILE_0119\"/></mets:div><mets:div ID=\"PHYS_0121\" ORDER=\"121\" ORDERLABEL=\"49\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0120_MIN\"/><mets:fptr FILEID=\"FILE_0120_DEFAULT\"/><mets:fptr FILEID=\"FILE_0120_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0120_MAX\"/><mets:fptr FILEID=\"FILE_0120_THUMBS\"/><mets:fptr FILEID=\"FILE_0120\"/></mets:div><mets:div ID=\"PHYS_0122\" ORDER=\"122\" ORDERLABEL=\"50\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0121_MIN\"/><mets:fptr FILEID=\"FILE_0121_DEFAULT\"/><mets:fptr FILEID=\"FILE_0121_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0121_MAX\"/><mets:fptr FILEID=\"FILE_0121_THUMBS\"/><mets:fptr FILEID=\"FILE_0121\"/></mets:div><mets:div ID=\"PHYS_0123\" ORDER=\"123\" ORDERLABEL=\"51\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0122_MIN\"/><mets:fptr FILEID=\"FILE_0122_DEFAULT\"/><mets:fptr FILEID=\"FILE_0122_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0122_MAX\"/><mets:fptr FILEID=\"FILE_0122_THUMBS\"/><mets:fptr FILEID=\"FILE_0122\"/></mets:div><mets:div ID=\"PHYS_0124\" ORDER=\"124\" ORDERLABEL=\"52\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0123_MIN\"/><mets:fptr FILEID=\"FILE_0123_DEFAULT\"/><mets:fptr FILEID=\"FILE_0123_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0123_MAX\"/><mets:fptr FILEID=\"FILE_0123_THUMBS\"/><mets:fptr FILEID=\"FILE_0123\"/></mets:div><mets:div ID=\"PHYS_0125\" ORDER=\"125\" ORDERLABEL=\"53\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0124_MIN\"/><mets:fptr FILEID=\"FILE_0124_DEFAULT\"/><mets:fptr FILEID=\"FILE_0124_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0124_MAX\"/><mets:fptr FILEID=\"FILE_0124_THUMBS\"/><mets:fptr FILEID=\"FILE_0124\"/></mets:div><mets:div ID=\"PHYS_0126\" ORDER=\"126\" ORDERLABEL=\"54\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0125_MIN\"/><mets:fptr FILEID=\"FILE_0125_DEFAULT\"/><mets:fptr FILEID=\"FILE_0125_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0125_MAX\"/><mets:fptr FILEID=\"FILE_0125_THUMBS\"/><mets:fptr FILEID=\"FILE_0125\"/></mets:div><mets:div ID=\"PHYS_0127\" ORDER=\"127\" ORDERLABEL=\"55\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0126_MIN\"/><mets:fptr FILEID=\"FILE_0126_DEFAULT\"/><mets:fptr FILEID=\"FILE_0126_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0126_MAX\"/><mets:fptr FILEID=\"FILE_0126_THUMBS\"/><mets:fptr FILEID=\"FILE_0126\"/></mets:div><mets:div ID=\"PHYS_0128\" ORDER=\"128\" ORDERLABEL=\"56\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0127_MIN\"/><mets:fptr FILEID=\"FILE_0127_DEFAULT\"/><mets:fptr FILEID=\"FILE_0127_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0127_MAX\"/><mets:fptr FILEID=\"FILE_0127_THUMBS\"/><mets:fptr FILEID=\"FILE_0127\"/></mets:div><mets:div ID=\"PHYS_0129\" ORDER=\"129\" ORDERLABEL=\"57\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0128_MIN\"/><mets:fptr FILEID=\"FILE_0128_DEFAULT\"/><mets:fptr FILEID=\"FILE_0128_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0128_MAX\"/><mets:fptr FILEID=\"FILE_0128_THUMBS\"/><mets:fptr FILEID=\"FILE_0128\"/></mets:div><mets:div ID=\"PHYS_0130\" ORDER=\"130\" ORDERLABEL=\"58\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0129_MIN\"/><mets:fptr FILEID=\"FILE_0129_DEFAULT\"/><mets:fptr FILEID=\"FILE_0129_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0129_MAX\"/><mets:fptr FILEID=\"FILE_0129_THUMBS\"/><mets:fptr FILEID=\"FILE_0129\"/></mets:div><mets:div ID=\"PHYS_0131\" ORDER=\"131\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0130_MIN\"/><mets:fptr FILEID=\"FILE_0130_DEFAULT\"/><mets:fptr FILEID=\"FILE_0130_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0130_MAX\"/><mets:fptr FILEID=\"FILE_0130_THUMBS\"/><mets:fptr FILEID=\"FILE_0130\"/></mets:div><mets:div ID=\"PHYS_0132\" ORDER=\"132\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0131_MIN\"/><mets:fptr FILEID=\"FILE_0131_DEFAULT\"/><mets:fptr FILEID=\"FILE_0131_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0131_MAX\"/><mets:fptr FILEID=\"FILE_0131_THUMBS\"/><mets:fptr FILEID=\"FILE_0131\"/></mets:div><mets:div ID=\"PHYS_0133\" ORDER=\"133\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0132_MIN\"/><mets:fptr FILEID=\"FILE_0132_DEFAULT\"/><mets:fptr FILEID=\"FILE_0132_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0132_MAX\"/><mets:fptr FILEID=\"FILE_0132_THUMBS\"/><mets:fptr FILEID=\"FILE_0132\"/></mets:div><mets:div ID=\"PHYS_0134\" ORDER=\"134\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0133_MIN\"/><mets:fptr FILEID=\"FILE_0133_DEFAULT\"/><mets:fptr FILEID=\"FILE_0133_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0133_MAX\"/><mets:fptr FILEID=\"FILE_0133_THUMBS\"/><mets:fptr FILEID=\"FILE_0133\"/></mets:div><mets:div ID=\"PHYS_0135\" ORDER=\"135\" ORDERLABEL=\"59\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0134_MIN\"/><mets:fptr FILEID=\"FILE_0134_DEFAULT\"/><mets:fptr FILEID=\"FILE_0134_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0134_MAX\"/><mets:fptr FILEID=\"FILE_0134_THUMBS\"/><mets:fptr FILEID=\"FILE_0134\"/></mets:div><mets:div ID=\"PHYS_0136\" ORDER=\"136\" ORDERLABEL=\"60\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0135_MIN\"/><mets:fptr FILEID=\"FILE_0135_DEFAULT\"/><mets:fptr FILEID=\"FILE_0135_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0135_MAX\"/><mets:fptr FILEID=\"FILE_0135_THUMBS\"/><mets:fptr FILEID=\"FILE_0135\"/></mets:div><mets:div ID=\"PHYS_0137\" ORDER=\"137\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0136_MIN\"/><mets:fptr FILEID=\"FILE_0136_DEFAULT\"/><mets:fptr FILEID=\"FILE_0136_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0136_MAX\"/><mets:fptr FILEID=\"FILE_0136_THUMBS\"/><mets:fptr FILEID=\"FILE_0136\"/></mets:div><mets:div ID=\"PHYS_0138\" ORDER=\"138\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0137_MIN\"/><mets:fptr FILEID=\"FILE_0137_DEFAULT\"/><mets:fptr FILEID=\"FILE_0137_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0137_MAX\"/><mets:fptr FILEID=\"FILE_0137_THUMBS\"/><mets:fptr FILEID=\"FILE_0137\"/></mets:div><mets:div ID=\"PHYS_0139\" ORDER=\"139\" ORDERLABEL=\"61\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0138_MIN\"/><mets:fptr FILEID=\"FILE_0138_DEFAULT\"/><mets:fptr FILEID=\"FILE_0138_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0138_MAX\"/><mets:fptr FILEID=\"FILE_0138_THUMBS\"/><mets:fptr FILEID=\"FILE_0138\"/></mets:div><mets:div ID=\"PHYS_0140\" ORDER=\"140\" ORDERLABEL=\"62\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0139_MIN\"/><mets:fptr FILEID=\"FILE_0139_DEFAULT\"/><mets:fptr FILEID=\"FILE_0139_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0139_MAX\"/><mets:fptr FILEID=\"FILE_0139_THUMBS\"/><mets:fptr FILEID=\"FILE_0139\"/></mets:div><mets:div ID=\"PHYS_0141\" ORDER=\"141\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0140_MIN\"/><mets:fptr FILEID=\"FILE_0140_DEFAULT\"/><mets:fptr FILEID=\"FILE_0140_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0140_MAX\"/><mets:fptr FILEID=\"FILE_0140_THUMBS\"/><mets:fptr FILEID=\"FILE_0140\"/></mets:div><mets:div ID=\"PHYS_0142\" ORDER=\"142\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0141_MIN\"/><mets:fptr FILEID=\"FILE_0141_DEFAULT\"/><mets:fptr FILEID=\"FILE_0141_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0141_MAX\"/><mets:fptr FILEID=\"FILE_0141_THUMBS\"/><mets:fptr FILEID=\"FILE_0141\"/></mets:div><mets:div ID=\"PHYS_0143\" ORDER=\"143\" ORDERLABEL=\"63\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0142_MIN\"/><mets:fptr FILEID=\"FILE_0142_DEFAULT\"/><mets:fptr FILEID=\"FILE_0142_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0142_MAX\"/><mets:fptr FILEID=\"FILE_0142_THUMBS\"/><mets:fptr FILEID=\"FILE_0142\"/></mets:div><mets:div ID=\"PHYS_0144\" ORDER=\"144\" ORDERLABEL=\"64\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0143_MIN\"/><mets:fptr FILEID=\"FILE_0143_DEFAULT\"/><mets:fptr FILEID=\"FILE_0143_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0143_MAX\"/><mets:fptr FILEID=\"FILE_0143_THUMBS\"/><mets:fptr FILEID=\"FILE_0143\"/></mets:div><mets:div ID=\"PHYS_0145\" ORDER=\"145\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0144_MIN\"/><mets:fptr FILEID=\"FILE_0144_DEFAULT\"/><mets:fptr FILEID=\"FILE_0144_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0144_MAX\"/><mets:fptr FILEID=\"FILE_0144_THUMBS\"/><mets:fptr FILEID=\"FILE_0144\"/></mets:div><mets:div ID=\"PHYS_0146\" ORDER=\"146\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0145_MIN\"/><mets:fptr FILEID=\"FILE_0145_DEFAULT\"/><mets:fptr FILEID=\"FILE_0145_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0145_MAX\"/><mets:fptr FILEID=\"FILE_0145_THUMBS\"/><mets:fptr FILEID=\"FILE_0145\"/></mets:div><mets:div ID=\"PHYS_0147\" ORDER=\"147\" ORDERLABEL=\"65\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0146_MIN\"/><mets:fptr FILEID=\"FILE_0146_DEFAULT\"/><mets:fptr FILEID=\"FILE_0146_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0146_MAX\"/><mets:fptr FILEID=\"FILE_0146_THUMBS\"/><mets:fptr FILEID=\"FILE_0146\"/></mets:div><mets:div ID=\"PHYS_0148\" ORDER=\"148\" ORDERLABEL=\"66\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0147_MIN\"/><mets:fptr FILEID=\"FILE_0147_DEFAULT\"/><mets:fptr FILEID=\"FILE_0147_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0147_MAX\"/><mets:fptr FILEID=\"FILE_0147_THUMBS\"/><mets:fptr FILEID=\"FILE_0147\"/></mets:div><mets:div ID=\"PHYS_0149\" ORDER=\"149\" ORDERLABEL=\"67\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0148_MIN\"/><mets:fptr FILEID=\"FILE_0148_DEFAULT\"/><mets:fptr FILEID=\"FILE_0148_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0148_MAX\"/><mets:fptr FILEID=\"FILE_0148_THUMBS\"/><mets:fptr FILEID=\"FILE_0148\"/></mets:div><mets:div ID=\"PHYS_0150\" ORDER=\"150\" ORDERLABEL=\"68\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0149_MIN\"/><mets:fptr FILEID=\"FILE_0149_DEFAULT\"/><mets:fptr FILEID=\"FILE_0149_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0149_MAX\"/><mets:fptr FILEID=\"FILE_0149_THUMBS\"/><mets:fptr FILEID=\"FILE_0149\"/></mets:div><mets:div ID=\"PHYS_0151\" ORDER=\"151\" ORDERLABEL=\"69\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0150_MIN\"/><mets:fptr FILEID=\"FILE_0150_DEFAULT\"/><mets:fptr FILEID=\"FILE_0150_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0150_MAX\"/><mets:fptr FILEID=\"FILE_0150_THUMBS\"/><mets:fptr FILEID=\"FILE_0150\"/></mets:div><mets:div ID=\"PHYS_0152\" ORDER=\"152\" ORDERLABEL=\"70\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0151_MIN\"/><mets:fptr FILEID=\"FILE_0151_DEFAULT\"/><mets:fptr FILEID=\"FILE_0151_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0151_MAX\"/><mets:fptr FILEID=\"FILE_0151_THUMBS\"/><mets:fptr FILEID=\"FILE_0151\"/></mets:div><mets:div ID=\"PHYS_0153\" ORDER=\"153\" ORDERLABEL=\"71\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0152_MIN\"/><mets:fptr FILEID=\"FILE_0152_DEFAULT\"/><mets:fptr FILEID=\"FILE_0152_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0152_MAX\"/><mets:fptr FILEID=\"FILE_0152_THUMBS\"/><mets:fptr FILEID=\"FILE_0152\"/></mets:div><mets:div ID=\"PHYS_0154\" ORDER=\"154\" ORDERLABEL=\"72\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0153_MIN\"/><mets:fptr FILEID=\"FILE_0153_DEFAULT\"/><mets:fptr FILEID=\"FILE_0153_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0153_MAX\"/><mets:fptr FILEID=\"FILE_0153_THUMBS\"/><mets:fptr FILEID=\"FILE_0153\"/></mets:div><mets:div ID=\"PHYS_0155\" ORDER=\"155\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0154_MIN\"/><mets:fptr FILEID=\"FILE_0154_DEFAULT\"/><mets:fptr FILEID=\"FILE_0154_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0154_MAX\"/><mets:fptr FILEID=\"FILE_0154_THUMBS\"/><mets:fptr FILEID=\"FILE_0154\"/></mets:div><mets:div ID=\"PHYS_0156\" ORDER=\"156\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0155_MIN\"/><mets:fptr FILEID=\"FILE_0155_DEFAULT\"/><mets:fptr FILEID=\"FILE_0155_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0155_MAX\"/><mets:fptr FILEID=\"FILE_0155_THUMBS\"/><mets:fptr FILEID=\"FILE_0155\"/></mets:div><mets:div ID=\"PHYS_0157\" ORDER=\"157\" ORDERLABEL=\"73\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0156_MIN\"/><mets:fptr FILEID=\"FILE_0156_DEFAULT\"/><mets:fptr FILEID=\"FILE_0156_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0156_MAX\"/><mets:fptr FILEID=\"FILE_0156_THUMBS\"/><mets:fptr FILEID=\"FILE_0156\"/></mets:div><mets:div ID=\"PHYS_0158\" ORDER=\"158\" ORDERLABEL=\"74\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0157_MIN\"/><mets:fptr FILEID=\"FILE_0157_DEFAULT\"/><mets:fptr FILEID=\"FILE_0157_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0157_MAX\"/><mets:fptr FILEID=\"FILE_0157_THUMBS\"/><mets:fptr FILEID=\"FILE_0157\"/></mets:div><mets:div ID=\"PHYS_0159\" ORDER=\"159\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0158_MIN\"/><mets:fptr FILEID=\"FILE_0158_DEFAULT\"/><mets:fptr FILEID=\"FILE_0158_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0158_MAX\"/><mets:fptr FILEID=\"FILE_0158_THUMBS\"/><mets:fptr FILEID=\"FILE_0158\"/></mets:div><mets:div ID=\"PHYS_0160\" ORDER=\"160\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0159_MIN\"/><mets:fptr FILEID=\"FILE_0159_DEFAULT\"/><mets:fptr FILEID=\"FILE_0159_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0159_MAX\"/><mets:fptr FILEID=\"FILE_0159_THUMBS\"/><mets:fptr FILEID=\"FILE_0159\"/></mets:div><mets:div ID=\"PHYS_0161\" ORDER=\"161\" ORDERLABEL=\"75\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0160_MIN\"/><mets:fptr FILEID=\"FILE_0160_DEFAULT\"/><mets:fptr FILEID=\"FILE_0160_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0160_MAX\"/><mets:fptr FILEID=\"FILE_0160_THUMBS\"/><mets:fptr FILEID=\"FILE_0160\"/></mets:div><mets:div ID=\"PHYS_0162\" ORDER=\"162\" ORDERLABEL=\"76\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0161_MIN\"/><mets:fptr FILEID=\"FILE_0161_DEFAULT\"/><mets:fptr FILEID=\"FILE_0161_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0161_MAX\"/><mets:fptr FILEID=\"FILE_0161_THUMBS\"/><mets:fptr FILEID=\"FILE_0161\"/></mets:div><mets:div ID=\"PHYS_0163\" ORDER=\"163\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0162_MIN\"/><mets:fptr FILEID=\"FILE_0162_DEFAULT\"/><mets:fptr FILEID=\"FILE_0162_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0162_MAX\"/><mets:fptr FILEID=\"FILE_0162_THUMBS\"/><mets:fptr FILEID=\"FILE_0162\"/></mets:div><mets:div ID=\"PHYS_0164\" ORDER=\"164\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0163_MIN\"/><mets:fptr FILEID=\"FILE_0163_DEFAULT\"/><mets:fptr FILEID=\"FILE_0163_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0163_MAX\"/><mets:fptr FILEID=\"FILE_0163_THUMBS\"/><mets:fptr FILEID=\"FILE_0163\"/></mets:div><mets:div ID=\"PHYS_0165\" ORDER=\"165\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0164_MIN\"/><mets:fptr FILEID=\"FILE_0164_DEFAULT\"/><mets:fptr FILEID=\"FILE_0164_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0164_MAX\"/><mets:fptr FILEID=\"FILE_0164_THUMBS\"/><mets:fptr FILEID=\"FILE_0164\"/></mets:div><mets:div ID=\"PHYS_0166\" ORDER=\"166\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0165_MIN\"/><mets:fptr FILEID=\"FILE_0165_DEFAULT\"/><mets:fptr FILEID=\"FILE_0165_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0165_MAX\"/><mets:fptr FILEID=\"FILE_0165_THUMBS\"/><mets:fptr FILEID=\"FILE_0165\"/></mets:div><mets:div ID=\"PHYS_0167\" ORDER=\"167\" ORDERLABEL=\"77\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0166_MIN\"/><mets:fptr FILEID=\"FILE_0166_DEFAULT\"/><mets:fptr FILEID=\"FILE_0166_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0166_MAX\"/><mets:fptr FILEID=\"FILE_0166_THUMBS\"/><mets:fptr FILEID=\"FILE_0166\"/></mets:div><mets:div ID=\"PHYS_0168\" ORDER=\"168\" ORDERLABEL=\"78\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0167_MIN\"/><mets:fptr FILEID=\"FILE_0167_DEFAULT\"/><mets:fptr FILEID=\"FILE_0167_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0167_MAX\"/><mets:fptr FILEID=\"FILE_0167_THUMBS\"/><mets:fptr FILEID=\"FILE_0167\"/></mets:div><mets:div ID=\"PHYS_0169\" ORDER=\"169\" ORDERLABEL=\"79\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0168_MIN\"/><mets:fptr FILEID=\"FILE_0168_DEFAULT\"/><mets:fptr FILEID=\"FILE_0168_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0168_MAX\"/><mets:fptr FILEID=\"FILE_0168_THUMBS\"/><mets:fptr FILEID=\"FILE_0168\"/></mets:div><mets:div ID=\"PHYS_0170\" ORDER=\"170\" ORDERLABEL=\"80\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0169_MIN\"/><mets:fptr FILEID=\"FILE_0169_DEFAULT\"/><mets:fptr FILEID=\"FILE_0169_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0169_MAX\"/><mets:fptr FILEID=\"FILE_0169_THUMBS\"/><mets:fptr FILEID=\"FILE_0169\"/></mets:div><mets:div ID=\"PHYS_0171\" ORDER=\"171\" ORDERLABEL=\"81\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0170_MIN\"/><mets:fptr FILEID=\"FILE_0170_DEFAULT\"/><mets:fptr FILEID=\"FILE_0170_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0170_MAX\"/><mets:fptr FILEID=\"FILE_0170_THUMBS\"/><mets:fptr FILEID=\"FILE_0170\"/></mets:div><mets:div ID=\"PHYS_0172\" ORDER=\"172\" ORDERLABEL=\"82\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0171_MIN\"/><mets:fptr FILEID=\"FILE_0171_DEFAULT\"/><mets:fptr FILEID=\"FILE_0171_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0171_MAX\"/><mets:fptr FILEID=\"FILE_0171_THUMBS\"/><mets:fptr FILEID=\"FILE_0171\"/></mets:div><mets:div ID=\"PHYS_0173\" ORDER=\"173\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0172_MIN\"/><mets:fptr FILEID=\"FILE_0172_DEFAULT\"/><mets:fptr FILEID=\"FILE_0172_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0172_MAX\"/><mets:fptr FILEID=\"FILE_0172_THUMBS\"/><mets:fptr FILEID=\"FILE_0172\"/></mets:div><mets:div ID=\"PHYS_0174\" ORDER=\"174\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0173_MIN\"/><mets:fptr FILEID=\"FILE_0173_DEFAULT\"/><mets:fptr FILEID=\"FILE_0173_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0173_MAX\"/><mets:fptr FILEID=\"FILE_0173_THUMBS\"/><mets:fptr FILEID=\"FILE_0173\"/></mets:div><mets:div ID=\"PHYS_0175\" ORDER=\"175\" ORDERLABEL=\"83\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0174_MIN\"/><mets:fptr FILEID=\"FILE_0174_DEFAULT\"/><mets:fptr FILEID=\"FILE_0174_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0174_MAX\"/><mets:fptr FILEID=\"FILE_0174_THUMBS\"/><mets:fptr FILEID=\"FILE_0174\"/></mets:div><mets:div ID=\"PHYS_0176\" ORDER=\"176\" ORDERLABEL=\"84\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0175_MIN\"/><mets:fptr FILEID=\"FILE_0175_DEFAULT\"/><mets:fptr FILEID=\"FILE_0175_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0175_MAX\"/><mets:fptr FILEID=\"FILE_0175_THUMBS\"/><mets:fptr FILEID=\"FILE_0175\"/></mets:div><mets:div ID=\"PHYS_0177\" ORDER=\"177\" ORDERLABEL=\"85\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0176_MIN\"/><mets:fptr FILEID=\"FILE_0176_DEFAULT\"/><mets:fptr FILEID=\"FILE_0176_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0176_MAX\"/><mets:fptr FILEID=\"FILE_0176_THUMBS\"/><mets:fptr FILEID=\"FILE_0176\"/></mets:div><mets:div ID=\"PHYS_0178\" ORDER=\"178\" ORDERLABEL=\"86\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0177_MIN\"/><mets:fptr FILEID=\"FILE_0177_DEFAULT\"/><mets:fptr FILEID=\"FILE_0177_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0177_MAX\"/><mets:fptr FILEID=\"FILE_0177_THUMBS\"/><mets:fptr FILEID=\"FILE_0177\"/></mets:div><mets:div ID=\"PHYS_0179\" ORDER=\"179\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0178_MIN\"/><mets:fptr FILEID=\"FILE_0178_DEFAULT\"/><mets:fptr FILEID=\"FILE_0178_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0178_MAX\"/><mets:fptr FILEID=\"FILE_0178_THUMBS\"/><mets:fptr FILEID=\"FILE_0178\"/></mets:div><mets:div ID=\"PHYS_0180\" ORDER=\"180\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0179_MIN\"/><mets:fptr FILEID=\"FILE_0179_DEFAULT\"/><mets:fptr FILEID=\"FILE_0179_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0179_MAX\"/><mets:fptr FILEID=\"FILE_0179_THUMBS\"/><mets:fptr FILEID=\"FILE_0179\"/></mets:div><mets:div ID=\"PHYS_0181\" ORDER=\"181\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0180_MIN\"/><mets:fptr FILEID=\"FILE_0180_DEFAULT\"/><mets:fptr FILEID=\"FILE_0180_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0180_MAX\"/><mets:fptr FILEID=\"FILE_0180_THUMBS\"/><mets:fptr FILEID=\"FILE_0180\"/></mets:div><mets:div ID=\"PHYS_0182\" ORDER=\"182\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0181_MIN\"/><mets:fptr FILEID=\"FILE_0181_DEFAULT\"/><mets:fptr FILEID=\"FILE_0181_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0181_MAX\"/><mets:fptr FILEID=\"FILE_0181_THUMBS\"/><mets:fptr FILEID=\"FILE_0181\"/></mets:div><mets:div ID=\"PHYS_0183\" ORDER=\"183\" ORDERLABEL=\"87\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0182_MIN\"/><mets:fptr FILEID=\"FILE_0182_DEFAULT\"/><mets:fptr FILEID=\"FILE_0182_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0182_MAX\"/><mets:fptr FILEID=\"FILE_0182_THUMBS\"/><mets:fptr FILEID=\"FILE_0182\"/></mets:div><mets:div ID=\"PHYS_0184\" ORDER=\"184\" ORDERLABEL=\"88\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0183_MIN\"/><mets:fptr FILEID=\"FILE_0183_DEFAULT\"/><mets:fptr FILEID=\"FILE_0183_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0183_MAX\"/><mets:fptr FILEID=\"FILE_0183_THUMBS\"/><mets:fptr FILEID=\"FILE_0183\"/></mets:div><mets:div ID=\"PHYS_0185\" ORDER=\"185\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0184_MIN\"/><mets:fptr FILEID=\"FILE_0184_DEFAULT\"/><mets:fptr FILEID=\"FILE_0184_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0184_MAX\"/><mets:fptr FILEID=\"FILE_0184_THUMBS\"/><mets:fptr FILEID=\"FILE_0184\"/></mets:div><mets:div ID=\"PHYS_0186\" ORDER=\"186\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0185_MIN\"/><mets:fptr FILEID=\"FILE_0185_DEFAULT\"/><mets:fptr FILEID=\"FILE_0185_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0185_MAX\"/><mets:fptr FILEID=\"FILE_0185_THUMBS\"/><mets:fptr FILEID=\"FILE_0185\"/></mets:div><mets:div ID=\"PHYS_0187\" ORDER=\"187\" ORDERLABEL=\"89\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0186_MIN\"/><mets:fptr FILEID=\"FILE_0186_DEFAULT\"/><mets:fptr FILEID=\"FILE_0186_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0186_MAX\"/><mets:fptr FILEID=\"FILE_0186_THUMBS\"/><mets:fptr FILEID=\"FILE_0186\"/></mets:div><mets:div ID=\"PHYS_0188\" ORDER=\"188\" ORDERLABEL=\"90\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0187_MIN\"/><mets:fptr FILEID=\"FILE_0187_DEFAULT\"/><mets:fptr FILEID=\"FILE_0187_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0187_MAX\"/><mets:fptr FILEID=\"FILE_0187_THUMBS\"/><mets:fptr FILEID=\"FILE_0187\"/></mets:div><mets:div ID=\"PHYS_0189\" ORDER=\"189\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0188_MIN\"/><mets:fptr FILEID=\"FILE_0188_DEFAULT\"/><mets:fptr FILEID=\"FILE_0188_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0188_MAX\"/><mets:fptr FILEID=\"FILE_0188_THUMBS\"/><mets:fptr FILEID=\"FILE_0188\"/></mets:div><mets:div ID=\"PHYS_0190\" ORDER=\"190\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0189_MIN\"/><mets:fptr FILEID=\"FILE_0189_DEFAULT\"/><mets:fptr FILEID=\"FILE_0189_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0189_MAX\"/><mets:fptr FILEID=\"FILE_0189_THUMBS\"/><mets:fptr FILEID=\"FILE_0189\"/></mets:div><mets:div ID=\"PHYS_0191\" ORDER=\"191\" ORDERLABEL=\"91\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0190_MIN\"/><mets:fptr FILEID=\"FILE_0190_DEFAULT\"/><mets:fptr FILEID=\"FILE_0190_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0190_MAX\"/><mets:fptr FILEID=\"FILE_0190_THUMBS\"/><mets:fptr FILEID=\"FILE_0190\"/></mets:div><mets:div ID=\"PHYS_0192\" ORDER=\"192\" ORDERLABEL=\"92\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0191_MIN\"/><mets:fptr FILEID=\"FILE_0191_DEFAULT\"/><mets:fptr FILEID=\"FILE_0191_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0191_MAX\"/><mets:fptr FILEID=\"FILE_0191_THUMBS\"/><mets:fptr FILEID=\"FILE_0191\"/></mets:div><mets:div ID=\"PHYS_0193\" ORDER=\"193\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0192_MIN\"/><mets:fptr FILEID=\"FILE_0192_DEFAULT\"/><mets:fptr FILEID=\"FILE_0192_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0192_MAX\"/><mets:fptr FILEID=\"FILE_0192_THUMBS\"/><mets:fptr FILEID=\"FILE_0192\"/></mets:div><mets:div ID=\"PHYS_0194\" ORDER=\"194\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0193_MIN\"/><mets:fptr FILEID=\"FILE_0193_DEFAULT\"/><mets:fptr FILEID=\"FILE_0193_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0193_MAX\"/><mets:fptr FILEID=\"FILE_0193_THUMBS\"/><mets:fptr FILEID=\"FILE_0193\"/></mets:div><mets:div ID=\"PHYS_0195\" ORDER=\"195\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0194_MIN\"/><mets:fptr FILEID=\"FILE_0194_DEFAULT\"/><mets:fptr FILEID=\"FILE_0194_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0194_MAX\"/><mets:fptr FILEID=\"FILE_0194_THUMBS\"/><mets:fptr FILEID=\"FILE_0194\"/></mets:div><mets:div ID=\"PHYS_0196\" ORDER=\"196\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0195_MIN\"/><mets:fptr FILEID=\"FILE_0195_DEFAULT\"/><mets:fptr FILEID=\"FILE_0195_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0195_MAX\"/><mets:fptr FILEID=\"FILE_0195_THUMBS\"/><mets:fptr FILEID=\"FILE_0195\"/></mets:div><mets:div ID=\"PHYS_0197\" ORDER=\"197\" ORDERLABEL=\"93\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0196_MIN\"/><mets:fptr FILEID=\"FILE_0196_DEFAULT\"/><mets:fptr FILEID=\"FILE_0196_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0196_MAX\"/><mets:fptr FILEID=\"FILE_0196_THUMBS\"/><mets:fptr FILEID=\"FILE_0196\"/></mets:div><mets:div ID=\"PHYS_0198\" ORDER=\"198\" ORDERLABEL=\"94\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0197_MIN\"/><mets:fptr FILEID=\"FILE_0197_DEFAULT\"/><mets:fptr FILEID=\"FILE_0197_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0197_MAX\"/><mets:fptr FILEID=\"FILE_0197_THUMBS\"/><mets:fptr FILEID=\"FILE_0197\"/></mets:div><mets:div ID=\"PHYS_0199\" ORDER=\"199\" ORDERLABEL=\"95\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0198_MIN\"/><mets:fptr FILEID=\"FILE_0198_DEFAULT\"/><mets:fptr FILEID=\"FILE_0198_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0198_MAX\"/><mets:fptr FILEID=\"FILE_0198_THUMBS\"/><mets:fptr FILEID=\"FILE_0198\"/></mets:div><mets:div ID=\"PHYS_0200\" ORDER=\"200\" ORDERLABEL=\"96\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0199_MIN\"/><mets:fptr FILEID=\"FILE_0199_DEFAULT\"/><mets:fptr FILEID=\"FILE_0199_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0199_MAX\"/><mets:fptr FILEID=\"FILE_0199_THUMBS\"/><mets:fptr FILEID=\"FILE_0199\"/></mets:div><mets:div ID=\"PHYS_0201\" ORDER=\"201\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0200_MIN\"/><mets:fptr FILEID=\"FILE_0200_DEFAULT\"/><mets:fptr FILEID=\"FILE_0200_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0200_MAX\"/><mets:fptr FILEID=\"FILE_0200_THUMBS\"/><mets:fptr FILEID=\"FILE_0200\"/></mets:div><mets:div ID=\"PHYS_0202\" ORDER=\"202\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0201_MIN\"/><mets:fptr FILEID=\"FILE_0201_DEFAULT\"/><mets:fptr FILEID=\"FILE_0201_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0201_MAX\"/><mets:fptr FILEID=\"FILE_0201_THUMBS\"/><mets:fptr FILEID=\"FILE_0201\"/></mets:div><mets:div ID=\"PHYS_0203\" ORDER=\"203\" ORDERLABEL=\"97\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0202_MIN\"/><mets:fptr FILEID=\"FILE_0202_DEFAULT\"/><mets:fptr FILEID=\"FILE_0202_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0202_MAX\"/><mets:fptr FILEID=\"FILE_0202_THUMBS\"/><mets:fptr FILEID=\"FILE_0202\"/></mets:div><mets:div ID=\"PHYS_0204\" ORDER=\"204\" ORDERLABEL=\"98\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0203_MIN\"/><mets:fptr FILEID=\"FILE_0203_DEFAULT\"/><mets:fptr FILEID=\"FILE_0203_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0203_MAX\"/><mets:fptr FILEID=\"FILE_0203_THUMBS\"/><mets:fptr FILEID=\"FILE_0203\"/></mets:div><mets:div ID=\"PHYS_0205\" ORDER=\"205\" ORDERLABEL=\"99\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0204_MIN\"/><mets:fptr FILEID=\"FILE_0204_DEFAULT\"/><mets:fptr FILEID=\"FILE_0204_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0204_MAX\"/><mets:fptr FILEID=\"FILE_0204_THUMBS\"/><mets:fptr FILEID=\"FILE_0204\"/></mets:div><mets:div ID=\"PHYS_0206\" ORDER=\"206\" ORDERLABEL=\"100\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0205_MIN\"/><mets:fptr FILEID=\"FILE_0205_DEFAULT\"/><mets:fptr FILEID=\"FILE_0205_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0205_MAX\"/><mets:fptr FILEID=\"FILE_0205_THUMBS\"/><mets:fptr FILEID=\"FILE_0205\"/></mets:div><mets:div ID=\"PHYS_0207\" ORDER=\"207\" ORDERLABEL=\"101\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0206_MIN\"/><mets:fptr FILEID=\"FILE_0206_DEFAULT\"/><mets:fptr FILEID=\"FILE_0206_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0206_MAX\"/><mets:fptr FILEID=\"FILE_0206_THUMBS\"/><mets:fptr FILEID=\"FILE_0206\"/></mets:div><mets:div ID=\"PHYS_0208\" ORDER=\"208\" ORDERLABEL=\"102\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0207_MIN\"/><mets:fptr FILEID=\"FILE_0207_DEFAULT\"/><mets:fptr FILEID=\"FILE_0207_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0207_MAX\"/><mets:fptr FILEID=\"FILE_0207_THUMBS\"/><mets:fptr FILEID=\"FILE_0207\"/></mets:div><mets:div ID=\"PHYS_0209\" ORDER=\"209\" ORDERLABEL=\"103\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0208_MIN\"/><mets:fptr FILEID=\"FILE_0208_DEFAULT\"/><mets:fptr FILEID=\"FILE_0208_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0208_MAX\"/><mets:fptr FILEID=\"FILE_0208_THUMBS\"/><mets:fptr FILEID=\"FILE_0208\"/></mets:div><mets:div ID=\"PHYS_0210\" ORDER=\"210\" ORDERLABEL=\"104\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0209_MIN\"/><mets:fptr FILEID=\"FILE_0209_DEFAULT\"/><mets:fptr FILEID=\"FILE_0209_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0209_MAX\"/><mets:fptr FILEID=\"FILE_0209_THUMBS\"/><mets:fptr FILEID=\"FILE_0209\"/></mets:div><mets:div ID=\"PHYS_0211\" ORDER=\"211\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0210_MIN\"/><mets:fptr FILEID=\"FILE_0210_DEFAULT\"/><mets:fptr FILEID=\"FILE_0210_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0210_MAX\"/><mets:fptr FILEID=\"FILE_0210_THUMBS\"/><mets:fptr FILEID=\"FILE_0210\"/></mets:div><mets:div ID=\"PHYS_0212\" ORDER=\"212\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0211_MIN\"/><mets:fptr FILEID=\"FILE_0211_DEFAULT\"/><mets:fptr FILEID=\"FILE_0211_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0211_MAX\"/><mets:fptr FILEID=\"FILE_0211_THUMBS\"/><mets:fptr FILEID=\"FILE_0211\"/></mets:div><mets:div ID=\"PHYS_0213\" ORDER=\"213\" ORDERLABEL=\"105\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0212_MIN\"/><mets:fptr FILEID=\"FILE_0212_DEFAULT\"/><mets:fptr FILEID=\"FILE_0212_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0212_MAX\"/><mets:fptr FILEID=\"FILE_0212_THUMBS\"/><mets:fptr FILEID=\"FILE_0212\"/></mets:div><mets:div ID=\"PHYS_0214\" ORDER=\"214\" ORDERLABEL=\"106\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0213_MIN\"/><mets:fptr FILEID=\"FILE_0213_DEFAULT\"/><mets:fptr FILEID=\"FILE_0213_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0213_MAX\"/><mets:fptr FILEID=\"FILE_0213_THUMBS\"/><mets:fptr FILEID=\"FILE_0213\"/></mets:div><mets:div ID=\"PHYS_0215\" ORDER=\"215\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0214_MIN\"/><mets:fptr FILEID=\"FILE_0214_DEFAULT\"/><mets:fptr FILEID=\"FILE_0214_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0214_MAX\"/><mets:fptr FILEID=\"FILE_0214_THUMBS\"/><mets:fptr FILEID=\"FILE_0214\"/></mets:div><mets:div ID=\"PHYS_0216\" ORDER=\"216\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0215_MIN\"/><mets:fptr FILEID=\"FILE_0215_DEFAULT\"/><mets:fptr FILEID=\"FILE_0215_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0215_MAX\"/><mets:fptr FILEID=\"FILE_0215_THUMBS\"/><mets:fptr FILEID=\"FILE_0215\"/></mets:div><mets:div ID=\"PHYS_0217\" ORDER=\"217\" ORDERLABEL=\"107\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0216_MIN\"/><mets:fptr FILEID=\"FILE_0216_DEFAULT\"/><mets:fptr FILEID=\"FILE_0216_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0216_MAX\"/><mets:fptr FILEID=\"FILE_0216_THUMBS\"/><mets:fptr FILEID=\"FILE_0216\"/></mets:div><mets:div ID=\"PHYS_0218\" ORDER=\"218\" ORDERLABEL=\"108\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0217_MIN\"/><mets:fptr FILEID=\"FILE_0217_DEFAULT\"/><mets:fptr FILEID=\"FILE_0217_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0217_MAX\"/><mets:fptr FILEID=\"FILE_0217_THUMBS\"/><mets:fptr FILEID=\"FILE_0217\"/></mets:div><mets:div ID=\"PHYS_0219\" ORDER=\"219\" ORDERLABEL=\"109\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0218_MIN\"/><mets:fptr FILEID=\"FILE_0218_DEFAULT\"/><mets:fptr FILEID=\"FILE_0218_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0218_MAX\"/><mets:fptr FILEID=\"FILE_0218_THUMBS\"/><mets:fptr FILEID=\"FILE_0218\"/></mets:div><mets:div ID=\"PHYS_0220\" ORDER=\"220\" ORDERLABEL=\"110\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0219_MIN\"/><mets:fptr FILEID=\"FILE_0219_DEFAULT\"/><mets:fptr FILEID=\"FILE_0219_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0219_MAX\"/><mets:fptr FILEID=\"FILE_0219_THUMBS\"/><mets:fptr FILEID=\"FILE_0219\"/></mets:div><mets:div ID=\"PHYS_0221\" ORDER=\"221\" ORDERLABEL=\"111\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0220_MIN\"/><mets:fptr FILEID=\"FILE_0220_DEFAULT\"/><mets:fptr FILEID=\"FILE_0220_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0220_MAX\"/><mets:fptr FILEID=\"FILE_0220_THUMBS\"/><mets:fptr FILEID=\"FILE_0220\"/></mets:div><mets:div ID=\"PHYS_0222\" ORDER=\"222\" ORDERLABEL=\"112\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0221_MIN\"/><mets:fptr FILEID=\"FILE_0221_DEFAULT\"/><mets:fptr FILEID=\"FILE_0221_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0221_MAX\"/><mets:fptr FILEID=\"FILE_0221_THUMBS\"/><mets:fptr FILEID=\"FILE_0221\"/></mets:div><mets:div ID=\"PHYS_0223\" ORDER=\"223\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0222_MIN\"/><mets:fptr FILEID=\"FILE_0222_DEFAULT\"/><mets:fptr FILEID=\"FILE_0222_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0222_MAX\"/><mets:fptr FILEID=\"FILE_0222_THUMBS\"/><mets:fptr FILEID=\"FILE_0222\"/></mets:div><mets:div ID=\"PHYS_0224\" ORDER=\"224\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0223_MIN\"/><mets:fptr FILEID=\"FILE_0223_DEFAULT\"/><mets:fptr FILEID=\"FILE_0223_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0223_MAX\"/><mets:fptr FILEID=\"FILE_0223_THUMBS\"/><mets:fptr FILEID=\"FILE_0223\"/></mets:div><mets:div ID=\"PHYS_0225\" ORDER=\"225\" ORDERLABEL=\"113\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0224_MIN\"/><mets:fptr FILEID=\"FILE_0224_DEFAULT\"/><mets:fptr FILEID=\"FILE_0224_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0224_MAX\"/><mets:fptr FILEID=\"FILE_0224_THUMBS\"/><mets:fptr FILEID=\"FILE_0224\"/></mets:div><mets:div ID=\"PHYS_0226\" ORDER=\"226\" ORDERLABEL=\"114\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0225_MIN\"/><mets:fptr FILEID=\"FILE_0225_DEFAULT\"/><mets:fptr FILEID=\"FILE_0225_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0225_MAX\"/><mets:fptr FILEID=\"FILE_0225_THUMBS\"/><mets:fptr FILEID=\"FILE_0225\"/></mets:div><mets:div ID=\"PHYS_0227\" ORDER=\"227\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0226_MIN\"/><mets:fptr FILEID=\"FILE_0226_DEFAULT\"/><mets:fptr FILEID=\"FILE_0226_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0226_MAX\"/><mets:fptr FILEID=\"FILE_0226_THUMBS\"/><mets:fptr FILEID=\"FILE_0226\"/></mets:div><mets:div ID=\"PHYS_0228\" ORDER=\"228\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0227_MIN\"/><mets:fptr FILEID=\"FILE_0227_DEFAULT\"/><mets:fptr FILEID=\"FILE_0227_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0227_MAX\"/><mets:fptr FILEID=\"FILE_0227_THUMBS\"/><mets:fptr FILEID=\"FILE_0227\"/></mets:div><mets:div ID=\"PHYS_0229\" ORDER=\"229\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0228_MIN\"/><mets:fptr FILEID=\"FILE_0228_DEFAULT\"/><mets:fptr FILEID=\"FILE_0228_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0228_MAX\"/><mets:fptr FILEID=\"FILE_0228_THUMBS\"/><mets:fptr FILEID=\"FILE_0228\"/></mets:div><mets:div ID=\"PHYS_0230\" ORDER=\"230\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0229_MIN\"/><mets:fptr FILEID=\"FILE_0229_DEFAULT\"/><mets:fptr FILEID=\"FILE_0229_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0229_MAX\"/><mets:fptr FILEID=\"FILE_0229_THUMBS\"/><mets:fptr FILEID=\"FILE_0229\"/></mets:div><mets:div ID=\"PHYS_0231\" ORDER=\"231\" ORDERLABEL=\"115\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0230_MIN\"/><mets:fptr FILEID=\"FILE_0230_DEFAULT\"/><mets:fptr FILEID=\"FILE_0230_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0230_MAX\"/><mets:fptr FILEID=\"FILE_0230_THUMBS\"/><mets:fptr FILEID=\"FILE_0230\"/></mets:div><mets:div ID=\"PHYS_0232\" ORDER=\"232\" ORDERLABEL=\"116\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0231_MIN\"/><mets:fptr FILEID=\"FILE_0231_DEFAULT\"/><mets:fptr FILEID=\"FILE_0231_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0231_MAX\"/><mets:fptr FILEID=\"FILE_0231_THUMBS\"/><mets:fptr FILEID=\"FILE_0231\"/></mets:div><mets:div ID=\"PHYS_0233\" ORDER=\"233\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0232_MIN\"/><mets:fptr FILEID=\"FILE_0232_DEFAULT\"/><mets:fptr FILEID=\"FILE_0232_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0232_MAX\"/><mets:fptr FILEID=\"FILE_0232_THUMBS\"/><mets:fptr FILEID=\"FILE_0232\"/></mets:div><mets:div ID=\"PHYS_0234\" ORDER=\"234\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0233_MIN\"/><mets:fptr FILEID=\"FILE_0233_DEFAULT\"/><mets:fptr FILEID=\"FILE_0233_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0233_MAX\"/><mets:fptr FILEID=\"FILE_0233_THUMBS\"/><mets:fptr FILEID=\"FILE_0233\"/></mets:div><mets:div ID=\"PHYS_0235\" ORDER=\"235\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0234_MIN\"/><mets:fptr FILEID=\"FILE_0234_DEFAULT\"/><mets:fptr FILEID=\"FILE_0234_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0234_MAX\"/><mets:fptr FILEID=\"FILE_0234_THUMBS\"/><mets:fptr FILEID=\"FILE_0234\"/></mets:div><mets:div ID=\"PHYS_0236\" ORDER=\"236\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0235_MIN\"/><mets:fptr FILEID=\"FILE_0235_DEFAULT\"/><mets:fptr FILEID=\"FILE_0235_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0235_MAX\"/><mets:fptr FILEID=\"FILE_0235_THUMBS\"/><mets:fptr FILEID=\"FILE_0235\"/></mets:div><mets:div ID=\"PHYS_0237\" ORDER=\"237\" ORDERLABEL=\"117\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0236_MIN\"/><mets:fptr FILEID=\"FILE_0236_DEFAULT\"/><mets:fptr FILEID=\"FILE_0236_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0236_MAX\"/><mets:fptr FILEID=\"FILE_0236_THUMBS\"/><mets:fptr FILEID=\"FILE_0236\"/></mets:div><mets:div ID=\"PHYS_0238\" ORDER=\"238\" ORDERLABEL=\"118\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0237_MIN\"/><mets:fptr FILEID=\"FILE_0237_DEFAULT\"/><mets:fptr FILEID=\"FILE_0237_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0237_MAX\"/><mets:fptr FILEID=\"FILE_0237_THUMBS\"/><mets:fptr FILEID=\"FILE_0237\"/></mets:div><mets:div ID=\"PHYS_0239\" ORDER=\"239\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0238_MIN\"/><mets:fptr FILEID=\"FILE_0238_DEFAULT\"/><mets:fptr FILEID=\"FILE_0238_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0238_MAX\"/><mets:fptr FILEID=\"FILE_0238_THUMBS\"/><mets:fptr FILEID=\"FILE_0238\"/></mets:div><mets:div ID=\"PHYS_0240\" ORDER=\"240\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0239_MIN\"/><mets:fptr FILEID=\"FILE_0239_DEFAULT\"/><mets:fptr FILEID=\"FILE_0239_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0239_MAX\"/><mets:fptr FILEID=\"FILE_0239_THUMBS\"/><mets:fptr FILEID=\"FILE_0239\"/></mets:div><mets:div ID=\"PHYS_0241\" ORDER=\"241\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0240_MIN\"/><mets:fptr FILEID=\"FILE_0240_DEFAULT\"/><mets:fptr FILEID=\"FILE_0240_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0240_MAX\"/><mets:fptr FILEID=\"FILE_0240_THUMBS\"/><mets:fptr FILEID=\"FILE_0240\"/></mets:div><mets:div ID=\"PHYS_0242\" ORDER=\"242\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0241_MIN\"/><mets:fptr FILEID=\"FILE_0241_DEFAULT\"/><mets:fptr FILEID=\"FILE_0241_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0241_MAX\"/><mets:fptr FILEID=\"FILE_0241_THUMBS\"/><mets:fptr FILEID=\"FILE_0241\"/></mets:div><mets:div ID=\"PHYS_0243\" ORDER=\"243\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0242_MIN\"/><mets:fptr FILEID=\"FILE_0242_DEFAULT\"/><mets:fptr FILEID=\"FILE_0242_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0242_MAX\"/><mets:fptr FILEID=\"FILE_0242_THUMBS\"/><mets:fptr FILEID=\"FILE_0242\"/></mets:div><mets:div ID=\"PHYS_0244\" ORDER=\"244\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0243_MIN\"/><mets:fptr FILEID=\"FILE_0243_DEFAULT\"/><mets:fptr FILEID=\"FILE_0243_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0243_MAX\"/><mets:fptr FILEID=\"FILE_0243_THUMBS\"/><mets:fptr FILEID=\"FILE_0243\"/></mets:div><mets:div ID=\"PHYS_0245\" ORDER=\"245\" ORDERLABEL=\"119\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0244_MIN\"/><mets:fptr FILEID=\"FILE_0244_DEFAULT\"/><mets:fptr FILEID=\"FILE_0244_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0244_MAX\"/><mets:fptr FILEID=\"FILE_0244_THUMBS\"/><mets:fptr FILEID=\"FILE_0244\"/></mets:div><mets:div ID=\"PHYS_0246\" ORDER=\"246\" ORDERLABEL=\"120\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0245_MIN\"/><mets:fptr FILEID=\"FILE_0245_DEFAULT\"/><mets:fptr FILEID=\"FILE_0245_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0245_MAX\"/><mets:fptr FILEID=\"FILE_0245_THUMBS\"/><mets:fptr FILEID=\"FILE_0245\"/></mets:div><mets:div ID=\"PHYS_0247\" ORDER=\"247\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0246_MIN\"/><mets:fptr FILEID=\"FILE_0246_DEFAULT\"/><mets:fptr FILEID=\"FILE_0246_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0246_MAX\"/><mets:fptr FILEID=\"FILE_0246_THUMBS\"/><mets:fptr FILEID=\"FILE_0246\"/></mets:div><mets:div ID=\"PHYS_0248\" ORDER=\"248\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0247_MIN\"/><mets:fptr FILEID=\"FILE_0247_DEFAULT\"/><mets:fptr FILEID=\"FILE_0247_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0247_MAX\"/><mets:fptr FILEID=\"FILE_0247_THUMBS\"/><mets:fptr FILEID=\"FILE_0247\"/></mets:div><mets:div ID=\"PHYS_0249\" ORDER=\"249\" ORDERLABEL=\"121\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0248_MIN\"/><mets:fptr FILEID=\"FILE_0248_DEFAULT\"/><mets:fptr FILEID=\"FILE_0248_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0248_MAX\"/><mets:fptr FILEID=\"FILE_0248_THUMBS\"/><mets:fptr FILEID=\"FILE_0248\"/></mets:div><mets:div ID=\"PHYS_0250\" ORDER=\"250\" ORDERLABEL=\"122\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0249_MIN\"/><mets:fptr FILEID=\"FILE_0249_DEFAULT\"/><mets:fptr FILEID=\"FILE_0249_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0249_MAX\"/><mets:fptr FILEID=\"FILE_0249_THUMBS\"/><mets:fptr FILEID=\"FILE_0249\"/></mets:div><mets:div ID=\"PHYS_0251\" ORDER=\"251\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0250_MIN\"/><mets:fptr FILEID=\"FILE_0250_DEFAULT\"/><mets:fptr FILEID=\"FILE_0250_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0250_MAX\"/><mets:fptr FILEID=\"FILE_0250_THUMBS\"/><mets:fptr FILEID=\"FILE_0250\"/></mets:div><mets:div ID=\"PHYS_0252\" ORDER=\"252\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0251_MIN\"/><mets:fptr FILEID=\"FILE_0251_DEFAULT\"/><mets:fptr FILEID=\"FILE_0251_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0251_MAX\"/><mets:fptr FILEID=\"FILE_0251_THUMBS\"/><mets:fptr FILEID=\"FILE_0251\"/></mets:div><mets:div ID=\"PHYS_0253\" ORDER=\"253\" ORDERLABEL=\"123\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0252_MIN\"/><mets:fptr FILEID=\"FILE_0252_DEFAULT\"/><mets:fptr FILEID=\"FILE_0252_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0252_MAX\"/><mets:fptr FILEID=\"FILE_0252_THUMBS\"/><mets:fptr FILEID=\"FILE_0252\"/></mets:div><mets:div ID=\"PHYS_0254\" ORDER=\"254\" ORDERLABEL=\"124\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0253_MIN\"/><mets:fptr FILEID=\"FILE_0253_DEFAULT\"/><mets:fptr FILEID=\"FILE_0253_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0253_MAX\"/><mets:fptr FILEID=\"FILE_0253_THUMBS\"/><mets:fptr FILEID=\"FILE_0253\"/></mets:div><mets:div ID=\"PHYS_0255\" ORDER=\"255\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0254_MIN\"/><mets:fptr FILEID=\"FILE_0254_DEFAULT\"/><mets:fptr FILEID=\"FILE_0254_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0254_MAX\"/><mets:fptr FILEID=\"FILE_0254_THUMBS\"/><mets:fptr FILEID=\"FILE_0254\"/></mets:div><mets:div ID=\"PHYS_0256\" ORDER=\"256\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0255_MIN\"/><mets:fptr FILEID=\"FILE_0255_DEFAULT\"/><mets:fptr FILEID=\"FILE_0255_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0255_MAX\"/><mets:fptr FILEID=\"FILE_0255_THUMBS\"/><mets:fptr FILEID=\"FILE_0255\"/></mets:div><mets:div ID=\"PHYS_0257\" ORDER=\"257\" ORDERLABEL=\"125\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0256_MIN\"/><mets:fptr FILEID=\"FILE_0256_DEFAULT\"/><mets:fptr FILEID=\"FILE_0256_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0256_MAX\"/><mets:fptr FILEID=\"FILE_0256_THUMBS\"/><mets:fptr FILEID=\"FILE_0256\"/></mets:div><mets:div ID=\"PHYS_0258\" ORDER=\"258\" ORDERLABEL=\"126\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0257_MIN\"/><mets:fptr FILEID=\"FILE_0257_DEFAULT\"/><mets:fptr FILEID=\"FILE_0257_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0257_MAX\"/><mets:fptr FILEID=\"FILE_0257_THUMBS\"/><mets:fptr FILEID=\"FILE_0257\"/></mets:div><mets:div ID=\"PHYS_0259\" ORDER=\"259\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0258_MIN\"/><mets:fptr FILEID=\"FILE_0258_DEFAULT\"/><mets:fptr FILEID=\"FILE_0258_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0258_MAX\"/><mets:fptr FILEID=\"FILE_0258_THUMBS\"/><mets:fptr FILEID=\"FILE_0258\"/></mets:div><mets:div ID=\"PHYS_0260\" ORDER=\"260\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0259_MIN\"/><mets:fptr FILEID=\"FILE_0259_DEFAULT\"/><mets:fptr FILEID=\"FILE_0259_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0259_MAX\"/><mets:fptr FILEID=\"FILE_0259_THUMBS\"/><mets:fptr FILEID=\"FILE_0259\"/></mets:div><mets:div ID=\"PHYS_0261\" ORDER=\"261\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0260_MIN\"/><mets:fptr FILEID=\"FILE_0260_DEFAULT\"/><mets:fptr FILEID=\"FILE_0260_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0260_MAX\"/><mets:fptr FILEID=\"FILE_0260_THUMBS\"/><mets:fptr FILEID=\"FILE_0260\"/></mets:div><mets:div ID=\"PHYS_0262\" ORDER=\"262\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0261_MIN\"/><mets:fptr FILEID=\"FILE_0261_DEFAULT\"/><mets:fptr FILEID=\"FILE_0261_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0261_MAX\"/><mets:fptr FILEID=\"FILE_0261_THUMBS\"/><mets:fptr FILEID=\"FILE_0261\"/></mets:div><mets:div ID=\"PHYS_0263\" ORDER=\"263\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0262_MIN\"/><mets:fptr FILEID=\"FILE_0262_DEFAULT\"/><mets:fptr FILEID=\"FILE_0262_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0262_MAX\"/><mets:fptr FILEID=\"FILE_0262_THUMBS\"/><mets:fptr FILEID=\"FILE_0262\"/></mets:div><mets:div ID=\"PHYS_0264\" ORDER=\"264\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0263_MIN\"/><mets:fptr FILEID=\"FILE_0263_DEFAULT\"/><mets:fptr FILEID=\"FILE_0263_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0263_MAX\"/><mets:fptr FILEID=\"FILE_0263_THUMBS\"/><mets:fptr FILEID=\"FILE_0263\"/></mets:div><mets:div ID=\"PHYS_0265\" ORDER=\"265\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0264_MIN\"/><mets:fptr FILEID=\"FILE_0264_DEFAULT\"/><mets:fptr FILEID=\"FILE_0264_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0264_MAX\"/><mets:fptr FILEID=\"FILE_0264_THUMBS\"/><mets:fptr FILEID=\"FILE_0264\"/></mets:div><mets:div ID=\"PHYS_0266\" ORDER=\"266\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0265_MIN\"/><mets:fptr FILEID=\"FILE_0265_DEFAULT\"/><mets:fptr FILEID=\"FILE_0265_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0265_MAX\"/><mets:fptr FILEID=\"FILE_0265_THUMBS\"/><mets:fptr FILEID=\"FILE_0265\"/></mets:div><mets:div ID=\"PHYS_0267\" ORDER=\"267\" ORDERLABEL=\"127\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0266_MIN\"/><mets:fptr FILEID=\"FILE_0266_DEFAULT\"/><mets:fptr FILEID=\"FILE_0266_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0266_MAX\"/><mets:fptr FILEID=\"FILE_0266_THUMBS\"/><mets:fptr FILEID=\"FILE_0266\"/></mets:div><mets:div ID=\"PHYS_0268\" ORDER=\"268\" ORDERLABEL=\"128\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0267_MIN\"/><mets:fptr FILEID=\"FILE_0267_DEFAULT\"/><mets:fptr FILEID=\"FILE_0267_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0267_MAX\"/><mets:fptr FILEID=\"FILE_0267_THUMBS\"/><mets:fptr FILEID=\"FILE_0267\"/></mets:div><mets:div ID=\"PHYS_0269\" ORDER=\"269\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0268_MIN\"/><mets:fptr FILEID=\"FILE_0268_DEFAULT\"/><mets:fptr FILEID=\"FILE_0268_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0268_MAX\"/><mets:fptr FILEID=\"FILE_0268_THUMBS\"/><mets:fptr FILEID=\"FILE_0268\"/></mets:div><mets:div ID=\"PHYS_0270\" ORDER=\"270\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0269_MIN\"/><mets:fptr FILEID=\"FILE_0269_DEFAULT\"/><mets:fptr FILEID=\"FILE_0269_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0269_MAX\"/><mets:fptr FILEID=\"FILE_0269_THUMBS\"/><mets:fptr FILEID=\"FILE_0269\"/></mets:div><mets:div ID=\"PHYS_0271\" ORDER=\"271\" ORDERLABEL=\"129\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0270_MIN\"/><mets:fptr FILEID=\"FILE_0270_DEFAULT\"/><mets:fptr FILEID=\"FILE_0270_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0270_MAX\"/><mets:fptr FILEID=\"FILE_0270_THUMBS\"/><mets:fptr FILEID=\"FILE_0270\"/></mets:div><mets:div ID=\"PHYS_0272\" ORDER=\"272\" ORDERLABEL=\"130\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0271_MIN\"/><mets:fptr FILEID=\"FILE_0271_DEFAULT\"/><mets:fptr FILEID=\"FILE_0271_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0271_MAX\"/><mets:fptr FILEID=\"FILE_0271_THUMBS\"/><mets:fptr FILEID=\"FILE_0271\"/></mets:div><mets:div ID=\"PHYS_0273\" ORDER=\"273\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0272_MIN\"/><mets:fptr FILEID=\"FILE_0272_DEFAULT\"/><mets:fptr FILEID=\"FILE_0272_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0272_MAX\"/><mets:fptr FILEID=\"FILE_0272_THUMBS\"/><mets:fptr FILEID=\"FILE_0272\"/></mets:div><mets:div ID=\"PHYS_0274\" ORDER=\"274\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0273_MIN\"/><mets:fptr FILEID=\"FILE_0273_DEFAULT\"/><mets:fptr FILEID=\"FILE_0273_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0273_MAX\"/><mets:fptr FILEID=\"FILE_0273_THUMBS\"/><mets:fptr FILEID=\"FILE_0273\"/></mets:div><mets:div ID=\"PHYS_0275\" ORDER=\"275\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0274_MIN\"/><mets:fptr FILEID=\"FILE_0274_DEFAULT\"/><mets:fptr FILEID=\"FILE_0274_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0274_MAX\"/><mets:fptr FILEID=\"FILE_0274_THUMBS\"/><mets:fptr FILEID=\"FILE_0274\"/></mets:div><mets:div ID=\"PHYS_0276\" ORDER=\"276\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0275_MIN\"/><mets:fptr FILEID=\"FILE_0275_DEFAULT\"/><mets:fptr FILEID=\"FILE_0275_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0275_MAX\"/><mets:fptr FILEID=\"FILE_0275_THUMBS\"/><mets:fptr FILEID=\"FILE_0275\"/></mets:div><mets:div ID=\"PHYS_0277\" ORDER=\"277\" ORDERLABEL=\"131\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0276_MIN\"/><mets:fptr FILEID=\"FILE_0276_DEFAULT\"/><mets:fptr FILEID=\"FILE_0276_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0276_MAX\"/><mets:fptr FILEID=\"FILE_0276_THUMBS\"/><mets:fptr FILEID=\"FILE_0276\"/></mets:div><mets:div ID=\"PHYS_0278\" ORDER=\"278\" ORDERLABEL=\"132\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0277_MIN\"/><mets:fptr FILEID=\"FILE_0277_DEFAULT\"/><mets:fptr FILEID=\"FILE_0277_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0277_MAX\"/><mets:fptr FILEID=\"FILE_0277_THUMBS\"/><mets:fptr FILEID=\"FILE_0277\"/></mets:div><mets:div ID=\"PHYS_0279\" ORDER=\"279\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0278_MIN\"/><mets:fptr FILEID=\"FILE_0278_DEFAULT\"/><mets:fptr FILEID=\"FILE_0278_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0278_MAX\"/><mets:fptr FILEID=\"FILE_0278_THUMBS\"/><mets:fptr FILEID=\"FILE_0278\"/></mets:div><mets:div ID=\"PHYS_0280\" ORDER=\"280\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0279_MIN\"/><mets:fptr FILEID=\"FILE_0279_DEFAULT\"/><mets:fptr FILEID=\"FILE_0279_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0279_MAX\"/><mets:fptr FILEID=\"FILE_0279_THUMBS\"/><mets:fptr FILEID=\"FILE_0279\"/></mets:div><mets:div ID=\"PHYS_0281\" ORDER=\"281\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0280_MIN\"/><mets:fptr FILEID=\"FILE_0280_DEFAULT\"/><mets:fptr FILEID=\"FILE_0280_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0280_MAX\"/><mets:fptr FILEID=\"FILE_0280_THUMBS\"/><mets:fptr FILEID=\"FILE_0280\"/></mets:div><mets:div ID=\"PHYS_0282\" ORDER=\"282\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0281_MIN\"/><mets:fptr FILEID=\"FILE_0281_DEFAULT\"/><mets:fptr FILEID=\"FILE_0281_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0281_MAX\"/><mets:fptr FILEID=\"FILE_0281_THUMBS\"/><mets:fptr FILEID=\"FILE_0281\"/></mets:div><mets:div ID=\"PHYS_0283\" ORDER=\"283\" ORDERLABEL=\"133\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0282_MIN\"/><mets:fptr FILEID=\"FILE_0282_DEFAULT\"/><mets:fptr FILEID=\"FILE_0282_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0282_MAX\"/><mets:fptr FILEID=\"FILE_0282_THUMBS\"/><mets:fptr FILEID=\"FILE_0282\"/></mets:div><mets:div ID=\"PHYS_0284\" ORDER=\"284\" ORDERLABEL=\"134\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0283_MIN\"/><mets:fptr FILEID=\"FILE_0283_DEFAULT\"/><mets:fptr FILEID=\"FILE_0283_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0283_MAX\"/><mets:fptr FILEID=\"FILE_0283_THUMBS\"/><mets:fptr FILEID=\"FILE_0283\"/></mets:div><mets:div ID=\"PHYS_0285\" ORDER=\"285\" ORDERLABEL=\"135\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0284_MIN\"/><mets:fptr FILEID=\"FILE_0284_DEFAULT\"/><mets:fptr FILEID=\"FILE_0284_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0284_MAX\"/><mets:fptr FILEID=\"FILE_0284_THUMBS\"/><mets:fptr FILEID=\"FILE_0284\"/></mets:div><mets:div ID=\"PHYS_0286\" ORDER=\"286\" ORDERLABEL=\"136\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0285_MIN\"/><mets:fptr FILEID=\"FILE_0285_DEFAULT\"/><mets:fptr FILEID=\"FILE_0285_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0285_MAX\"/><mets:fptr FILEID=\"FILE_0285_THUMBS\"/><mets:fptr FILEID=\"FILE_0285\"/></mets:div><mets:div ID=\"PHYS_0287\" ORDER=\"287\" ORDERLABEL=\"137\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0286_MIN\"/><mets:fptr FILEID=\"FILE_0286_DEFAULT\"/><mets:fptr FILEID=\"FILE_0286_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0286_MAX\"/><mets:fptr FILEID=\"FILE_0286_THUMBS\"/><mets:fptr FILEID=\"FILE_0286\"/></mets:div><mets:div ID=\"PHYS_0288\" ORDER=\"288\" ORDERLABEL=\"138\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0287_MIN\"/><mets:fptr FILEID=\"FILE_0287_DEFAULT\"/><mets:fptr FILEID=\"FILE_0287_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0287_MAX\"/><mets:fptr FILEID=\"FILE_0287_THUMBS\"/><mets:fptr FILEID=\"FILE_0287\"/></mets:div><mets:div ID=\"PHYS_0289\" ORDER=\"289\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0288_MIN\"/><mets:fptr FILEID=\"FILE_0288_DEFAULT\"/><mets:fptr FILEID=\"FILE_0288_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0288_MAX\"/><mets:fptr FILEID=\"FILE_0288_THUMBS\"/><mets:fptr FILEID=\"FILE_0288\"/></mets:div><mets:div ID=\"PHYS_0290\" ORDER=\"290\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0289_MIN\"/><mets:fptr FILEID=\"FILE_0289_DEFAULT\"/><mets:fptr FILEID=\"FILE_0289_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0289_MAX\"/><mets:fptr FILEID=\"FILE_0289_THUMBS\"/><mets:fptr FILEID=\"FILE_0289\"/></mets:div><mets:div ID=\"PHYS_0291\" ORDER=\"291\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0290_MIN\"/><mets:fptr FILEID=\"FILE_0290_DEFAULT\"/><mets:fptr FILEID=\"FILE_0290_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0290_MAX\"/><mets:fptr FILEID=\"FILE_0290_THUMBS\"/><mets:fptr FILEID=\"FILE_0290\"/></mets:div><mets:div ID=\"PHYS_0292\" ORDER=\"292\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0291_MIN\"/><mets:fptr FILEID=\"FILE_0291_DEFAULT\"/><mets:fptr FILEID=\"FILE_0291_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0291_MAX\"/><mets:fptr FILEID=\"FILE_0291_THUMBS\"/><mets:fptr FILEID=\"FILE_0291\"/></mets:div><mets:div ID=\"PHYS_0293\" ORDER=\"293\" ORDERLABEL=\"139\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0292_MIN\"/><mets:fptr FILEID=\"FILE_0292_DEFAULT\"/><mets:fptr FILEID=\"FILE_0292_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0292_MAX\"/><mets:fptr FILEID=\"FILE_0292_THUMBS\"/><mets:fptr FILEID=\"FILE_0292\"/></mets:div><mets:div ID=\"PHYS_0294\" ORDER=\"294\" ORDERLABEL=\"140\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0293_MIN\"/><mets:fptr FILEID=\"FILE_0293_DEFAULT\"/><mets:fptr FILEID=\"FILE_0293_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0293_MAX\"/><mets:fptr FILEID=\"FILE_0293_THUMBS\"/><mets:fptr FILEID=\"FILE_0293\"/></mets:div><mets:div ID=\"PHYS_0295\" ORDER=\"295\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0294_MIN\"/><mets:fptr FILEID=\"FILE_0294_DEFAULT\"/><mets:fptr FILEID=\"FILE_0294_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0294_MAX\"/><mets:fptr FILEID=\"FILE_0294_THUMBS\"/><mets:fptr FILEID=\"FILE_0294\"/></mets:div><mets:div ID=\"PHYS_0296\" ORDER=\"296\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0295_MIN\"/><mets:fptr FILEID=\"FILE_0295_DEFAULT\"/><mets:fptr FILEID=\"FILE_0295_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0295_MAX\"/><mets:fptr FILEID=\"FILE_0295_THUMBS\"/><mets:fptr FILEID=\"FILE_0295\"/></mets:div><mets:div ID=\"PHYS_0297\" ORDER=\"297\" ORDERLABEL=\"141\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0296_MIN\"/><mets:fptr FILEID=\"FILE_0296_DEFAULT\"/><mets:fptr FILEID=\"FILE_0296_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0296_MAX\"/><mets:fptr FILEID=\"FILE_0296_THUMBS\"/><mets:fptr FILEID=\"FILE_0296\"/></mets:div><mets:div ID=\"PHYS_0298\" ORDER=\"298\" ORDERLABEL=\"142\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0297_MIN\"/><mets:fptr FILEID=\"FILE_0297_DEFAULT\"/><mets:fptr FILEID=\"FILE_0297_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0297_MAX\"/><mets:fptr FILEID=\"FILE_0297_THUMBS\"/><mets:fptr FILEID=\"FILE_0297\"/></mets:div><mets:div ID=\"PHYS_0299\" ORDER=\"299\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0298_MIN\"/><mets:fptr FILEID=\"FILE_0298_DEFAULT\"/><mets:fptr FILEID=\"FILE_0298_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0298_MAX\"/><mets:fptr FILEID=\"FILE_0298_THUMBS\"/><mets:fptr FILEID=\"FILE_0298\"/></mets:div><mets:div ID=\"PHYS_0300\" ORDER=\"300\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0299_MIN\"/><mets:fptr FILEID=\"FILE_0299_DEFAULT\"/><mets:fptr FILEID=\"FILE_0299_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0299_MAX\"/><mets:fptr FILEID=\"FILE_0299_THUMBS\"/><mets:fptr FILEID=\"FILE_0299\"/></mets:div><mets:div ID=\"PHYS_0301\" ORDER=\"301\" ORDERLABEL=\"143\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0300_MIN\"/><mets:fptr FILEID=\"FILE_0300_DEFAULT\"/><mets:fptr FILEID=\"FILE_0300_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0300_MAX\"/><mets:fptr FILEID=\"FILE_0300_THUMBS\"/><mets:fptr FILEID=\"FILE_0300\"/></mets:div><mets:div ID=\"PHYS_0302\" ORDER=\"302\" TYPE=\"page\"><mets:fptr FILEID=\"FILE_0301_MIN\"/><mets:fptr FILEID=\"FILE_0301_DEFAULT\"/><mets:fptr FILEID=\"FILE_0301_PRESENTATION\"/><mets:fptr FILEID=\"FILE_0301_MAX\"/><mets:fptr FILEID=\"FILE_0301_THUMBS\"/><mets:fptr FILEID=\"FILE_0301\"/></mets:div></mets:div></mets:structMap><mets:structLink><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0001\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0002\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0003\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0004\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0005\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0006\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0007\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0008\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0009\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0010\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0011\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0012\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0013\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0014\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0015\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0016\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0017\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0018\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0019\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0020\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0021\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0022\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0023\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0024\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0025\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0026\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0027\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0028\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0029\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0030\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0031\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0032\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0033\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0034\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0035\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0036\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0037\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0038\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0039\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0040\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0041\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0042\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0043\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0044\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0045\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0046\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0047\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0048\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0049\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0050\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0051\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0052\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0053\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0054\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0055\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0056\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0057\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0058\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0059\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0060\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0061\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0062\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0063\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0064\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0065\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0066\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0067\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0068\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0069\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0070\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0071\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0072\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0073\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0074\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0075\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0076\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0077\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0078\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0079\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0080\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0081\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0082\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0083\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0084\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0085\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0086\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0087\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0088\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0089\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0090\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0091\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0092\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0093\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0094\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0095\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0096\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0097\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0098\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0099\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0100\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0101\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0102\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0103\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0103\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0104\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0105\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0106\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0107\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0108\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0109\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0110\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0111\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0112\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0113\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0114\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0115\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0116\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0117\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0118\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0119\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0120\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0121\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0122\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0123\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0124\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0125\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0126\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0127\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0128\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0128\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0129\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0130\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0131\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0132\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0133\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0134\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0135\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0136\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0137\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0138\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0139\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0140\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0141\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0142\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0143\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0144\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0145\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0146\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0147\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0148\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0149\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0150\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0151\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0152\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0153\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0154\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0155\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0156\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0157\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0158\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0159\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0160\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0161\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0162\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0163\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0164\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0165\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0166\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0167\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0168\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0169\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0170\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0171\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0171\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0172\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0173\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0174\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0175\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0176\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0177\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0178\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0179\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0180\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0181\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0182\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0183\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0184\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0185\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0186\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0187\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0188\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0189\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0190\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0191\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0192\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0193\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0194\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0195\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0196\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0197\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0198\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0199\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0200\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0201\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0202\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0203\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0204\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0205\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0205\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0206\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0207\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0208\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0209\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0210\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0211\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0212\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0213\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0214\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0215\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0216\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0217\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0218\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0219\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0220\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0221\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0222\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0223\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0224\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0225\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0226\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0227\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0228\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0229\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0230\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0231\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0232\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0233\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0234\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0235\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0236\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0237\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0238\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0239\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0240\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0241\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0242\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0243\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0244\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0245\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0246\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0247\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0248\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0249\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0250\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0251\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0252\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0253\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0254\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0255\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0256\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0257\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0258\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0259\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0260\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0261\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0262\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0263\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0264\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0265\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0266\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0267\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0268\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0269\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0270\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0271\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0272\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0273\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0274\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0275\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0276\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0277\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0278\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0279\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0280\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0281\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0282\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0283\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0284\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0284\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0285\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0286\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0287\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0288\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0289\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0290\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0291\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0292\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0293\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0294\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0295\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0296\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0297\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0298\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0299\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0300\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0301\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0302\" xlink:from=\"LOG_0003\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0001\" xlink:from=\"LOG_0004\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0002\" xlink:from=\"LOG_0004\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0003\" xlink:from=\"LOG_0005\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0004\" xlink:from=\"LOG_0005\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0005\" xlink:from=\"LOG_0006\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0006\" xlink:from=\"LOG_0006\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0007\" xlink:from=\"LOG_0007\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0008\" xlink:from=\"LOG_0007\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0009\" xlink:from=\"LOG_0008\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0010\" xlink:from=\"LOG_0008\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0011\" xlink:from=\"LOG_0008\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0012\" xlink:from=\"LOG_0008\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0013\" xlink:from=\"LOG_0008\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0014\" xlink:from=\"LOG_0009\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0015\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0016\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0017\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0018\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0019\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0020\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0021\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0022\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0023\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0024\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0025\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0026\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0027\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0028\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0029\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0030\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0031\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0032\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0033\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0034\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0035\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0036\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0037\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0038\" xlink:from=\"LOG_0010\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0039\" xlink:from=\"LOG_0011\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0040\" xlink:from=\"LOG_0011\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0041\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0042\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0043\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0044\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0045\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0046\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0047\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0048\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0049\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0050\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0051\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0052\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0053\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0054\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0055\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0056\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0057\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0058\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0059\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0060\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0061\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0062\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0063\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0064\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0065\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0066\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0067\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0068\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0069\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0070\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0071\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0072\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0073\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0074\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0075\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0076\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0077\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0078\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0079\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0080\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0081\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0082\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0083\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0084\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0085\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0086\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0087\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0088\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0089\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0090\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0091\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0092\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0093\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0094\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0095\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0096\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0097\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0098\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0099\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0100\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0101\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0102\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0103\" xlink:from=\"LOG_0012\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0041\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0042\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0043\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0044\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0045\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0046\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0047\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0048\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0049\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0050\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0051\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0052\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0053\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0054\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0055\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0056\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0057\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0058\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0059\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0060\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0061\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0062\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0063\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0064\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0065\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0066\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0067\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0068\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0069\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0070\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0071\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0072\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0073\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0074\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0075\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0076\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0077\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0078\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0079\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0080\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0081\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0082\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0083\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0084\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0085\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0086\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0087\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0088\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0089\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0090\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0091\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0092\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0093\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0094\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0095\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0096\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0097\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0098\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0099\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0100\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0101\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0102\" xlink:from=\"LOG_0013\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0043\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0044\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0045\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0046\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0047\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0048\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0049\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0050\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0051\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0052\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0053\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0054\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0055\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0056\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0057\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0058\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0059\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0060\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0061\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0062\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0063\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0064\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0065\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0066\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0067\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0068\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0069\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0070\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0071\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0072\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0073\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0074\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0075\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0076\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0077\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0078\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0079\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0080\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0081\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0082\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0083\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0084\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0085\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0086\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0087\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0088\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0089\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0090\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0091\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0092\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0093\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0094\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0095\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0096\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0097\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0098\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0099\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0100\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0101\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0102\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0103\" xlink:from=\"LOG_0014\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0103\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0104\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0105\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0106\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0107\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0108\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0109\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0110\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0111\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0112\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0113\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0114\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0115\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0116\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0117\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0118\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0119\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0120\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0121\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0122\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0123\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0124\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0125\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0126\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0127\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0128\" xlink:from=\"LOG_0015\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0107\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0108\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0109\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0110\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0111\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0112\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0113\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0114\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0115\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0116\" xlink:from=\"LOG_0016\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0128\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0129\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0130\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0131\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0132\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0133\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0134\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0135\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0136\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0137\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0138\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0139\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0140\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0141\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0142\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0143\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0144\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0145\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0146\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0147\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0148\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0149\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0150\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0151\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0152\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0153\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0154\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0155\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0156\" xlink:from=\"LOG_0017\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0131\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0132\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0133\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0134\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0135\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0136\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0137\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0138\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0139\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0140\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0141\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0142\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0143\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0144\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0145\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0146\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0147\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0148\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0149\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0150\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0151\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0152\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0153\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0154\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0155\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0156\" xlink:from=\"LOG_0018\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0157\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0158\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0159\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0160\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0161\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0162\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0163\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0164\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0165\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0166\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0167\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0168\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0169\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0170\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0171\" xlink:from=\"LOG_0019\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0159\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0160\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0161\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0162\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0163\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0164\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0165\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0166\" xlink:from=\"LOG_0020\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0171\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0172\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0173\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0174\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0175\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0176\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0177\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0178\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0179\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0180\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0181\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0182\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0183\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0184\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0185\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0186\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0187\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0188\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0189\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0190\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0191\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0192\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0193\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0194\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0195\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0196\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0197\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0198\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0199\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0200\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0201\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0202\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0203\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0204\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0205\" xlink:from=\"LOG_0021\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0173\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0174\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0175\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0176\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0177\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0178\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0179\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0180\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0181\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0182\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0183\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0184\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0185\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0186\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0187\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0188\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0189\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0190\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0191\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0192\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0193\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0194\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0195\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0196\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0197\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0198\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0199\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0200\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0201\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0202\" xlink:from=\"LOG_0022\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0205\" xlink:from=\"LOG_0023\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0206\" xlink:from=\"LOG_0023\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0207\" xlink:from=\"LOG_0023\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0208\" xlink:from=\"LOG_0023\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0209\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0210\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0211\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0212\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0213\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0214\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0215\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0216\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0217\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0218\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0219\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0220\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0221\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0222\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0223\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0224\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0225\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0226\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0227\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0228\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0229\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0230\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0231\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0232\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0233\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0234\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0235\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0236\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0237\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0238\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0239\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0240\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0241\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0242\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0243\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0244\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0245\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0246\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0247\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0248\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0249\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0250\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0251\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0252\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0253\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0254\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0255\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0256\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0257\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0258\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0259\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0260\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0261\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0262\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0263\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0264\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0265\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0266\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0267\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0268\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0269\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0270\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0271\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0272\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0273\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0274\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0275\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0276\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0277\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0278\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0279\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0280\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0281\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0282\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0283\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0284\" xlink:from=\"LOG_0024\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0211\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0212\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0213\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0214\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0215\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0216\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0217\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0218\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0219\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0220\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0221\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0222\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0223\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0224\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0225\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0226\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0227\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0228\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0229\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0230\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0231\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0232\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0233\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0234\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0235\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0236\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0237\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0238\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0239\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0240\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0241\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0242\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0243\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0244\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0245\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0246\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0247\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0248\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0249\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0250\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0251\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0252\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0253\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0254\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0255\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0256\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0257\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0258\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0259\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0260\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0261\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0262\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0263\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0264\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0265\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0266\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0267\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0268\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0269\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0270\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0271\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0272\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0273\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0274\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0275\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0276\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0277\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0278\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0279\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0280\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0281\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0282\" xlink:from=\"LOG_0025\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0284\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0285\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0286\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0287\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0288\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0289\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0290\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0291\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0292\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0293\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0294\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0295\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0296\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0297\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0298\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0299\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0300\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0301\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0302\" xlink:from=\"LOG_0026\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0289\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0290\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0291\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0292\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0293\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0294\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0295\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0296\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0297\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0298\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0299\" xlink:from=\"LOG_0027\"/><mets:smLink xmlns:xlink=\"http://www.w3.org/1999/xlink\" xlink:to=\"PHYS_0300\" xlink:from=\"LOG_0027\"/></mets:structLink></mets:mets>\n",
		"items": [
			{
				"itemType": "bookSection",
				"title": "Accurata, E Succinta Descrizione Topografica Delle Antichitá Di Roma",
				"creators": [
					{
						"firstName": "Ridolfino",
						"lastName": "Venuti",
						"creatorType": "author"
					}
				],
				"date": "1763",
				"language": "la",
				"place": "Roma",
				"publisher": "Bernabò, e Lazzarini",
				"attachments": [],
				"tags": [],
				"notes": [],
				"seeAlso": []
			}
		]
	}
]
/** END TEST CASES **/
