/*
    ***** BEGIN LICENSE BLOCK *****
    
    Copyright © 2009 Center for History and New Media
                     George Mason University, Fairfax, Virginia, USA
                     http://zotero.org
    
    This file is part of Zotero.
    
    Zotero is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Zotero is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.
    
    You should have received a copy of the GNU Affero General Public License
    along with Zotero.  If not, see <http://www.gnu.org/licenses/>.
    
    ***** END LICENSE BLOCK *****
*/


/*
 * Primary interface for accessing Zotero items
 */
Zotero.Items = function () {
  this.constructor = null;

  this._ZDO_object = 'item';

  // This needs to wait until all Zotero components are loaded to initialize,
  // but otherwise it can be just a simple property
  Zotero.defineProperty(this, "_primaryDataSQLParts", {
    get: function () {
      var itemTypeAttachment = Zotero.ItemTypes.getID('attachment');
      var itemTypeNote = Zotero.ItemTypes.getID('note');
      var itemTypeAnnotation = Zotero.ItemTypes.getID('annotation');

      return {
        itemID: "O.itemID",
        itemTypeID: "O.itemTypeID",
        dateAdded: "O.dateAdded",
        dateModified: "O.dateModified",
        libraryID: "O.libraryID",
        key: "O.key",
        version: "O.version",
        synced: "O.synced",

        createdByUserID: "createdByUserID",
        lastModifiedByUserID: "lastModifiedByUserID",

        firstCreator: _getFirstCreatorSQL(),
        sortCreator: _getSortCreatorSQL(),

        deleted: "DI.itemID IS NOT NULL AS deleted",
        inPublications: "PI.itemID IS NOT NULL AS inPublications",

        parentID: `(CASE O.itemTypeID ` +
        `WHEN ${itemTypeAttachment} THEN IAP.itemID ` +
        `WHEN ${itemTypeNote} THEN INoP.itemID ` +
        `WHEN ${itemTypeAnnotation} THEN IAnP.itemID ` +
        `END) AS parentID`,
        parentKey: `(CASE O.itemTypeID ` +
        `WHEN ${itemTypeAttachment} THEN IAP.key ` +
        `WHEN ${itemTypeNote} THEN INoP.key ` +
        `WHEN ${itemTypeAnnotation} THEN IAnP.key ` +
        `END) AS parentKey`,

        attachmentCharset: "CS.charset AS attachmentCharset",
        attachmentLinkMode: "IA.linkMode AS attachmentLinkMode",
        attachmentContentType: "IA.contentType AS attachmentContentType",
        attachmentPath: "IA.path AS attachmentPath",
        attachmentSyncState: "IA.syncState AS attachmentSyncState",
        attachmentSyncedModificationTime: "IA.storageModTime AS attachmentSyncedModificationTime",
        attachmentSyncedHash: "IA.storageHash AS attachmentSyncedHash",
        attachmentLastProcessedModificationTime: "IA.lastProcessedModificationTime AS attachmentLastProcessedModificationTime" };

    } },
  { lazy: true });


  this._primaryDataSQLFrom = "FROM items O " +
  "LEFT JOIN itemAttachments IA USING (itemID) " +
  "LEFT JOIN items IAP ON (IA.parentItemID=IAP.itemID) " +
  "LEFT JOIN itemNotes INo ON (O.itemID=INo.itemID) " +
  "LEFT JOIN items INoP ON (INo.parentItemID=INoP.itemID) " +
  "LEFT JOIN itemAnnotations IAn ON (O.itemID=IAn.itemID) " +
  "LEFT JOIN items IAnP ON (IAn.parentItemID=IAnP.itemID) " +
  "LEFT JOIN deletedItems DI ON (O.itemID=DI.itemID) " +
  "LEFT JOIN publicationsItems PI ON (O.itemID=PI.itemID) " +
  "LEFT JOIN charsets CS ON (IA.charsetID=CS.charsetID)" +
  "LEFT JOIN groupItems GI ON (O.itemID=GI.itemID)";

  this._relationsTable = "itemRelations";


  /**
   * @param {Integer} libraryID
   * @return {Promise<Boolean>} - True if library has items in trash, false otherwise
   */
  this.hasDeleted = Zotero.Promise.coroutine(function* (libraryID) {
    var sql = "SELECT COUNT(*) > 0 FROM items JOIN deletedItems USING (itemID) WHERE libraryID=?";
    return !!(yield Zotero.DB.valueQueryAsync(sql, [libraryID]));
  });


  /**
   * Returns all items in a given library
   *
   * @param  {Integer}  libraryID
   * @param  {Boolean}  [onlyTopLevel=false]   If true, don't include child items
   * @param  {Boolean}  [includeDeleted=false] If true, include deleted items
   * @param  {Boolean}  [asIDs=false] 		 If true, resolves only with IDs
   * @return {Promise<Array<Zotero.Item|Integer>>}
   */
  this.getAll = Zotero.Promise.coroutine(function* (libraryID, onlyTopLevel, includeDeleted, asIDs = false) {
    var sql = 'SELECT A.itemID FROM items A';
    if (onlyTopLevel) {
      sql += ' LEFT JOIN itemNotes B USING (itemID) ' +
      'LEFT JOIN itemAttachments C ON (C.itemID=A.itemID) ' +
      'WHERE B.parentItemID IS NULL AND C.parentItemID IS NULL';
    } else
    {
      sql += " WHERE 1";
    }
    if (!includeDeleted) {
      sql += " AND A.itemID NOT IN (SELECT itemID FROM deletedItems)";
    }
    sql += " AND libraryID=?";
    var ids = yield Zotero.DB.columnQueryAsync(sql, libraryID);
    if (asIDs) {
      return ids;
    }
    return this.getAsync(ids);
  });


  /**
   * Return item data in web API format
   *
   * var data = Zotero.Items.getAPIData(0, 'collections/NF3GJ38A/items');
   *
   * @param {Number} libraryID
   * @param {String} [apiPath='items'] - Web API style
   * @return {Promise<String>}.
   */
  this.getAPIData = Zotero.Promise.coroutine(function* (libraryID, apiPath) {
    var gen = this.getAPIDataGenerator(...arguments);
    var data = "";
    while (true) {
      var result = gen.next();
      if (result.done) {
        break;
      }
      var val = yield result.value;
      if (typeof val == 'string') {
        data += val;
      } else
      if (val === undefined) {
        continue;
      } else
      {
        throw new Error("Invalid return value from generator");
      }
    }
    return data;
  });


  /**
   * Zotero.Utilities.Internal.getAsyncInputStream-compatible generator that yields item data
   * in web API format as strings
   *
   * @param {Object} params - Request parameters from Zotero.API.parsePath()
   */
  this.apiDataGenerator = function* (params) {
    Zotero.debug(params);
    var s = new Zotero.Search();
    s.addCondition('libraryID', 'is', params.libraryID);
    if (params.scopeObject == 'collections') {
      s.addCondition('collection', 'is', params.scopeObjectKey);
    }
    s.addCondition('title', 'contains', 'test');
    var ids = yield s.search();

    yield '[\n';

    for (let i = 0; i < ids.length; i++) {
      let prefix = i > 0 ? ',\n' : '';
      let item = yield this.getAsync(ids[i], { noCache: true });
      var json = item.toResponseJSON();
      yield prefix + JSON.stringify(json, null, 4);
    }

    yield '\n]';
  };


  //
  // Bulk data loading functions
  //
  // These are called by Zotero.DataObjects.prototype._loadDataType().
  //
  this._loadItemData = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var missingItems = {};
    var itemFieldsCached = {};

    var sql = "SELECT itemID, fieldID, value FROM items " +
    "JOIN itemData USING (itemID) " +
    "JOIN itemDataValues USING (valueID) WHERE libraryID=? AND itemTypeID!=?" + idSQL;
    var params = [libraryID, Zotero.ItemTypes.getID('note')];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);
        let fieldID = row.getResultByIndex(1);
        let value = row.getResultByIndex(2);

        //Zotero.debug('Setting field ' + fieldID + ' for item ' + itemID);
        if (this._objectCache[itemID]) {
          if (value === null) {
            value = false;
          }
          this._objectCache[itemID].setField(fieldID, value, true);
        } else
        {
          if (!missingItems[itemID]) {
            missingItems[itemID] = true;
            Zotero.logError("itemData row references nonexistent item " + itemID);
          }
        }
        if (!itemFieldsCached[itemID]) {
          itemFieldsCached[itemID] = {};
        }
        itemFieldsCached[itemID][fieldID] = true;
      }.bind(this) });



    var sql = "SELECT itemID FROM items WHERE libraryID=?" + idSQL;
    var params = [libraryID];
    var allItemIDs = [];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);
        let item = this._objectCache[itemID];

        // Set nonexistent fields in the cache list to false (instead of null)
        let fieldIDs = Zotero.ItemFields.getItemTypeFields(item.itemTypeID);
        for (let j = 0; j < fieldIDs.length; j++) {
          let fieldID = fieldIDs[j];
          if (!itemFieldsCached[itemID] || !itemFieldsCached[itemID][fieldID]) {
            //Zotero.debug('Setting field ' + fieldID + ' to false for item ' + itemID);
            item.setField(fieldID, false, true);
          }
        }

        allItemIDs.push(itemID);
      }.bind(this) });




    var titleFieldID = Zotero.ItemFields.getID('title');

    // Note titles
    var sql = "SELECT itemID, title FROM items JOIN itemNotes USING (itemID) " +
    "WHERE libraryID=? AND itemID NOT IN (SELECT itemID FROM itemAttachments)" + idSQL;
    var params = [libraryID];

    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);
        let title = row.getResultByIndex(1);

        //Zotero.debug('Setting title for note ' + row.itemID);
        if (this._objectCache[itemID]) {
          this._objectCache[itemID].setField(titleFieldID, title, true);
        } else
        {
          if (!missingItems[itemID]) {
            missingItems[itemID] = true;
            Zotero.logError("itemData row references nonexistent item " + itemID);
          }
        }
      }.bind(this) });



    for (let i = 0; i < allItemIDs.length; i++) {
      let itemID = allItemIDs[i];
      let item = this._objectCache[itemID];

      // Mark as loaded
      item._loaded.itemData = true;
      item._clearChanged('itemData');

      // Display titles
      try {
        item.updateDisplayTitle();
      }
      catch (e) {
        // A few item types need creators to be loaded. Instead of making
        // updateDisplayTitle() async and loading conditionally, just catch the error
        // and load on demand
        if (e instanceof Zotero.Exception.UnloadedDataException) {
          yield item.loadDataType('creators');
          item.updateDisplayTitle();
        } else
        {
          throw e;
        }
      }
    }
  });


  this._loadCreators = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var sql = 'SELECT itemID, creatorID, creatorTypeID, orderIndex ' +
    'FROM items LEFT JOIN itemCreators USING (itemID) ' +
    'WHERE libraryID=?' + idSQL + " ORDER BY itemID, orderIndex";
    var params = [libraryID];
    var rows = yield Zotero.DB.queryAsync(sql, params, { noCache: true });

    // Mark creator indexes above the number of creators as changed,
    // so that they're cleared if the item is saved
    var fixIncorrectIndexes = function (item, numCreators, maxOrderIndex) {
      Zotero.debug("Fixing incorrect creator indexes for item " + item.libraryKey +
      " (" + numCreators + ", " + maxOrderIndex + ")", 2);
      var i = numCreators;
      if (!item._changed.creators) {
        item._changed.creators = {};
      }
      while (i <= maxOrderIndex) {
        item._changed.creators[i] = true;
        i++;
      }
    };

    var lastItemID;
    var item;
    var index = 0;
    var maxOrderIndex = -1;
    for (let i = 0; i < rows.length; i++) {
      let row = rows[i];
      let itemID = row.itemID;

      if (itemID != lastItemID) {
        if (!this._objectCache[itemID]) {
          throw new Error("Item " + itemID + " not loaded");
        }
        item = this._objectCache[itemID];

        item._creators = [];
        item._creatorIDs = [];
        item._loaded.creators = true;
        item._clearChanged('creators');

        if (!row.creatorID) {
          lastItemID = row.itemID;
          continue;
        }

        if (index <= maxOrderIndex) {
          fixIncorrectIndexes(item, index, maxOrderIndex);
        }

        index = 0;
        maxOrderIndex = -1;
      }

      lastItemID = row.itemID;

      if (row.orderIndex > maxOrderIndex) {
        maxOrderIndex = row.orderIndex;
      }

      let creatorData = Zotero.Creators.get(row.creatorID);
      creatorData.creatorTypeID = row.creatorTypeID;
      item._creators[index] = creatorData;
      item._creatorIDs[index] = row.creatorID;
      index++;
    }

    if (index <= maxOrderIndex) {
      fixIncorrectIndexes(item, index, maxOrderIndex);
    }
  });


  this._loadNotes = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var notesToUpdate = [];

    var sql = "SELECT itemID, note FROM items " +
    "JOIN itemNotes USING (itemID) " +
    "WHERE libraryID=?" + idSQL;
    var params = [libraryID];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);
        let item = this._objectCache[itemID];
        if (!item) {
          throw new Error("Item " + itemID + " not found");
        }
        let note = row.getResultByIndex(1);

        // Convert non-HTML notes on-the-fly
        if (note !== "") {
          if (typeof note == 'number') {
            note = '' + note;
          }
          if (typeof note == 'string') {
            if (!note.substr(0, 36).match(/^<div class="zotero-note znv[0-9]+">/)) {
              note = Zotero.Utilities.htmlSpecialChars(note);
              note = Zotero.Notes.notePrefix + '<p>' +
              note.replace(/\n/g, '</p><p>').
              replace(/\t/g, '&nbsp;&nbsp;&nbsp;&nbsp;').
              replace(/  /g, '&nbsp;&nbsp;') +
              '</p>' + Zotero.Notes.noteSuffix;
              note = note.replace(/<p>\s*<\/p>/g, '<p>&nbsp;</p>');
              notesToUpdate.push([item.id, note]);
            }

            // Don't include <div> wrapper when returning value
            let startLen = note.substr(0, 36).match(/^<div class="zotero-note znv[0-9]+">/)[0].length;
            let endLen = 6; // "</div>".length
            note = note.substr(startLen, note.length - startLen - endLen);
          }
          // Clear null notes
          else {
            note = '';
            notesToUpdate.push([item.id, '']);
          }
        }

        item._noteText = note ? note : '';
        item._loaded.note = true;
        item._clearChanged('note');
      }.bind(this) });



    if (notesToUpdate.length) {
      yield Zotero.DB.executeTransaction(function* () {
        for (let i = 0; i < notesToUpdate.length; i++) {
          let row = notesToUpdate[i];
          let sql = "UPDATE itemNotes SET note=? WHERE itemID=?";
          yield Zotero.DB.queryAsync(sql, [row[1], row[0]]);
        }
      }.bind(this));
    }

    // Mark notes and attachments without notes as loaded
    sql = "SELECT itemID FROM items WHERE libraryID=?" + idSQL +
    " AND itemTypeID IN (?, ?) AND itemID NOT IN (SELECT itemID FROM itemNotes)";
    params = [libraryID, Zotero.ItemTypes.getID('note'), Zotero.ItemTypes.getID('attachment')];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);
        let item = this._objectCache[itemID];
        if (!item) {
          throw new Error("Item " + itemID + " not loaded");
        }

        item._noteText = '';
        item._loaded.note = true;
        item._clearChanged('note');
      }.bind(this) });


  });


  this._loadAnnotations = async function (libraryID, ids, idSQL) {
    var sql = "SELECT itemID, IA.parentItemID, IA.type, IA.authorName, IA.text, IA.comment, " +
    "IA.color, IA.sortIndex, IA.isExternal " +
    "FROM items JOIN itemAnnotations IA USING (itemID) " +
    "WHERE libraryID=?" + idSQL;
    var params = [libraryID];
    await Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);

        let item = this._objectCache[itemID];
        if (!item) {
          throw new Error("Item " + itemID + " not found");
        }

        item._parentItemID = row.getResultByIndex(1);
        var typeID = row.getResultByIndex(2);
        var type;
        switch (typeID) {
          case Zotero.Annotations.ANNOTATION_TYPE_HIGHLIGHT:
            type = 'highlight';
            break;

          case Zotero.Annotations.ANNOTATION_TYPE_NOTE:
            type = 'note';
            break;

          case Zotero.Annotations.ANNOTATION_TYPE_IMAGE:
            type = 'image';
            break;

          case Zotero.Annotations.ANNOTATION_TYPE_INK:
            type = 'ink';
            break;

          default:
            throw new Error(`Unknown annotation type id ${typeID}`);}

        item._annotationType = type;
        item._annotationAuthorName = row.getResultByIndex(3);
        item._annotationText = row.getResultByIndex(4);
        item._annotationComment = row.getResultByIndex(5);
        item._annotationColor = row.getResultByIndex(6);
        item._annotationSortIndex = row.getResultByIndex(7);
        item._annotationIsExternal = !!row.getResultByIndex(8);

        item._loaded.annotation = true;
        item._clearChanged('annotation');
      }.bind(this) });


  };


  this._loadAnnotationsDeferred = async function (libraryID, ids, idSQL) {
    var sql = "SELECT itemID, IA.position, IA.pageLabel FROM items " +
    "JOIN itemAnnotations IA USING (itemID) " +
    "WHERE libraryID=?" + idSQL;
    var params = [libraryID];
    await Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);

        let item = this._objectCache[itemID];
        if (!item) {
          throw new Error("Item " + itemID + " not found");
        }

        item._annotationPosition = row.getResultByIndex(1);
        item._annotationPageLabel = row.getResultByIndex(2);

        item._loaded.annotationDeferred = true;
        item._clearChanged('annotationDeferred');
      }.bind(this) });


  };


  this._loadChildItems = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var params = [libraryID];
    var rows = [];
    var onRow = function (row, setFunc) {
      var itemID = row.getResultByIndex(0);

      // If we've finished a set of rows for an item, process them
      if (lastItemID && itemID !== lastItemID) {
        setFunc(lastItemID, rows);
        rows = [];
      }

      lastItemID = itemID;
      rows.push({
        itemID: row.getResultByIndex(1),
        title: row.getResultByIndex(2),
        trashed: row.getResultByIndex(3) });

    };

    //
    // Attachments
    //
    var titleFieldID = Zotero.ItemFields.getID('title');
    var sql = "SELECT parentItemID, A.itemID, value AS title, " +
    "CASE WHEN DI.itemID IS NULL THEN 0 ELSE 1 END AS trashed " +
    "FROM itemAttachments A " +
    "JOIN items I ON (A.parentItemID=I.itemID) " +
    `LEFT JOIN itemData ID ON (fieldID=${titleFieldID} AND A.itemID=ID.itemID) ` +
    "LEFT JOIN itemDataValues IDV USING (valueID) " +
    "LEFT JOIN deletedItems DI USING (itemID) " +
    "WHERE libraryID=?" + (
    ids.length ? " AND parentItemID IN (" + ids.map((id) => parseInt(id)).join(", ") + ")" : "") +
    " ORDER BY parentItemID";
    // Since we do the sort here and cache these results, a restart will be required
    // if this pref (off by default) is turned on, but that's OK
    if (Zotero.Prefs.get('sortAttachmentsChronologically')) {
      sql += ", dateAdded";
    }
    var setAttachmentItem = function (itemID, rows) {
      var item = this._objectCache[itemID];
      if (!item) {
        throw new Error("Item " + itemID + " not loaded");
      }

      item._attachments = {
        rows,
        chronologicalWithTrashed: null,
        chronologicalWithoutTrashed: null,
        alphabeticalWithTrashed: null,
        alphabeticalWithoutTrashed: null };

    }.bind(this);
    var lastItemID = null;
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        onRow(row, setAttachmentItem);
      } });


    // Process unprocessed rows
    if (lastItemID) {
      setAttachmentItem(lastItemID, rows);
    }
    // Otherwise clear existing entries for passed items
    else if (ids.length) {
      ids.forEach((id) => setAttachmentItem(id, []));
    }

    //
    // Notes
    //
    sql = "SELECT parentItemID, N.itemID, title, " +
    "CASE WHEN DI.itemID IS NULL THEN 0 ELSE 1 END AS trashed " +
    "FROM itemNotes N " +
    "JOIN items I ON (N.parentItemID=I.itemID) " +
    "LEFT JOIN deletedItems DI USING (itemID) " +
    "WHERE libraryID=?" + (
    ids.length ? " AND parentItemID IN (" + ids.map((id) => parseInt(id)).join(", ") + ")" : "") +
    " ORDER BY parentItemID";
    if (Zotero.Prefs.get('sortNotesChronologically')) {
      sql += ", dateAdded";
    }
    var setNoteItem = function (itemID, rows) {
      var item = this._objectCache[itemID];
      if (!item) {
        throw new Error("Item " + itemID + " not loaded");
      }

      item._notes = {
        rows,
        rowsEmbedded: null,
        chronologicalWithTrashed: null,
        chronologicalWithoutTrashed: null,
        alphabeticalWithTrashed: null,
        alphabeticalWithoutTrashed: null,
        numWithTrashed: null,
        numWithoutTrashed: null,
        numWithTrashedWithEmbedded: null,
        numWithoutTrashedWithoutEmbedded: null };

    }.bind(this);
    lastItemID = null;
    rows = [];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        onRow(row, setNoteItem);
      } });


    // Process unprocessed rows
    if (lastItemID) {
      setNoteItem(lastItemID, rows);
    }
    // Otherwise clear existing entries for passed items
    else if (ids.length) {
      ids.forEach((id) => setNoteItem(id, []));
    }

    //
    // Annotations
    //
    sql = "SELECT parentItemID, IAn.itemID, " +
    "text || ' - ' || comment AS title, " // TODO: Make better
    + "CASE WHEN DI.itemID IS NULL THEN 0 ELSE 1 END AS trashed " +
    "FROM itemAnnotations IAn " +
    "JOIN items I ON (IAn.parentItemID=I.itemID) " +
    "LEFT JOIN deletedItems DI USING (itemID) " +
    "WHERE libraryID=?" + (
    ids.length ? " AND parentItemID IN (" + ids.map((id) => parseInt(id)).join(", ") + ")" : "") +
    " ORDER BY parentItemID, sortIndex";
    var setAnnotationItem = function (itemID, rows) {
      var item = this._objectCache[itemID];
      if (!item) {
        throw new Error("Item " + itemID + " not loaded");
      }
      rows.sort((a, b) => a.sortIndex - b.sortIndex);
      item._annotations = {
        rows,
        withTrashed: null,
        withoutTrashed: null };

    }.bind(this);
    lastItemID = null;
    rows = [];
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        onRow(row, setAnnotationItem);
      } });


    // Process unprocessed rows
    if (lastItemID) {
      setAnnotationItem(lastItemID, rows);
    }
    // Otherwise clear existing entries for passed items
    else if (ids.length) {
      ids.forEach((id) => setAnnotationItem(id, []));
    }

    // Mark either all passed items or all items as having child items loaded
    sql = "SELECT itemID FROM items I WHERE libraryID=?";
    if (idSQL) {
      sql += idSQL;
    }
    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        var itemID = row.getResultByIndex(0);
        var item = this._objectCache[itemID];
        if (!item) {
          throw new Error("Item " + itemID + " not loaded");
        }
        item._loaded.childItems = true;
        item._clearChanged('childItems');
      }.bind(this) });


  });


  this._loadTags = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var sql = "SELECT itemID, name, type FROM items " +
    "LEFT JOIN itemTags USING (itemID) " +
    "LEFT JOIN tags USING (tagID) WHERE libraryID=?" + idSQL;
    var params = [libraryID];

    var lastItemID;
    var rows = [];
    var setRows = function (itemID, rows) {
      var item = this._objectCache[itemID];
      if (!item) {
        throw new Error("Item " + itemID + " not found");
      }

      item._tags = [];
      for (let i = 0; i < rows.length; i++) {
        let row = rows[i];
        item._tags.push(Zotero.Tags.cleanData(row));
      }

      item._loaded.tags = true;
    }.bind(this);

    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);

        if (lastItemID && itemID !== lastItemID) {
          setRows(lastItemID, rows);
          rows = [];
        }

        lastItemID = itemID;

        // Item has no tags
        let tag = row.getResultByIndex(1);
        if (tag === null) {
          return;
        }

        rows.push({
          tag: tag,
          type: row.getResultByIndex(2) });

      }.bind(this) });


    if (lastItemID) {
      setRows(lastItemID, rows);
    }
  });


  this._loadCollections = Zotero.Promise.coroutine(function* (libraryID, ids, idSQL) {
    var sql = "SELECT itemID, collectionID FROM items " +
    "LEFT JOIN collectionItems USING (itemID) " +
    "WHERE libraryID=?" + idSQL;
    var params = [libraryID];

    var lastItemID;
    var rows = [];
    var setRows = function (itemID, rows) {
      var item = this._objectCache[itemID];
      if (!item) {
        throw new Error("Item " + itemID + " not found");
      }

      item._collections = rows;
      item._loaded.collections = true;
      item._clearChanged('collections');
    }.bind(this);

    yield Zotero.DB.queryAsync(
    sql,
    params,
    {
      noCache: true,
      onRow: function (row) {
        let itemID = row.getResultByIndex(0);

        if (lastItemID && itemID !== lastItemID) {
          setRows(lastItemID, rows);
          rows = [];
        }

        lastItemID = itemID;
        let collectionID = row.getResultByIndex(1);
        // No collections
        if (collectionID === null) {
          return;
        }
        rows.push(collectionID);
      }.bind(this) });


    if (lastItemID) {
      setRows(lastItemID, rows);
    }
  });


  /**
   * Copy child items from one item to another (e.g., in another library)
   *
   * Requires a transaction
   */
  this.copyChildItems = async function (fromItem, toItem) {
    Zotero.DB.requireTransaction();

    var fromGroup = fromItem.library.isGroup;

    // Annotations on files
    if (fromItem.isFileAttachment()) {
      let annotations = fromItem.getAnnotations();
      for (let annotation of annotations) {
        // Don't copy embedded PDF annotations
        if (annotation.annotationIsExternal) {
          continue;
        }
        let newAnnotation = annotation.clone(toItem.libraryID);
        newAnnotation.parentItemID = toItem.id;
        // If there's no explicit author and we're copying an annotation created by another
        // user from a group, set the author to the creating user
        if (fromGroup &&
        !annotation.annotationAuthorName &&
        annotation.createdByUserID != Zotero.Users.getCurrentUserID()) {
          newAnnotation.annotationAuthorName =
          Zotero.Users.getName(annotation.createdByUserID);
        }
        await newAnnotation.save();
      }
    }

    // TODO: Other things as necessary
  };


  /**
   * Move child items from one item to another
   *
   * Requires a transaction
   *
   * @param {Zotero.Item} fromItem
   * @param {Zotero.Item} toItem
   * @param {Boolean} includeTrashed
   * @return {Promise}
   */
  this.moveChildItems = async function (fromItem, toItem, includeTrashed = false) {
    Zotero.DB.requireTransaction();

    // Annotations on files
    if (fromItem.isFileAttachment()) {
      let annotations = fromItem.getAnnotations(includeTrashed);
      for (let annotation of annotations) {
        annotation.parentItemID = toItem.id;
        await annotation.save();
      }
    }

    // TODO: Other things as necessary
  };


  this.merge = function (item, otherItems) {
    Zotero.debug("Merging items");

    return Zotero.DB.executeTransaction(function* () {
      var replPred = Zotero.Relations.replacedItemPredicate;
      var toSave = {};
      toSave[item.id] = item;

      var earliestDateAdded = item.dateAdded;

      let remapAttachmentKeys = yield this._mergePDFAttachments(item, otherItems);
      yield this._mergeWebAttachments(item, otherItems);
      yield this._mergeOtherAttachments(item, otherItems);

      for (let otherItem of otherItems) {
        if (otherItem.libraryID !== item.libraryID) {
          throw new Error('Items being merged must be in the same library');
        }

        // Use the earliest date added of all the items
        if (otherItem.dateAdded < earliestDateAdded) {
          earliestDateAdded = otherItem.dateAdded;
        }

        // Move notes to master
        var noteIDs = otherItem.getNotes(true);
        for (let id of noteIDs) {
          var note = yield this.getAsync(id);
          note.parentItemID = item.id;
          Zotero.Notes.replaceItemKey(note, otherItem.key, item.key);
          Zotero.Notes.replaceAllItemKeys(note, remapAttachmentKeys);
          toSave[note.id] = note;
        }

        // Move relations to master
        yield this._moveRelations(otherItem, item);

        // All other operations are additive only and do not affect the
        // old item, which will be put in the trash

        // Add collections to master
        otherItem.getCollections().forEach((id) => item.addToCollection(id));

        // Add tags to master
        var tags = otherItem.getTags();
        for (let j = 0; j < tags.length; j++) {
          let tagName = tags[j].tag;
          if (item.hasTag(tagName)) {
            let type = item.getTagType(tagName);
            // If existing manual tag, leave that
            if (type == 0) {
              continue;
            }
            // Otherwise, add the non-master item's tag, which may be manual, in which
            // case it will remain at the end
            item.addTag(tagName, tags[j].type);
          }
          // If no existing tag, add with the type from the non-master item
          else {
            item.addTag(tagName, tags[j].type);
          }
        }

        // Trash other item
        otherItem.deleted = true;
        toSave[otherItem.id] = otherItem;
      }

      item.setField('dateAdded', earliestDateAdded);

      // Hack to remove master item from duplicates view without recalculating duplicates
      // Pass force = true so observers will be notified before this transaction is committed
      yield Zotero.Notifier.trigger('removeDuplicatesMaster', 'item', item.id, null, true);

      for (let i in toSave) {
        yield toSave[i].save();
      }
    }.bind(this));
  };


  this._mergePDFAttachments = async function (item, otherItems) {
    Zotero.DB.requireTransaction();

    let remapAttachmentKeys = new Map();
    let masterAttachmentHashes = await this._hashItem(item, 'bytes');
    let hashesIncludeText = false;

    for (let otherItem of otherItems) {
      let mergedMasterAttachments = new Set();

      let doMerge = async (fromAttachment, toAttachment) => {
        mergedMasterAttachments.add(toAttachment.id);

        await this.moveChildItems(fromAttachment, toAttachment, true);
        await this._moveEmbeddedNote(fromAttachment, toAttachment);
        await this._moveRelations(fromAttachment, toAttachment);

        fromAttachment.deleted = true;
        await fromAttachment.save();

        // Later on, when processing notes, we'll use this to remap
        // URLs pointing to the old attachment.
        remapAttachmentKeys.set(fromAttachment.key, toAttachment.key);

        // Items can only have one replaced item predicate
        if (!toAttachment.getRelationsByPredicate(Zotero.Relations.replacedItemPredicate)) {
          toAttachment.addRelation(Zotero.Relations.replacedItemPredicate,
          Zotero.URI.getItemURI(fromAttachment));
        }

        await toAttachment.save();
      };

      for (let otherAttachment of await this.getAsync(otherItem.getAttachments(true))) {
        if (!otherAttachment.isPDFAttachment()) {
          continue;
        }

        // First check if master has an attachment with identical MD5 hash
        let matchingHash = await otherAttachment.attachmentHash;
        let masterAttachmentID = masterAttachmentHashes.get(matchingHash);

        if (!masterAttachmentID && item.numAttachments()) {
          // If that didn't work, hash master attachments by the
          // most common words in their text and check again.
          if (!hashesIncludeText) {
            masterAttachmentHashes = new Map([
            ...masterAttachmentHashes,
            ...(await this._hashItem(item, 'text'))]);

            hashesIncludeText = true;
          }

          matchingHash = await this._hashAttachmentText(otherAttachment);
          masterAttachmentID = masterAttachmentHashes.get(matchingHash);
        }

        if (!masterAttachmentID || mergedMasterAttachments.has(masterAttachmentID)) {
          Zotero.debug(`No unmerged match for attachment ${otherAttachment.key} in master item - moving`);
          otherAttachment.parentItemID = item.id;
          await otherAttachment.save();
          continue;
        }

        let masterAttachment = await this.getAsync(masterAttachmentID);

        if (masterAttachment.attachmentContentType !== otherAttachment.attachmentContentType) {
          Zotero.debug(`Master attachment ${masterAttachment.key} matches ${otherAttachment.key}, ` +
          'but content types differ - keeping both');
          otherAttachment.parentItemID = item.id;
          await otherAttachment.save();
          continue;
        }

        if (!(masterAttachment.isImportedAttachment() && otherAttachment.isImportedAttachment() ||
        masterAttachment.isLinkedFileAttachment() && otherAttachment.isLinkedFileAttachment())) {
          Zotero.debug(`Master attachment ${masterAttachment.key} matches ${otherAttachment.key}, ` +
          'but link modes differ - keeping both');
          otherAttachment.parentItemID = item.id;
          await otherAttachment.save();
          continue;
        }

        // Check whether master and other have embedded annotations
        // Master yes, other yes -> keep both
        // Master yes, other no -> keep master
        // Master no, other yes -> keep other
        if (await otherAttachment.hasEmbeddedAnnotations()) {
          if (await masterAttachment.hasEmbeddedAnnotations()) {
            Zotero.debug(`Master attachment ${masterAttachment.key} matches ${otherAttachment.key}, ` +
            'but both have embedded annotations - keeping both');
            otherAttachment.parentItemID = item.id;
            await otherAttachment.save();
          } else
          {
            Zotero.debug(`Master attachment ${masterAttachment.key} matches ${otherAttachment.key}, ` +
            'but other has embedded annotations - merging into other');
            await doMerge(masterAttachment, otherAttachment);
            otherAttachment.parentItemID = item.id;
            await otherAttachment.save();
          }
          continue;
        }

        Zotero.debug(`Master attachment ${masterAttachment.key} matches ${otherAttachment.key} - merging into master`);
        await doMerge(otherAttachment, masterAttachment);
      }
    }

    return remapAttachmentKeys;
  };


  this._mergeWebAttachments = async function (item, otherItems) {
    Zotero.DB.requireTransaction();

    let masterAttachments = (await this.getAsync(item.getAttachments(true))).
    filter((attachment) => attachment.isWebAttachment());

    for (let otherItem of otherItems) {
      for (let otherAttachment of await this.getAsync(otherItem.getAttachments(true))) {
        if (!otherAttachment.isWebAttachment()) {
          continue;
        }

        // If we can find an attachment with the same title *and* URL, use it.
        let masterAttachment =
        masterAttachments.find((attachment) => attachment.getField('title') == otherAttachment.getField('title') &&
        attachment.getField('url') == otherAttachment.getField('url') &&
        attachment.attachmentLinkMode === otherAttachment.attachmentLinkMode) ||
        masterAttachments.find((attachment) => attachment.getField('title') == otherAttachment.getField('title') &&
        attachment.attachmentLinkMode === otherAttachment.attachmentLinkMode);


        if (!masterAttachment) {
          Zotero.debug(`No match for web attachment ${otherAttachment.key} in master item - moving`);
          otherAttachment.parentItemID = item.id;
          await otherAttachment.save();
          continue;
        }

        otherAttachment.deleted = true;
        await this._moveRelations(otherAttachment, masterAttachment);
        await otherAttachment.save();

        masterAttachment.addRelation(Zotero.Relations.replacedItemPredicate,
        Zotero.URI.getItemURI(otherAttachment));
        await masterAttachment.save();

        // Don't match with this attachment again
        masterAttachments = masterAttachments.filter((a) => a !== masterAttachment);
      }
    }
  };


  this._mergeOtherAttachments = async function (item, otherItems) {
    Zotero.DB.requireTransaction();

    for (let otherItem of otherItems) {
      for (let otherAttachment of await this.getAsync(otherItem.getAttachments(true))) {
        if (otherAttachment.isPDFAttachment() || otherAttachment.isWebAttachment()) {
          continue;
        }

        otherAttachment.parentItemID = item.id;
        await otherAttachment.save();
      }
    }
  };


  /**
   * Hash each attachment of the provided item. Return a map from hashes to
   * attachment IDs.
   *
   * @param {Zotero.Item} item
   * @param {String} hashType 'bytes' or 'text'
   * @return {Promise<Map<String, String>>}
   */
  this._hashItem = async function (item, hashType) {
    if (!['bytes', 'text'].includes(hashType)) {
      throw new Error('Invalid hash type');
    }

    let attachments = (await this.getAsync(item.getAttachments())).
    filter((attachment) => attachment.isFileAttachment());
    let hashes = new Map();
    await Promise.all(attachments.map(async (attachment) => {
      let hash = hashType === 'bytes' ?
      await attachment.attachmentHash :
      await this._hashAttachmentText(attachment);
      if (hash) {
        hashes.set(hash, attachment.id);
      }
    }));
    return hashes;
  };


  /**
   * Hash an attachment by the most common words in its text.
   * @param {Zotero.Item} attachment
   * @return {Promise<String>}
   */
  this._hashAttachmentText = async function (attachment) {
    var fileInfo;
    try {
      fileInfo = await OS.File.stat(attachment.getFilePath());
    }
    catch (e) {
      if (e instanceof OS.File.Error && e.becauseNoSuchFile) {
        Zotero.debug('_hashAttachmentText: Attachment not found');
        return null;
      }
      Zotero.logError(e);
      return null;
    }
    if (fileInfo.size > 5e8) {
      Zotero.debug('_hashAttachmentText: Attachment too large');
      return null;
    }

    let text;
    try {
      text = await attachment.attachmentText;
    }
    catch (e) {
      Zotero.logError(e);
    }
    if (!text) {
      Zotero.debug('_hashAttachmentText: Attachment has no text');
      return null;
    }

    let mostCommonWords = this._getMostCommonWords(text, 50);
    if (mostCommonWords.length < 10) {
      Zotero.debug('_hashAttachmentText: Not enough unique words');
      return null;
    }
    return Zotero.Utilities.Internal.md5(mostCommonWords.sort().join(' '));
  };


  /**
   * Get the n most common words in s in descending order of frequency.
   * If s contains fewer than n unique words, the size of the returned array
   * will be less than n.
   *
   * @param {String} s
   * @param {Number} n
   * @return {String[]}
   */
  this._getMostCommonWords = function (s, n) {
    // Use an iterative approach for better performance.

    const whitespaceRe = /\s/;
    const wordCharRe = /[A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u037F\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u052F\u0531-\u0556\u0559\u0560-\u0588\u05D0-\u05EA\u05EF-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u0860-\u086A\u0870-\u0887\u0889-\u088E\u08A0-\u08C9\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0980\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u09FC\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0AF9\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C39\u0C3D\u0C58-\u0C5A\u0C5D\u0C60\u0C61\u0C80\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D04-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D54-\u0D56\u0D5F-\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E86-\u0E8A\u0E8C-\u0EA3\u0EA5\u0EA7-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F5\u13F8-\u13FD\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16F1-\u16F8\u1700-\u1711\u171F-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1878\u1880-\u1884\u1887-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191E\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4C\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1C80-\u1C88\u1C90-\u1CBA\u1CBD-\u1CBF\u1CE9-\u1CEC\u1CEE-\u1CF3\u1CF5\u1CF6\u1CFA\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312F\u3131-\u318E\u31A0-\u31BF\u31F0-\u31FF\u3400-\u4DBF\u4E00-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA69D\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B-\uA7CA\uA7D0\uA7D1\uA7D3\uA7D5-\uA7D9\uA7F2-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA8FD\uA8FE\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uA9E0-\uA9E4\uA9E6-\uA9EF\uA9FA-\uA9FE\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA7E-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uAB30-\uAB5A\uAB5C-\uAB69\uAB70-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC\u{10000}-\u{1000B}\u{1000D}-\u{10026}\u{10028}-\u{1003A}\u{1003C}\u{1003D}\u{1003F}-\u{1004D}\u{10050}-\u{1005D}\u{10080}-\u{100FA}\u{10280}-\u{1029C}\u{102A0}-\u{102D0}\u{10300}-\u{1031F}\u{1032D}-\u{10340}\u{10342}-\u{10349}\u{10350}-\u{10375}\u{10380}-\u{1039D}\u{103A0}-\u{103C3}\u{103C8}-\u{103CF}\u{10400}-\u{1049D}\u{104B0}-\u{104D3}\u{104D8}-\u{104FB}\u{10500}-\u{10527}\u{10530}-\u{10563}\u{10570}-\u{1057A}\u{1057C}-\u{1058A}\u{1058C}-\u{10592}\u{10594}\u{10595}\u{10597}-\u{105A1}\u{105A3}-\u{105B1}\u{105B3}-\u{105B9}\u{105BB}\u{105BC}\u{10600}-\u{10736}\u{10740}-\u{10755}\u{10760}-\u{10767}\u{10780}-\u{10785}\u{10787}-\u{107B0}\u{107B2}-\u{107BA}\u{10800}-\u{10805}\u{10808}\u{1080A}-\u{10835}\u{10837}\u{10838}\u{1083C}\u{1083F}-\u{10855}\u{10860}-\u{10876}\u{10880}-\u{1089E}\u{108E0}-\u{108F2}\u{108F4}\u{108F5}\u{10900}-\u{10915}\u{10920}-\u{10939}\u{10980}-\u{109B7}\u{109BE}\u{109BF}\u{10A00}\u{10A10}-\u{10A13}\u{10A15}-\u{10A17}\u{10A19}-\u{10A35}\u{10A60}-\u{10A7C}\u{10A80}-\u{10A9C}\u{10AC0}-\u{10AC7}\u{10AC9}-\u{10AE4}\u{10B00}-\u{10B35}\u{10B40}-\u{10B55}\u{10B60}-\u{10B72}\u{10B80}-\u{10B91}\u{10C00}-\u{10C48}\u{10C80}-\u{10CB2}\u{10CC0}-\u{10CF2}\u{10D00}-\u{10D23}\u{10E80}-\u{10EA9}\u{10EB0}\u{10EB1}\u{10F00}-\u{10F1C}\u{10F27}\u{10F30}-\u{10F45}\u{10F70}-\u{10F81}\u{10FB0}-\u{10FC4}\u{10FE0}-\u{10FF6}\u{11003}-\u{11037}\u{11071}\u{11072}\u{11075}\u{11083}-\u{110AF}\u{110D0}-\u{110E8}\u{11103}-\u{11126}\u{11144}\u{11147}\u{11150}-\u{11172}\u{11176}\u{11183}-\u{111B2}\u{111C1}-\u{111C4}\u{111DA}\u{111DC}\u{11200}-\u{11211}\u{11213}-\u{1122B}\u{11280}-\u{11286}\u{11288}\u{1128A}-\u{1128D}\u{1128F}-\u{1129D}\u{1129F}-\u{112A8}\u{112B0}-\u{112DE}\u{11305}-\u{1130C}\u{1130F}\u{11310}\u{11313}-\u{11328}\u{1132A}-\u{11330}\u{11332}\u{11333}\u{11335}-\u{11339}\u{1133D}\u{11350}\u{1135D}-\u{11361}\u{11400}-\u{11434}\u{11447}-\u{1144A}\u{1145F}-\u{11461}\u{11480}-\u{114AF}\u{114C4}\u{114C5}\u{114C7}\u{11580}-\u{115AE}\u{115D8}-\u{115DB}\u{11600}-\u{1162F}\u{11644}\u{11680}-\u{116AA}\u{116B8}\u{11700}-\u{1171A}\u{11740}-\u{11746}\u{11800}-\u{1182B}\u{118A0}-\u{118DF}\u{118FF}-\u{11906}\u{11909}\u{1190C}-\u{11913}\u{11915}\u{11916}\u{11918}-\u{1192F}\u{1193F}\u{11941}\u{119A0}-\u{119A7}\u{119AA}-\u{119D0}\u{119E1}\u{119E3}\u{11A00}\u{11A0B}-\u{11A32}\u{11A3A}\u{11A50}\u{11A5C}-\u{11A89}\u{11A9D}\u{11AB0}-\u{11AF8}\u{11C00}-\u{11C08}\u{11C0A}-\u{11C2E}\u{11C40}\u{11C72}-\u{11C8F}\u{11D00}-\u{11D06}\u{11D08}\u{11D09}\u{11D0B}-\u{11D30}\u{11D46}\u{11D60}-\u{11D65}\u{11D67}\u{11D68}\u{11D6A}-\u{11D89}\u{11D98}\u{11EE0}-\u{11EF2}\u{11FB0}\u{12000}-\u{12399}\u{12480}-\u{12543}\u{12F90}-\u{12FF0}\u{13000}-\u{1342E}\u{14400}-\u{14646}\u{16800}-\u{16A38}\u{16A40}-\u{16A5E}\u{16A70}-\u{16ABE}\u{16AD0}-\u{16AED}\u{16B00}-\u{16B2F}\u{16B40}-\u{16B43}\u{16B63}-\u{16B77}\u{16B7D}-\u{16B8F}\u{16E40}-\u{16E7F}\u{16F00}-\u{16F4A}\u{16F50}\u{16F93}-\u{16F9F}\u{16FE0}\u{16FE1}\u{16FE3}\u{17000}-\u{187F7}\u{18800}-\u{18CD5}\u{18D00}-\u{18D08}\u{1AFF0}-\u{1AFF3}\u{1AFF5}-\u{1AFFB}\u{1AFFD}\u{1AFFE}\u{1B000}-\u{1B122}\u{1B150}-\u{1B152}\u{1B164}-\u{1B167}\u{1B170}-\u{1B2FB}\u{1BC00}-\u{1BC6A}\u{1BC70}-\u{1BC7C}\u{1BC80}-\u{1BC88}\u{1BC90}-\u{1BC99}\u{1D400}-\u{1D454}\u{1D456}-\u{1D49C}\u{1D49E}\u{1D49F}\u{1D4A2}\u{1D4A5}\u{1D4A6}\u{1D4A9}-\u{1D4AC}\u{1D4AE}-\u{1D4B9}\u{1D4BB}\u{1D4BD}-\u{1D4C3}\u{1D4C5}-\u{1D505}\u{1D507}-\u{1D50A}\u{1D50D}-\u{1D514}\u{1D516}-\u{1D51C}\u{1D51E}-\u{1D539}\u{1D53B}-\u{1D53E}\u{1D540}-\u{1D544}\u{1D546}\u{1D54A}-\u{1D550}\u{1D552}-\u{1D6A5}\u{1D6A8}-\u{1D6C0}\u{1D6C2}-\u{1D6DA}\u{1D6DC}-\u{1D6FA}\u{1D6FC}-\u{1D714}\u{1D716}-\u{1D734}\u{1D736}-\u{1D74E}\u{1D750}-\u{1D76E}\u{1D770}-\u{1D788}\u{1D78A}-\u{1D7A8}\u{1D7AA}-\u{1D7C2}\u{1D7C4}-\u{1D7CB}\u{1DF00}-\u{1DF1E}\u{1E100}-\u{1E12C}\u{1E137}-\u{1E13D}\u{1E14E}\u{1E290}-\u{1E2AD}\u{1E2C0}-\u{1E2EB}\u{1E7E0}-\u{1E7E6}\u{1E7E8}-\u{1E7EB}\u{1E7ED}\u{1E7EE}\u{1E7F0}-\u{1E7FE}\u{1E800}-\u{1E8C4}\u{1E900}-\u{1E943}\u{1E94B}\u{1EE00}-\u{1EE03}\u{1EE05}-\u{1EE1F}\u{1EE21}\u{1EE22}\u{1EE24}\u{1EE27}\u{1EE29}-\u{1EE32}\u{1EE34}-\u{1EE37}\u{1EE39}\u{1EE3B}\u{1EE42}\u{1EE47}\u{1EE49}\u{1EE4B}\u{1EE4D}-\u{1EE4F}\u{1EE51}\u{1EE52}\u{1EE54}\u{1EE57}\u{1EE59}\u{1EE5B}\u{1EE5D}\u{1EE5F}\u{1EE61}\u{1EE62}\u{1EE64}\u{1EE67}-\u{1EE6A}\u{1EE6C}-\u{1EE72}\u{1EE74}-\u{1EE77}\u{1EE79}-\u{1EE7C}\u{1EE7E}\u{1EE80}-\u{1EE89}\u{1EE8B}-\u{1EE9B}\u{1EEA1}-\u{1EEA3}\u{1EEA5}-\u{1EEA9}\u{1EEAB}-\u{1EEBB}\u{20000}-\u{2A6DF}\u{2A700}-\u{2B738}\u{2B740}-\u{2B81D}\u{2B820}-\u{2CEA1}\u{2CEB0}-\u{2EBE0}\u{2F800}-\u{2FA1D}\u{30000}-\u{3134A}]/u; // [a-z] only matches Latin

    let freqs = new Map();
    let currentWord = '';

    for (let codePoint of s) {
      if (whitespaceRe.test(codePoint)) {
        if (currentWord.length > 3) {
          freqs.set(currentWord, (freqs.get(currentWord) || 0) + 1);
        }

        currentWord = '';
        continue;
      }

      if (wordCharRe.test(codePoint)) {
        currentWord += codePoint.toLowerCase();
      }
    }

    // Break ties in locale order.
    return [...freqs.keys()].
    sort((a, b) => freqs.get(b) - freqs.get(a) || Zotero.localeCompare(a, b)).
    slice(0, n);
  };

  /**
   * Move fromItem's embedded note, if it has one, to toItem.
   * If toItem already has an embedded note, the note will be added as a new
   * child note item on toItem's parent.
   * Requires a transaction.
   */
  this._moveEmbeddedNote = async function (fromItem, toItem) {
    Zotero.DB.requireTransaction();

    if (fromItem.getNote()) {
      let noteItem = toItem;
      if (toItem.getNote()) {
        noteItem = new Zotero.Item('note');
        noteItem.parentItemID = toItem.parentItemID;
      }
      noteItem.setNote(fromItem.getNote());
      fromItem.setNote('');
      Zotero.Notes.replaceItemKey(noteItem, fromItem.key, toItem.key);
      await noteItem.save();
    }
  };


  /**
   * Move fromItem's relations to toItem as part of a merge.
   * Requires a transaction.
   *
   * @param {Zotero.Item} fromItem
   * @param {Zotero.Item} toItem
   * @return {Promise}
   */
  this._moveRelations = async function (fromItem, toItem) {
    Zotero.DB.requireTransaction();

    let replPred = Zotero.Relations.replacedItemPredicate;
    let fromURI = Zotero.URI.getItemURI(fromItem);
    let toURI = Zotero.URI.getItemURI(toItem);

    // Add relations to toItem
    let oldRelations = fromItem.getRelations();
    for (let pred in oldRelations) {
      oldRelations[pred].forEach((obj) => toItem.addRelation(pred, obj));
    }

    // Remove merge-tracking relations from fromItem, so that there aren't two
    // subjects for a given deleted object
    let replItems = fromItem.getRelationsByPredicate(replPred);
    for (let replItem of replItems) {
      fromItem.removeRelation(replPred, replItem);
    }

    // Update relations on items in the library that point to the other item
    // to point to the master instead
    let rels = await Zotero.Relations.getByObject('item', fromURI);
    for (let rel of rels) {
      // Skip merge-tracking relations, which are dealt with above
      if (rel.predicate == replPred) continue;
      // Skip items in other libraries. They might not be editable, and even
      // if they are, merging items in one library shouldn't affect another library,
      // so those will follow the merge-tracking relations and can optimize their
      // path if they're resaved.
      if (rel.subject.libraryID != toItem.libraryID) continue;
      rel.subject.removeRelation(rel.predicate, fromURI);
      rel.subject.addRelation(rel.predicate, toURI);
      await rel.subject.save();
    }

    // Add relation to track merge
    toItem.addRelation(replPred, fromURI);

    await fromItem.save();
    await toItem.save();
  };


  this.trash = Zotero.Promise.coroutine(function* (ids) {
    Zotero.DB.requireTransaction();

    var libraryIDs = new Set();
    ids = Zotero.flattenArguments(ids);
    var items = [];
    for (let id of ids) {
      let item = this.get(id);
      if (!item) {
        Zotero.debug('Item ' + id + ' does not exist in Items.trash()!', 1);
        Zotero.Notifier.queue('trash', 'item', id);
        continue;
      }

      if (!item.isEditable()) {
        throw new Error(item._ObjectType + " " + item.libraryKey + " is not editable");
      }

      if (!Zotero.Libraries.get(item.libraryID).hasTrash) {
        throw new Error(Zotero.Libraries.getName(item.libraryID) + " does not have a trash");
      }

      items.push(item);
      libraryIDs.add(item.libraryID);
    }

    var parentItemIDs = new Set();
    items.forEach((item) => {
      item.setDeleted(true);
      item.synced = false;
      if (item.parentItemID) {
        parentItemIDs.add(item.parentItemID);
      }
    });
    yield Zotero.Utilities.Internal.forEachChunkAsync(ids, 250, Zotero.Promise.coroutine(function* (chunk) {
      yield Zotero.DB.queryAsync(
      "UPDATE items SET synced=0, clientDateModified=CURRENT_TIMESTAMP " +
      `WHERE itemID IN (${chunk.map((id) => parseInt(id)).join(", ")})`);

      yield Zotero.DB.queryAsync(
      "INSERT OR IGNORE INTO deletedItems (itemID) VALUES " +
      chunk.map((id) => "(" + id + ")").join(", "));

    }.bind(this)));

    // Keep in sync with Zotero.Item::saveData()
    for (let parentItemID of parentItemIDs) {
      let parentItem = yield Zotero.Items.getAsync(parentItemID);
      yield parentItem.reload(['primaryData', 'childItems'], true);
    }
    Zotero.Notifier.queue('modify', 'item', ids);
    Zotero.Notifier.queue('trash', 'item', ids);
    Array.from(libraryIDs).forEach((libraryID) => {
      Zotero.Notifier.queue('refresh', 'trash', libraryID);
    });
  });


  this.trashTx = function (ids) {
    return Zotero.DB.executeTransaction(function* () {
      return this.trash(ids);
    }.bind(this));
  };


  /**
   * @param {Integer} libraryID - Library to delete from
   * @param {Object} [options]
   * @param {Function} [options.onProgress] - fn(progress, progressMax)
   * @param {Integer} [options.days] - Only delete items deleted more than this many days ago
   * @param {Integer} [options.limit] - Number of items to delete
   */
  this.emptyTrash = async function (libraryID, options = {}) {
    if (arguments.length > 2 || typeof arguments[1] == 'number') {
      Zotero.warn("Zotero.Items.emptyTrash() has changed -- update your code");
      options.days = arguments[1];
      options.limit = arguments[2];
    }

    if (!libraryID) {
      throw new Error("Library ID not provided");
    }

    var t = new Date();

    var deleted = await this.getDeleted(libraryID, false, options.days);

    if (options.limit) {
      deleted = deleted.slice(0, options.limit);
    }

    var processed = 0;
    if (deleted.length) {
      let toDelete = {
        top: [],
        child: [] };

      deleted.forEach((item) => {
        item.isTopLevelItem() ? toDelete.top.push(item.id) : toDelete.child.push(item.id);
      });

      // Show progress meter during deletions
      let eraseOptions = options.onProgress ?
      {
        onProgress: function (progress, progressMax) {
          options.onProgress(processed + progress, deleted.length);
        } } :

      undefined;
      for (let x of ['top', 'child']) {
        await Zotero.Utilities.Internal.forEachChunkAsync(
        toDelete[x],
        1000,
        async function (chunk) {
          await this.erase(chunk, eraseOptions);
          processed += chunk.length;
        }.bind(this));

      }
      Zotero.debug("Emptied " + deleted.length + " item(s) from trash in " + (new Date() - t) + " ms");
      Zotero.Notifier.trigger('refresh', 'trash', libraryID);
    }

    return deleted.length;
  };


  /**
   * Start idle observer to delete trashed items older than a certain number of days
   */
  this._emptyTrashIdleObserver = null;
  this._emptyTrashTimeoutID = null;
  this.startEmptyTrashTimer = function () {
    this._emptyTrashIdleObserver = {
      observe: (subject, topic, data) => {
        if (topic == 'idle' || topic == 'timer-callback') {
          var days = Zotero.Prefs.get('trashAutoEmptyDays');
          if (!days) {
            return;
          }

          // TODO: empty group trashes if permissions

          // Delete a few items a time
          //
          // TODO: increase number after dealing with slow
          // tag.getLinkedItems() call during deletes
          let num = 50;
          this.emptyTrash(
          Zotero.Libraries.userLibraryID,
          {
            days,
            limit: num }).


          then((deleted) => {
            if (!deleted) {
              this._emptyTrashTimeoutID = null;
              return;
            }

            // Set a timer to do more every few seconds
            this._emptyTrashTimeoutID = setTimeout(() => {
              this._emptyTrashIdleObserver.observe(null, 'timer-callback', null);
            }, 2500);
          });
        }
        // When no longer idle, cancel timer
        else if (topic === 'active') {
          if (this._emptyTrashTimeoutID) {
            clearTimeout(this._emptyTrashTimeoutID);
            this._emptyTrashTimeoutID = null;
          }
        }
      } };


    var idleService = Components.classes["@mozilla.org/widget/idleservice;1"].
    getService(Components.interfaces.nsIIdleService);
    idleService.addIdleObserver(this._emptyTrashIdleObserver, 305);
  };


  this.addToPublications = function (items, options = {}) {
    if (!items.length) return;

    return Zotero.DB.executeTransaction(function* () {
      var timestamp = Zotero.DB.transactionTimestamp;

      var allItems = [...items];

      if (options.license) {
        for (let item of items) {
          if (!options.keepRights || !item.getField('rights')) {
            item.setField('rights', options.licenseName);
          }
        }
      }

      if (options.childNotes) {
        for (let item of items) {
          item.getNotes().forEach((id) => allItems.push(Zotero.Items.get(id)));
        }
      }

      if (options.childFileAttachments || options.childLinks) {
        for (let item of items) {
          item.getAttachments().forEach((id) => {
            var attachment = Zotero.Items.get(id);
            var linkMode = attachment.attachmentLinkMode;

            if (linkMode == Zotero.Attachments.LINK_MODE_LINKED_FILE) {
              Zotero.debug("Skipping child linked file attachment on drag");
              return;
            }
            if (linkMode == Zotero.Attachments.LINK_MODE_LINKED_URL) {
              if (!options.childLinks) {
                Zotero.debug("Skipping child link attachment on drag");
                return;
              }
            } else
            if (!options.childFileAttachments) {
              Zotero.debug("Skipping child file attachment on drag");
              return;
            }
            allItems.push(attachment);
          });
        }
      }

      yield Zotero.Utilities.Internal.forEachChunkAsync(allItems, 250, Zotero.Promise.coroutine(function* (chunk) {
        for (let item of chunk) {
          item.setPublications(true);
          item.synced = false;
        }
        let ids = chunk.map((item) => item.id);
        yield Zotero.DB.queryAsync(
        `UPDATE items SET synced=0, clientDateModified=? WHERE itemID IN (${ids.join(", ")})`,
        timestamp);

        yield Zotero.DB.queryAsync(
        `INSERT OR IGNORE INTO publicationsItems VALUES (${ids.join("), (")})`);

      }.bind(this)));
      Zotero.Notifier.queue('modify', 'item', allItems.map((item) => item.id));
    }.bind(this));
  };


  this.removeFromPublications = function (items) {
    return Zotero.DB.executeTransaction(function* () {
      let allItems = [];
      for (let item of items) {
        if (!item.inPublications) {
          throw new Error(`Item ${item.libraryKey} is not in My Publications`);
        }

        // Remove all child items too
        if (item.isRegularItem()) {
          allItems.push(...this.get(item.getNotes(true).concat(item.getAttachments(true))));
        }

        allItems.push(item);
      }

      allItems.forEach((item) => {
        item.setPublications(false);
        item.synced = false;
      });

      var timestamp = Zotero.DB.transactionTimestamp;
      yield Zotero.Utilities.Internal.forEachChunkAsync(allItems, 250, Zotero.Promise.coroutine(function* (chunk) {
        let idStr = chunk.map((item) => item.id).join(", ");
        yield Zotero.DB.queryAsync(
        `UPDATE items SET synced=0, clientDateModified=? WHERE itemID IN (${idStr})`,
        timestamp);

        yield Zotero.DB.queryAsync(`DELETE FROM publicationsItems WHERE itemID IN (${idStr})`);
      }.bind(this)));
      Zotero.Notifier.queue('modify', 'item', items.map((item) => item.id));
    }.bind(this));
  };


  /**
   * Purge unused data values
   */
  this.purge = Zotero.Promise.coroutine(function* () {
    Zotero.DB.requireTransaction();

    if (!Zotero.Prefs.get('purge.items')) {
      return;
    }

    var sql = "DELETE FROM itemDataValues WHERE valueID NOT IN " +
    "(SELECT valueID FROM itemData)";
    yield Zotero.DB.queryAsync(sql);

    Zotero.Prefs.set('purge.items', false);
  });



  this.getFirstCreatorFromJSON = function (json) {
    Zotero.warn("Zotero.Items.getFirstCreatorFromJSON() is deprecated " +
    "-- use Zotero.Utilities.Internal.getFirstCreatorFromItemJSON()");
    return Zotero.Utilities.Internal.getFirstCreatorFromItemJSON(json);
  };


  /**
   * Return a firstCreator string from internal creators data (from Zotero.Item::getCreators()).
   *
   * Used in Zotero.Item::getField() for unsaved items
   *
   * @param {Integer} itemTypeID
   * @param {Object} creatorData
   * @return {String}
   */
  this.getFirstCreatorFromData = function (itemTypeID, creatorsData) {
    if (creatorsData.length === 0) {
      return "";
    }

    var validCreatorTypes = [
    Zotero.CreatorTypes.getPrimaryIDForType(itemTypeID),
    Zotero.CreatorTypes.getID('editor'),
    Zotero.CreatorTypes.getID('contributor')];


    for (let creatorTypeID of validCreatorTypes) {
      let matches = creatorsData.filter((data) => data.creatorTypeID == creatorTypeID);
      if (!matches.length) {
        continue;
      }
      if (matches.length === 1) {
        return matches[0].lastName;
      }
      if (matches.length === 2) {
        let a = matches[0];
        let b = matches[1];
        return a.lastName + " " + Zotero.getString('general.and') + " " + b.lastName;
      }
      if (matches.length >= 3) {
        return matches[0].lastName + " " + Zotero.getString('general.etAl');
      }
    }

    return "";
  };


  /**
   * Get the top-level items of all passed items
   *
   * @param {Zotero.Item[]} items
   * @return {Zotero.Item[]}
   */
  this.getTopLevel = function (items) {
    return [...new Set(items.map((item) => item.topLevelItem))];
  };


  /**
   * Return an array of items with descendants of selected top-level items removed
   *
   * Non-top-level items that aren't descendents of selected items are kept.
   *
   * @param {Zotero.Item[]}
   * @return {Zotero.Item[]}
   */
  this.keepTopLevel = function (items) {
    var topLevelItems = new Set(
    items.filter((item) => item.isTopLevelItem()));

    return items.filter((item) => {
      var topLevelItem = !item.isTopLevelItem() && item.topLevelItem;
      // Not a child item or not a child of one of the passed items
      return !topLevelItem || !topLevelItems.has(topLevelItem);
    });
  };


  this.keepParents = function (items) {
    Zotero.debug("Zotero.Items.keepParents() is deprecated -- use Zotero.Items.keepTopLevel() instead");
    return this.keepTopLevel(items);
  };


  /*
   * Generate SQL to retrieve firstCreator field
   *
   * Why do we do this entirely in SQL? Because we're crazy. Crazy like foxes.
   */
  var _firstCreatorSQL = '';
  function _getFirstCreatorSQL() {
    if (_firstCreatorSQL) {
      return _firstCreatorSQL;
    }

    var editorCreatorTypeID = Zotero.CreatorTypes.getID('editor');
    var contributorCreatorTypeID = Zotero.CreatorTypes.getID('contributor');

    /* This whole block is to get the firstCreator */
    var localizedAnd = Zotero.getString('general.and');
    var localizedEtAl = Zotero.getString('general.etAl');
    var sql = "COALESCE(" +
    // First try for primary creator types
    "CASE (" +
    "SELECT COUNT(*) FROM itemCreators IC " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1" +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT lastName FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1" +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedAnd + " ' || " +
    "(SELECT lastName FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1,1)" +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedEtAl + "' " +
    ") " +
    "END, " +

    // Then try editors
    "CASE (" +
    "SELECT COUNT(*) FROM itemCreators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID}` +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID}` +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedAnd + " ' || " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1) " +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedEtAl + "' " +
    ") " +
    "END, " +

    // Then try contributors
    "CASE (" +
    "SELECT COUNT(*) FROM itemCreators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID}` +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID}` +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedAnd + " ' || " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1) " +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT lastName FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' " + localizedEtAl + "' " +
    ") " +
    "END" +
    ") AS firstCreator";

    _firstCreatorSQL = sql;
    return sql;
  }


  /*
   * Generate SQL to retrieve sortCreator field
   */
  var _sortCreatorSQL = '';
  function _getSortCreatorSQL() {
    if (_sortCreatorSQL) {
      return _sortCreatorSQL;
    }

    var editorCreatorTypeID = Zotero.CreatorTypes.getID('editor');
    var contributorCreatorTypeID = Zotero.CreatorTypes.getID('contributor');

    var nameSQL = "lastName || ' ' || firstName ";

    var sql = "COALESCE("
    // First try for primary creator types
    + "CASE (" +
    "SELECT COUNT(*) FROM itemCreators IC " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1" +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT " + nameSQL + "FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1" +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1,1)" +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 1,1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators IC NATURAL JOIN creators " +
    "LEFT JOIN itemTypeCreatorTypes ITCT " +
    "ON (IC.creatorTypeID=ITCT.creatorTypeID AND ITCT.itemTypeID=O.itemTypeID) " +
    "WHERE itemID=O.itemID AND primaryField=1 ORDER BY orderIndex LIMIT 2,1)" +
    ") " +
    "END, "

    // Then try editors
    + "CASE (" +
    "SELECT COUNT(*) FROM itemCreators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID}` +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID}` +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1) " +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${editorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 2,1)" +
    ") " +
    "END, "

    // Then try contributors
    + "CASE (" +
    "SELECT COUNT(*) FROM itemCreators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID}` +
    ") " +
    "WHEN 0 THEN NULL " +
    "WHEN 1 THEN (" +
    "SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID}` +
    ") " +
    "WHEN 2 THEN (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1) " +
    ") " +
    "ELSE (" +
    "SELECT " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 1,1)" +
    " || ' ' || " +
    "(SELECT " + nameSQL + " FROM itemCreators NATURAL JOIN creators " +
    `WHERE itemID=O.itemID AND creatorTypeID=${contributorCreatorTypeID} ` +
    "ORDER BY orderIndex LIMIT 2,1)" +
    ") " +
    "END" +
    ") AS sortCreator";

    _sortCreatorSQL = sql;
    return sql;
  }


  let _stripFromSortTitle = [
  '</?i>',
  '</?b>',
  '</?sub>',
  '</?sup>',
  '<span style="font-variant:small-caps;">',
  '<span class="nocase">',
  '</span>',
  // Any punctuation at the beginning of the string, repeated any number
  // of times, and any opening punctuation that follows
  '^\\s*([^\\P{P}@#*])\\1*[\\p{Ps}"\']*'].
  map((re) => Zotero.Utilities.XRegExp(re, 'g'));


  this.getSortTitle = function (title) {
    if (!title) {
      return '';
    }

    if (typeof title == 'number') {
      return title.toString();
    }

    for (let re of _stripFromSortTitle) {
      title = title.replace(re, '');
    }
    return title.trim();
  };


  Zotero.DataObjects.call(this);

  return this;
}.bind(Object.create(Zotero.DataObjects.prototype))();